/**
 * Layer to encapsulate the keyword extraction API and
 * make it accessible to gnunet-insert.
 * @file src/textui/keywords.c
 * @author Christian Grothoff
 **/

#include "textui/keywords.h"
#include "util/xmalloc.h"
#include "configuration.h"
#include <string.h>
#ifdef HAVE_EXTRACTOR_H
#include <extractor.h>
#endif

/**
 * Extract keywords, mime-type and description from a file
 * @param filename the name of the file
 * @param description the description (the user may have
 *        supplied a description already (*description != NULL),
 *        in that case, append, mind the maximum size!
 * @param mimetype the mimetype, again, the user may
 *        have supplied one
 * @param keywords the list of keywords, allocate space at
 *        another location if required, copy existing keywords
 *        over to that space! Do NEVER free *keywords!
 * @param num_keywords the number of keywords in the
 *        existing *keywords array that was passed in.
 *        Set *num_keywords to the new number of keywords!
 **/
void extractKeywords(char * filename,
		     char ** description,
		     char ** mimetype,
		     char *** keywords,
		     int * num_keywords) {
#ifdef HAVE_LIBEXTRACTOR
  ExtractorList * exList;
  exList = getExtractors();
  extractKeywordsMulti(filename, description,
		       mimetype, keywords, 
		       num_keywords, exList);
  removeAll(exList);
#else
  *num_keywords = 0;
#endif
}

#ifdef HAVE_LIBEXTRACTOR
ExtractorList * getExtractors() {
  ExtractorList * exList = loadDefaultLibraries();
  return loadConfigLibraries(exList,
			     getExtractorConfig());
}
#endif

/**
 * Extract keywords, mime-type and description from a file
 * @param filename the name of the file
 * @param description the description (the user may have
 *        supplied a description already (*description != NULL),
 *        in that case, append, mind the maximum size!
 * @param mimetype the mimetype, again, the user may
 *        have supplied one
 * @param keywords the list of keywords, allocate space at
 *        another location if required, copy existing keywords
 *        over to that space! Do NEVER free *keywords!
 * @param num_keywords the number of keywords in the
 *        existing *keywords array that was passed in.
 *        Set *num_keywords to the new number of keywords!
 **/
void extractKeywordsMulti(char * filename,
			  char ** description,
			  char ** mimetype,
			  char *** keywords,
			  int * num_keywords,
			  ExtractorList * exList) {
#ifdef HAVE_LIBEXTRACTOR
  KeywordList * keyList;
  KeywordList * pos;
  char ** newKeywords;
  char * key;
  int count;
  int i;
  keyList = getKeywords(exList, filename);
  keyList = removeDuplicateKeywords(keyList,
				    DUPLICATES_REMOVE_UNKNOWN);
  if (*mimetype == NULL) {
    key = extractLast(MIMETYPE, keyList);
    if (key != NULL)
      *mimetype = strdup(key);
  }
  if (*description == NULL) {
    key = extractLast(DESCRIPTION, keyList);
    if (key != NULL)    
      *description = strdup(key);
  }
  keyList = removeDuplicateKeywords(keyList,
				    DUPLICATES_TYPELESS);
  count = countKeywords(keyList);
  newKeywords = (char**) xmalloc((count+(*num_keywords)) * sizeof(char*),
				 "extractKeywords: new keywords");
  for (i=0;i<*num_keywords;i++)
    newKeywords[i] = (*keywords)[i];
  pos = keyList;
  for (i=0;i<count;i++) {
    newKeywords[*num_keywords+i] = strdup(pos->keyword);
    pos = pos->next;
  }
  /* assert(keyList == NULL); */
  freeKeywords(keyList);
  *keywords = newKeywords;
  *num_keywords = *num_keywords + count;
#else
  *num_keywords = 0;
#endif
}
