/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * This module represents one of the nodes representing the
 * 1k entities in the tree when downloading a file.<p>
 * This code is currently NOT thread-safe, as the
 * initial version of the code will only have one thread
 * for receiving TCP packets anyway. 
 * @author Christian Grothoff
 **/

#include "config.h"
#include "textui/node.h"
#include "textui/requestmanager.h"
#include "symcipher.h"
#include <sys/types.h>
#include <unistd.h>

#define DEBUG_HASHCHECK 0
#define DEBUG_TREE 0
#define DEBUG_NODE 0

/**
 * Check that the block "data" actually
 * matches the block we are looking for (hash-wise)
 * @param data the data which being checked for a match
 * @param hashC the hash code we expect
 * @param len the length of the part of delta that
 *        we are expecting to be the valid part.
 * @return true if the data we have received
 *         matches our query
 **/
static int hashCheck(CONTENT_Block * data,
		     HashCode160 * hashC,
		     int len) {	
  HashCode160 xhash;
#if DEBUG_HASHCHECK
  HashCode160 yhash;
  HexName hex;
#endif
  int ok;

  hash(data, len, &xhash);
  ok= equalsHashCode160(&xhash, hashC);
#if DEBUG_HASHCHECK
  if (0 == ok) {
    hash2hex(hashC, &hex);
    fprintf(stderr,"expected %s, had ", (char*)&hex);
    hash2hex(&xhash, &hex);
    fprintf(stderr,"%s, double hash is ", (char*)&hex);
    hash(hashC, sizeof(HashCode160),&yhash);
    hash2hex(&yhash, &hex);
    fprintf(stderr,"%s double hash of what we got is ", (char*)&hex);
    hash(&xhash, sizeof(HashCode160),&yhash);
    hash2hex(&yhash, &hex);
    fprintf(stderr,"%s\n", (char*)&hex);
  }
#endif
  return ok;
}

/**
 * Get the RNode of the current node. 
 * Attention: casts INodes to Nodes!
 **/
static RNode * getRoot(Node * node) {
  Node * pos;
  pos = node;
  while (pos->parent != NULL)
    pos = (Node*) pos->parent;  
  return (RNode*)pos;
}
 
/**
 * Return the length of the block without padding
 * @return CONTENT_BLOCK_SIZE except for
 *         the last block if the last block is padded.	 
 **/
int nodeGetBlockLength(Node * node) {
  RNode * root;
  size_t delta;

  root = getRoot(node);
  delta = root->fileLength - node->pos;
  if (delta > CONTENT_SIZE)
    delta = CONTENT_SIZE;
  return (int) delta;
}

/**
 * Get the length of the file that is covered by each child
 * @return the number of bytes each child of this node may
 *         cover (max, disregarding padding)
 **/
static size_t getChildLen(int depth,
			  int treeDepth) {
  size_t childlen = 0;
  int i;

  if (depth > treeDepth)
    return 0;
  childlen = CONTENT_SIZE; 
  // how big is each child-node (full tree below)	
  for (i=depth;i<treeDepth-1;i++)
    childlen *= NODES_PER_INODE; 
  // is child-node an inner node?
  return childlen;
}
    
/**
 * Compute the number of children of this node,
 * acknowleding padded inodes
 * at the end of the file.
 * @return the number of children this node will 
 * have, will be less or equal to NODES_PER_INODE.
 **/
static int getNumberOfChildren(size_t pos,
			       int depth,
			       RNode * root) {
  size_t childlen = 0;
  size_t npi = 0;
  if (depth > root->treeDepth) 
    return 0;
  // compute number and position of subtrees
  childlen = getChildLen(depth, root->treeDepth);
  npi = (((int)root->fileLength) - pos + childlen - 1) 
    / childlen; // how many child nodes?
#if DEBUG_TREE
  fprintf(stderr,"computed %d direct childs of size %d (treedepth is %d, our position is %d:%d)\n",
	  npi,
	  childlen, root->treeDepth,
	  depth, pos);
#endif
  if (npi > NODES_PER_INODE)
    npi = NODES_PER_INODE; // cap at maximum	   
  return (int) npi;
}

/**
 * This method is used to notify the node that
 * some relevant data may have arrived.
 * @param node the node to be notified of data arrival
 * @param hash filename of data that has arrived
 * @param data data corresponding to the hash code
 * @return OK if the data was ok, SYSERR if it must
 *  be re-requested
 **/
int nodeReceive(Node * node,
		HashCode160 * hash, 
		CONTENT_Block * data) { 
  RNode * root;
  int len;
#if PRINT_WARNINGS
  HexName hex;
#endif
  if (!equalsHashCode160(hash, &node->request)) 
    errexit("Receive called, but hash does not match!?");	    

#if DEBUG_NODE  
  hash2hex(hash,
	   &hex);
  fprintf(stderr,"GBlock received reply for query %s\n",
	  (char*)&hex);
#endif  
  if (SYSERR == decryptContent(data,
			       &node->key,
			       &node->data)) {
    /* requestManagerRequest(node, nodeReceive);  */
    return SYSERR; /*re-issue request! */
  }	    
  len = nodeGetBlockLength(node);
  if (!hashCheck(&node->data,
		 &node->key,
		 len)) { 
#if PRINT_WARNINGS
    hash2hex(&node->key,
	     &hex);
    fprintf(stderr,
	    "Invalid packet for Block at position %d with length %d (hash %s does not decode). Ignored.\n",
	    node->pos, 
	    nodeGetBlockLength(node),
	    (char*)&hex);
#endif
    /* requestManagerRequest(node, nodeReceive); */
    return SYSERR; /*re-issue request! */
  } 
  node->resolved = 2; /* yes */
  root = getRoot(node);
  lseek(root->fileHandle,
	node->pos,
	SEEK_SET);
  if (len != write(root->fileHandle,
		   &node->data,
		   len)) {
    close(root->fileHandle);
    errexit("Write operation failed. Aborting.\n");
  }  
  root->bytes_downloaded += len;
  if (node->depth != 0)
    inodeNotifyReceivedData((INode*)node->parent);
  return OK;
}

void tryPresent(Node * node) {
  RNode * root;
  HashCode160 hash1;
  int len;

  root = getRoot(node);
  if (node->pos != lseek(root->fileHandle,
			 node->pos,
			 SEEK_SET)) {
     /* seek failed, do the request */
    requestManagerRequest(node, nodeReceive);    
    return;   
  }
  len = nodeGetBlockLength(node);
  if (len != read(root->fileHandle,
		  &node->data,
		  len)) {
    /* read failed, do the request */
    requestManagerRequest(node, nodeReceive);
    return;
  }
  /* read succeeded, now try if the hash is ok! */
  hash(&node->data, len, &hash1);
  if (equalsHashCode160(&hash1, &node->key)) { 
    /* yes, already present, let's reuse! */
    /*fprintf(stderr,
      "re-using data!\n");*/
    node->resolved = 2;
    root->bytes_downloaded += len;
    if (node->depth != 0) {
      inodeNotifyReceivedData((INode*)node->parent);
    }
  } else {
    requestManagerRequest(node, nodeReceive); /* no, must go again */
  }
}


/* ********************* additional methods ***************x
   x******************* for *inner* Nodes ****************** */

/**
 * Return the length of the block without padding that
 * matches actual content of the INode.
 * @return the length in bytes, including
 *         the hashcode and the indirection-hashes
 **/
int inodeGetBlockLength(INode * inode) {
  int npi;

  npi = getNumberOfChildren(inode->node.pos,
			    inode->node.depth,
			    getRoot((Node*)inode));
  return npi*sizeof(HashCode160)+sizeof(Ulong);
}
	
/**
 * This method is used to notify the node that
 * some relevant data may have arrived.
 * @param inode inode to be notified of data arrival
 * @param hashC hashcode representing the filename of data that has arrived
 * @param data data corresponding to the hash code
 * @return OK if the data was ok, SYSERR if it must
 *  be re-requested
 **/
int inodeReceive(INode * inode,
		  HashCode160 * hashC, 
		  CONTENT_Block * data) { 
  size_t childlen;
  RNode * root;
  int i;
  int childCount;
  Node * child;
  int depth; 
#if DEBUG_NODE
  HexName hex;
#else
#if PRINT_WARNINGS
  HexName hex;
#endif
#endif

  if (!equalsHashCode160(hashC, &inode->node.request))
    errexit("receive called, but hash does not match!?");
#if DEBUG_NODE 
  hash2hex(hashC,
	   &hex);
  fprintf(stderr,
	  "IBlock received reply for query %s for inode\n",
	  (char*)&hex);
#endif

  // we can now resolve this node!
  if (SYSERR == decryptContent(data,
			       &inode->node.key,
			       &inode->node.data)) {
    /* requestManagerRequest((Node*)inode, inodeReceive); re-issue request! */
   return SYSERR; /* invalid data received, re-issue-request! */
  }	    
  if (!hashCheck(&inode->node.data,
		 &inode->node.key,
		 inodeGetBlockLength(inode))) { 
#if PRINT_WARNINGS
    hash2hex(&inode->node.key,
	     &hex);
    fprintf(stderr,
	    "Invalid packet for IBlock at position %d with length %d (hash %s does not decode). Ignored.\n",
	    inode->node.pos, 
	    inodeGetBlockLength(inode),
	    (char*)&hex);
#endif
    /* requestManagerRequest((Node*)inode, inodeReceive);*/ 
    return SYSERR; /* re-issue request */
  }
  inode->node.resolved = 1; /* partial! */
  root = getRoot((Node*)inode);
  depth = inode->node.depth;
  childlen = getChildLen(depth,root->treeDepth);  
  childCount =  getNumberOfChildren(inode->node.pos,
				    inode->node.depth,
				    root);
  inode->crc32 = ntohl(((Ulong*) &inode->node.data)[childCount*sizeof(HashCode160)/sizeof(Ulong)]);
  for (i=0;i<NODES_PER_INODE;i++)
    inode->childs[i] = NULL; 
  for (i=0;i<childCount;i++) {
    if (depth==root->treeDepth) 
      inode->childs[i] = xmalloc(sizeof(Node),
				 "inodeReceive: child (leaf)");
    else 
      inode->childs[i] = xmalloc(sizeof(INode),
				 "inodeReceive: child (inode)");
    child = inode->childs[i];
    memcpy(&child->key,
	   &inode->node.data.content[sizeof(HashCode160)*i],
	   sizeof(HashCode160));
    hash(&child->key,
	 sizeof(HashCode160),
	 &child->request);
    child->parent = inode;
    child->pos = inode->node.pos+childlen*i;
    child->depth = depth+1;
    child->resolved = 0;
  }

#if DEBUG_NODE
  print("Level %d IBlock requests downloads for %d children.\n",
	 depth, childCount);
#endif
  for (i=0;i<childCount;i++) {
#if DEBUG_NODE
    hash2hex(&inode->childs[i]->request,
	     &hex);
    print("%d of %d: %s\n",
	   i, childCount,(char*)&hex);
#endif
    if (depth+1==root->treeDepth) 
      tryPresent(inode->childs[i]);
    else
      requestManagerRequest((Node*)inode->childs[i], 
			    inodeReceive);
  }	    
  return OK;
}

/**
 * Abort this download.
 **/
void inodeAbort(INode * inode) {
  int i;
  if (inode->node.depth==getRoot((Node*)inode)->treeDepth) {
    for (i=0;i<NODES_PER_INODE;i++) 
      if (inode->childs[i] != NULL) {
	requestManagerAbort(inode->childs[i]);
	xfree(inode->childs[i],
	      "inodeAbort: free (1)");      
      }
  } else {
    for (i=0;i<NODES_PER_INODE;i++) 
      if (inode->childs[i] != NULL) {
	inodeAbort((INode*)inode->childs[i]);
	xfree(inode->childs[i],
	      "inodeAbort: free (2)");	
      }  
  }
}

/**
 * This code is called both to notify 
 * this node that another of its children is complete
 * as well as to notify this node's parent that
 * this node is complete when all of its
 * children's downloads are complete.
 **/
void inodeNotifyReceivedData(INode * inode) {
  int childDone;
  int i;
  int  crcBlock[NODES_PER_INODE];
  Ulong crc;
  RNode * root;

  childDone = 0;
  for (i=0;i<NODES_PER_INODE;i++)
    if (inode->childs[i] != NULL) {
      if (inode->childs[i]->resolved == 2)
	childDone++;
    } else
      childDone++;
  if (childDone == NODES_PER_INODE) {
    root = getRoot((Node*)inode);
    /* subtree complete, crc check */
    for (i=0;i<NODES_PER_INODE;i++) {
      if (inode->childs[i] == NULL)
	break; /* last */
      /* check if node or inode here ! */
      if (inode->node.depth==root->treeDepth-1) 
	crcBlock[i] = htonl(crc32N(&inode->childs[i]->data.content[0],
				   nodeGetBlockLength(inode->childs[i])));
      else
	crcBlock[i] = htonl(crc32N(&inode->childs[i]->data.content[0],
				   inodeGetBlockLength((INode*) inode->childs[i])));
    }
    crc = crc32N(&crcBlock[0], sizeof(Ulong) * i);
    if (inode->crc32 != crc) {
      fprintf(stderr,"This is an Inode at depth %d and pos %d\n",
	      inode->node.depth, inode->node.pos);
      fprintf(stderr,
	      "%s%s%s%s (%ld != %ld)\n",
	      "CRC error. Collision in one of the blocks!\n",
	      "You either hit the jackpot (51:2^160 chances), a bug or a malicously inserted file.\n",
	      "Please report if you can reproduce this.\n",
	      "You may also want to check the integrity of your file...\n",
	      crc, inode->crc32);  
      for (childDone=0;childDone<i;childDone++)
	if (inode->node.depth==root->treeDepth-1) 
	  fprintf(stderr,
		  "sub-crc[%d]=%d (length: %d)\n",
		  childDone, crcBlock[childDone],
		  nodeGetBlockLength(inode->childs[childDone]));
	else
	  fprintf(stderr,
		  "%d: sub-crc[%d]=%d (length: %d, children %d)\n",
		  inode->node.depth,
		  childDone, crcBlock[childDone],
		  inodeGetBlockLength((INode*)inode->childs[childDone]),
		  getNumberOfChildren(((INode*)inode->childs[childDone])->node.pos,
				      ((INode*)inode->childs[childDone])->node.depth,
				      getRoot((Node*)((INode*)inode->childs[childDone]))));;
      exit(-1);
    }    
    for (i=0;i<NODES_PER_INODE;i++) 
      if (inode->childs[i] != NULL)
	xfree(inode->childs[i],
	      "inodeNotifyReceivedData: child");
    inode->node.resolved = 2; /* DONE! */
  }
  if (inode->node.depth > 0)
    inodeNotifyReceivedData((INode*)inode->node.parent);
}       

/* end of node.c */
