/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/** 
 * The RequestManager keeps track and re-issues
 * requests for Content-Nodes.
 * @file textui/requestmanager.c
 * @author Christian Grothoff
 **/ 

#include "config.h"
#include "debugging.h"
#include "util/checksum.h"
#include "textui/requestmanager.h"
#include "configuration.h"
#include <math.h>
#include <pthread.h>


/* how often do we try a query  (in a row, it will be
   re-scheduled later!) */
#define MAX_TRIES 5

/**
 * Current list of all pending requests
 **/ 
static RequestEntry ** REQUESTMANAGER_requestList_;

/**
 * Current list of all requests that have been send out
 **/ 
static RequestEntry * REQUESTMANAGER_requestedList_[MAX_REQUESTS];

/**
 * Number of pending requests (highest index)
 **/
static int REQUESTMANAGER_sendRequestCount_;

/**
 * Number of pending requests (highest index)
 **/
static int REQUESTMANAGER_requestListSize_;

/**
 * Mutex for locking on requestmanager shared state
 **/
static Mutex REQUESTMANAGER_Lock_;

/**
 * Current "good" TTL (initial) [64s]
 **/
static int REQUESTMANAGER_initialTTL_;

/**
 * Current estimate of "duplication"
 **/
static int REQUESTMANAGER_duplicationEstimate_;

/**
 * Socket for the downloads (will be closed [=-1] if inactive).
 **/
static GNUNET_TCP_SOCKET REQUESTMANAGER_socket_;

static void performRequest(RequestEntry * request) {
#if DEBUG_RM > 1
  HexName hn;
  int liveQueries;
#endif
  int i;
  time_t now;
  unsigned int ttl;
  
  time(&now);
  if (request->lasttime + ntohl(request->message.ttl) > now) {
#if DEBUG_RM > 3
    hash2hex(&request->message.query,
	     &hn);
    fprintf(stderr,
	    "Delaying request for %s with ttl %d\n",
	    (char*)&hn, 
	    ntohl(request->message.ttl));
#endif
   
    return;
  }
#if DEBUG_RM > 1
  hash2hex(&request->message.query,
	   &hn);
  liveQueries = 0;
  for (i=0;i<MAX_REQUESTS;i++)
    if ( REQUESTMANAGER_requestedList_[i] != NULL)
      liveQueries++;
  fprintf(stderr,
	  "asking for %s with ttl %d and priority %d (retry: %d) Pending: %d/%d (%d)\n",
	  (char*)&hn, 
	  ntohl(request->message.ttl),
	  ntohl(request->message.priority),
	  request->tries,
	  REQUESTMANAGER_sendRequestCount_,
	  REQUESTMANAGER_requestListSize_,
	  liveQueries);
#endif  
  writeToSocket(&REQUESTMANAGER_socket_,
		(TCP_SOCKET_BUFFER*)&request->message);	     
  request->message.priority 
    = htonl(ntohl(request->message.priority)*2+randomi(2*TTL_DECREMENT));
  request->lasttime = now;

  /*#define MACRO(a,b) a = htonl(ntohl(a)b)
    MACRO(request->message.priority,+1);*/
  
  ttl = ntohl(request->message.ttl);
  ttl = ttl*2+randomi(2*TTL_DECREMENT);
  i = forEachHost(NULL, 0, NULL); /* estimate network size! */
  if (i<3) i=3; /* ensure log is >= 1 */
  /* bound the TTL by 'one TTL_DECREMENT' per log(#hosts)' */
  if (ttl > TTL_DECREMENT * (int) log(i))
    ttl = TTL_DECREMENT*i+randomi(2*TTL_DECREMENT);
  request->message.ttl = htonl(ttl);
  request->tries++;
  if ( (request->tries > MAX_TRIES) &&
       (REQUESTMANAGER_sendRequestCount_ > 0) ){
    RequestEntry tmp;
    RequestEntry * target;
    request->tries = 0;
    memcpy(&tmp,
	   request,
	   sizeof(RequestEntry));
    target = REQUESTMANAGER_requestList_
      [ randomi(REQUESTMANAGER_sendRequestCount_)];
    memcpy(request,
	   target,
	   sizeof(RequestEntry));
    memcpy(target,
	   &tmp,
	   sizeof(RequestEntry));
  }
}

static void cronRequestManager(void * unused) {
  int i;
  int active;
  RequestEntry * send_i;
  
#if PRINT_CRON
  print("CRON: enter cronRequestManager\n");
#endif
  active = 0;
  MUTEX_LOCK(&REQUESTMANAGER_Lock_);
  for (i=0;i<MAX_REQUESTS;i++) {
    send_i = REQUESTMANAGER_requestedList_[i];
    if (send_i != NULL) {
      performRequest(send_i);    
      active++;
    }
    /*    else
      fprintf(stderr,"request manager has no request pending at slot %d (%d)\n",
      i,REQUESTMANAGER_sendRequestCount_);*/
  }
  if ( (isOpenConnection(&REQUESTMANAGER_socket_)) &&
       (active == 0) ) {
    /*fprintf(stderr,"request manager has no pending requests, closing socket %d.\n",
      REQUESTMANAGER_socket_.socket);*/
    closeSocketTemporarily(&REQUESTMANAGER_socket_);
  }
  MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);
#if PRINT_CRON
  print("CRON: exit cronRequestManager\n");
#endif
}

void initRequestManager() {
  initGNUnetClientSocket(getGNUnetPort(),
			 "localhost",
			 &REQUESTMANAGER_socket_);
  REQUESTMANAGER_initialTTL_ = 1+TTL_DECREMENT;
  REQUESTMANAGER_duplicationEstimate_ = 0;
  create_recursive_mutex(&REQUESTMANAGER_Lock_);
  REQUESTMANAGER_sendRequestCount_ = 0;
  REQUESTMANAGER_requestListSize_ = MAX_REQUESTS;
  REQUESTMANAGER_requestList_ 
    = (RequestEntry **) xmalloc
    (REQUESTMANAGER_requestListSize_*sizeof(RequestEntry*),
     "initRequestManager: requestList");
  addCronJob(&cronRequestManager, 1, 1,
	     NULL);
}

/**
 * This implementation is only half-baked.
 * We may also want to free a couple of the request
 * datastructures and do other things.
 **/
void doneRequestManager() {
  delCronJob(&cronRequestManager, 1,
	     NULL);
  destroySocket(&REQUESTMANAGER_socket_);
  destroy_mutex(&REQUESTMANAGER_Lock_);
  xfree(REQUESTMANAGER_requestList_,
	"doneRequestManager: requestList");
}

/**
 * This method receives data corresponding to the indicated filename (hashcode)
 * Find the Listener that scheduled this request and
 * drop it from the list of pending requests.
 * @param hash hashcode representing filename
 * @param data data corresponding to this node
 **/ 
void requestManagerReceive(HashCode160 * hash, 
			   CONTENT_Block * data) {
  int i;
  RequestEntry * send_i;
  time_t currentTime;
  HexName hn;
  int discarded;
  
  hash2hex(hash,
	   &hn);
  time(&currentTime);
  discarded = 0;
  MUTEX_LOCK(&REQUESTMANAGER_Lock_);
  for (i=0;i<MAX_REQUESTS;i++) {
    send_i = REQUESTMANAGER_requestedList_[i];
    if (send_i == NULL)
      continue;
    if (equalsHashCode160(hash, &send_i->message.query)) {
      discarded++;
      REQUESTMANAGER_initialTTL_ 
	= ((((int) ((currentTime-send_i->lasttime)) * 4) +
	   (REQUESTMANAGER_initialTTL_*12))) / 16;
      if (REQUESTMANAGER_initialTTL_ < TTL_DECREMENT)
	REQUESTMANAGER_initialTTL_ = TTL_DECREMENT; /* lower bound! */

      if (SYSERR == (((int (*)(Node*,HashCode160 *,CONTENT_Block*))
		      send_i->receiver))(send_i->metadata,hash,data)) {
	MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);
	return;
      }	
      xfree(send_i,
	    "requestManagerReceive: requestEntry");
      send_i = NULL;
      
      if (REQUESTMANAGER_sendRequestCount_>0) {
	REQUESTMANAGER_sendRequestCount_--;
	send_i = REQUESTMANAGER_requestList_[REQUESTMANAGER_sendRequestCount_];
	send_i->lasttime = currentTime;
	send_i->message.priority = htonl(10);
	send_i->message.header.size = htons(sizeof(TCP_Query_Request));
	send_i->message.header.tcpType = htons(TCP_REQUEST_QUERY);
	send_i->message.ttl = htonl((randomi(REQUESTMANAGER_initialTTL_)+TTL_DECREMENT));
 	performRequest(send_i); /* request now - no wait! */
      }
      REQUESTMANAGER_requestedList_[i] = send_i;
    }
  }
  MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);  
#if DEBUG_RM > 1
  if (discarded == 0) {
    REQUESTMANAGER_duplicationEstimate_ ++;
    fprintf(stderr,"received useless crap %d (%d)!\n",
	    REQUESTMANAGER_duplicationEstimate_ ,
	    REQUESTMANAGER_initialTTL_);
  }
#endif
}

/**
 * Start download thread that will exit once the
 * download for the given root-node is complete.
 *
 * @param root the file to download
 * @param data 3rd argument to the method specified in root->downloadModel
 **/
void requestManagerReceiveResults(RNode * root,
				  void * data) {
  TCP_SOCKET_BUFFER buffer;
  TCP_Content_Reply * reply;
  int timeout;
  HexName hex;
#if PRINT_STATUS
  time_t startTime;
  time_t now;
  float bps = 0;
  unsigned int togo = 0;
#endif

  reply = (TCP_Content_Reply *) &buffer;
  timeout = 1;
#if PRINT_STATUS
  time(&startTime);
#endif
  while (root->bytes_downloaded < root->fileLength) {
    if (SYSERR == readFromSocket(&REQUESTMANAGER_socket_,
				 &buffer)) {
      sleep(timeout);
      timeout+=2;
      continue;
    }
    switch (ntohs(reply->header.tcpType)) {
    case TCP_REPLY_CONTENT:
     if (ntohs(reply->header.size) != sizeof(TCP_Content_Reply)) {
       fprintf(stderr,"Received bad reply: %d\n",
	       ntohs(reply->header.tcpType));
   	closeSocketTemporarily(&REQUESTMANAGER_socket_);
	break;
      }
      timeout = 1; 
      hash2hex(&reply->hash, &hex);
#if PRINT_STATUS
      if (root->bytes_downloaded == 0)
	time(&startTime);
     time(&now);
      if (now - startTime > 5) {
	bps = (root->bytes_downloaded / (now-startTime));
	if (bps > 0.001) {
	  togo = (unsigned int) ( (root->fileLength - root->bytes_downloaded) / bps);
	} else
	  togo = (unsigned int) -1;
      } else {
	bps = 0;
	togo = 0;
      }
      fprintf(stderr,
	      "%8d of %8d (last: %s) at %8u bps (average), %4us to go\n",
	      root->bytes_downloaded, root->fileLength,
	      (char*)&hex,
	      (unsigned int)bps,
	      togo);
#endif
      requestManagerReceive(&reply->hash,
			    &reply->content);    
      if (root->downloadModel != NULL)
	((void*(*)(size_t, size_t, void*)) root->downloadModel)(root->bytes_downloaded,
								root->fileLength,
								data);
     break;
    default:
#if PRINT_WARNINGS
      fprintf(stderr,"TCP: Received unknown type: %d\n",
	      ntohs(reply->header.tcpType));
#endif
      /* ignore */
      break;
    }
  }
  /*  destroySocket(&REQUESTMANAGER_socket_);*/
}

/**
 * Queue a request for execution.
 **/
void requestManagerRequest(Node * node,
			   Listener l) {
  int i;
  time_t currentTime;
  RequestEntry ** tmp;
  RequestEntry * send_i;
  HashCode160 * query;

  query = &node->request;
  time(&currentTime);
  MUTEX_LOCK(&REQUESTMANAGER_Lock_);
  for (i=0;i<MAX_REQUESTS;i++) { 
    send_i = REQUESTMANAGER_requestedList_[i];
    if (send_i != NULL) 
      continue;
    send_i = xmalloc(sizeof(RequestEntry),
		     "requestManagerRequest (requestEntry)");
    REQUESTMANAGER_requestedList_[i] = send_i;
    send_i->receiver = l;
    memcpy(&send_i->message.query,
	   query,
	   sizeof(HashCode160));
    send_i->lasttime = currentTime;
    send_i->metadata = node;
    send_i->message.header.size = htons(sizeof(TCP_Query_Request));
    send_i->message.header.tcpType = htons(TCP_REQUEST_QUERY);
    send_i->message.ttl = htonl(randomi(REQUESTMANAGER_initialTTL_)+TTL_DECREMENT);
    send_i->message.priority = htonl(1);
    send_i->tries = 0;
    performRequest(send_i);
    MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);
    return;
  }	    
  /* do we have to grow the queue? */
  if (REQUESTMANAGER_requestListSize_ == 
      REQUESTMANAGER_sendRequestCount_) {
    tmp = REQUESTMANAGER_requestList_;
    REQUESTMANAGER_requestList_ 
      = (RequestEntry **) xmalloc
      (2*REQUESTMANAGER_requestListSize_*sizeof(RequestEntry*),
       "requestManagerRequest. requestList");
    memcpy(REQUESTMANAGER_requestList_, tmp,
	   REQUESTMANAGER_requestListSize_*sizeof(RequestEntry*));
    REQUESTMANAGER_requestListSize_ *= 2;
  }
  /* append to the queue */
  REQUESTMANAGER_requestList_
    [ REQUESTMANAGER_sendRequestCount_]
    = xmalloc(sizeof(RequestEntry),
	      "requestManagerRequest: requestEntry (2)");
  send_i = REQUESTMANAGER_requestList_
    [ REQUESTMANAGER_sendRequestCount_];
  REQUESTMANAGER_sendRequestCount_++;
  send_i->receiver = l;
  memcpy(&send_i->message.query,
	 query,
	 sizeof(HashCode160)); 
  send_i->lasttime = currentTime;
  send_i->metadata = node;
  send_i->message.header.size = htons(sizeof(TCP_Query_Request));
  send_i->message.header.tcpType = htons(TCP_REQUEST_QUERY);
  send_i->message.ttl = htonl(randomi(REQUESTMANAGER_initialTTL_)+TTL_DECREMENT);
  send_i->message.priority = htonl(1);
  send_i->tries = 0;
  MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);
}
  
/**
 * Abort a request
 * @param node what was the query
 **/
void requestManagerAbort(Node * node) {
  int i;
  MUTEX_LOCK(&REQUESTMANAGER_Lock_);
  for (i=0;i<MAX_REQUESTS;i++) {
    if (REQUESTMANAGER_requestedList_[i]->metadata == node) {
      xfree(REQUESTMANAGER_requestedList_[i],
	    "requestManagerAbort: requestedList");
      
      if (REQUESTMANAGER_sendRequestCount_ > 0) 
	REQUESTMANAGER_requestedList_[i] 
	  = REQUESTMANAGER_requestList_[--REQUESTMANAGER_sendRequestCount_];
    }
  }
  for (i=0;i<REQUESTMANAGER_requestListSize_;i++)
    if (REQUESTMANAGER_requestList_[i]->metadata == node) {
      xfree(REQUESTMANAGER_requestList_[i],
	    "requestManagerAbort: requestList");
      REQUESTMANAGER_requestList_[i] 
	= REQUESTMANAGER_requestList_[--REQUESTMANAGER_sendRequestCount_];
    }
  MUTEX_UNLOCK(&REQUESTMANAGER_Lock_);  
}



/* end of requestmanager.c */
