/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Helper functions for searching.
 *
 * @author Christian Grothoff
 * @file textui/searchutil.c 
 **/

#include "config.h"
#include "util/tcpio.h"
#include <unistd.h>
#include <netinet/in.h>
#include "symcipher.h"

typedef struct {
  /* the results we've got so far (hash of root-node) */
  HashCode160 * resultsReceived;
  /* the number of valid entries in resultsReceived */
  int countResultsReceived;
  /* size of the resultsReceived array */
  int sizeRR;
  /* unmatched ("AND") results so far, list of root-node hashes that were received for each keyword */
  HashCode160 ** key2hash;
  /* number of entries in the key2hash (for each dimension) */
  int * key2hashCount;
  /* allocated space in the key2hash (for each dimension) */
  int * key2hashSize;
  /* which method should be called for each result (print method), called with the root-node as the argument */
  void * resultHandler;
  /* arguments to the result handler */
  void * resultHandlerArgs;
} ResultContext;

static GNUNET_TCP_SOCKET SEARCH_socket_;

/**
 * Initialize this modul.
 * @return OK if successful, SYSERR on error
 **/ 
int initSearchUtil(char * hostname,
		   unsigned short port) {
  return initGNUnetClientSocket(port,
				hostname,
				&SEARCH_socket_);
}

void doneSearchUtil() {
  destroySocket(&SEARCH_socket_);
}

/**
 * Display the result, but make sure that
 * every file is only displayed once.
 **/
static void processResult(RootNode * rootNode,
			  ResultContext * rc) {
  HashCode160 * tmp;
  int i;
  
  for (i=0;i<rc->countResultsReceived;i++) {
    if (equalsHashCode160(&rc->resultsReceived[i],
			  &rootNode->hashCode))
      return; /* seen before */
  }
  if (rc->countResultsReceived == rc->sizeRR) {
    tmp = xmalloc(sizeof(HashCode160)*rc->sizeRR*2,
		  "processResults: growing resultsReceived");
    memcpy(tmp,
	   rc->resultsReceived,
	   sizeof(HashCode160)*rc->sizeRR);
    xfree(rc->resultsReceived,
	  "processResult: resultsReceived (so far)");
    rc->resultsReceived = tmp;
  }
  memcpy(&rc->resultsReceived[rc->countResultsReceived++],
	 &rootNode->hashCode,
	 sizeof(HashCode160));
  ((void * (*)(RootNode*, void*)) rc->resultHandler)(rootNode,
						     rc->resultHandlerArgs);
}

/**
 * Filter results that do not match ALL keywords.
 * @param rootNode the new reply
 * @param keyIndex for which key this result matches
 * @param keyCount the number of keys that are ANDed
 * @param rc the context to keep track of which replies we got so far
 **/
static void filterResult(RootNode * rootNode,
			 int keyIndex,
			 int keyCount,
			 ResultContext * rc) {
  int i;
  int j;
  HashCode160 * tmp;

  for (i=0;i<rc->key2hashCount[keyIndex];i++)
    if (equalsHashCode160(&rc->key2hash[keyIndex][i],
			  &rootNode->hashCode))
      return; /* seen before */
  /* maybe we have to grow key2hash */
  if (rc->key2hashSize[keyIndex] ==
      rc->key2hashCount[keyIndex]) {
    rc->key2hashSize[keyIndex] *= 2;
    tmp =  xmalloc(sizeof(HashCode160)*rc->key2hashSize[keyIndex],
		   "filterResult. growing key2hash");
    memcpy(tmp,
	   rc->key2hash[keyIndex],
	   sizeof(HashCode160)*rc->key2hashCount[keyIndex]);
    xfree(rc->key2hash[keyIndex],
	  "filterResult: old key2hash");
    rc->key2hash[keyIndex] = tmp;
  }
  /* add to the matching files for this key */
  memcpy(&rc->key2hash[keyIndex][rc->key2hashCount[keyIndex]++],
	 &rootNode->hashCode,
	 sizeof(HashCode160));
  /* check if the file now matches all keys */
  for (i=0;i<keyCount;i++) {
    for (j=0;j<rc->key2hashCount[i];j++)
      if (equalsHashCode160(&rc->key2hash[i][j],
			    &rootNode->hashCode))
	break; /* break inner for-loop */
    if (j == rc->key2hashCount[i])
      return; /* not found, exit! */
  }
  /*ok, rootNode matches all the AND criteria, display */
  processResult(rootNode,rc);
}

/**
 * Initialize a result context.
 * @param rc the context to intialize
 * @param keyCount the number of keywords
 * @param handler the method to call for results
 * @param handlerArgs the arguments to the result handler method
 **/
static void initResultContext(ResultContext * rc,
			      int keyCount,
			      void * handler,
			      void * handlerArgs) {
  int c;
    
  rc->countResultsReceived = 0;
  rc->sizeRR = 16;
  rc->resultsReceived = xmalloc(sizeof(HashCode160)*rc->sizeRR,
				"initResultContext: resultsReceived");
  rc->key2hash = xmalloc(sizeof(HashCode160*)*keyCount,
			 "initResultContext: key2hash");
  rc->key2hashCount = xmalloc(sizeof(int)*keyCount,
			      "initResultContext: key2hashcounts");
  rc->key2hashSize = xmalloc(sizeof(int)*keyCount,
			     "initResultContext key2hashSize");
  for (c=0;c<keyCount;c++) {
    /* fprintf(stderr, "keycount: %d c: %d\n", keyCount,c); */
    rc->key2hash[c] = xmalloc(sizeof(HashCode160)*16,
			      "initResultContext: key2hash i");
    rc->key2hashCount[c] = 0;
    rc->key2hashSize[c] = 16;
  }	
  rc->resultHandler = handler;
  rc->resultHandlerArgs = handlerArgs;
}

/**
 * Start retrieving results from GNUnet.
 * @param keyCount the number of keywords
 * @param keywords the keywords (for decryption)
 * @param messages the queries (to match against)
 * @param handler the method to call on each result matching all keywords
 * @param handlerArgs the arguments to the result handler method
 **/
void receiveResults(int keyCount,
		    HashCode160 * keywords,
		    TCP_Query_Request * messages,
		    void * handler,
		    void * handlerArgs) {
  ResultContext rc;
  TCP_SOCKET_BUFFER buf;
  TCP_Content_Reply * buffer;
  CONTENT_Block result;
  RootNode * rootNode;
  int ttl;
  int i;
  
  buffer = (TCP_Content_Reply *) &buf;
  initResultContext(&rc, keyCount, handler, 
		    handlerArgs);
  ttl = 1;
  while (1) {
    if (SYSERR == readFromSocket(&SEARCH_socket_,
				 &buf)) {
      sleep(ttl);
      ttl+=2;
      continue;
    }
    switch (ntohs(buffer->header.tcpType)) {
    case TCP_REPLY_CONTENT:
      if (ntohs(buffer->header.size) != 
	  sizeof(TCP_Content_Reply)) {
	closeSocketTemporarily(&SEARCH_socket_);
	break;
      }
	
      ttl = 1; 
      /* now decrypt the reply & call a method to use
	 it */
      for (i=0;i<keyCount;i++) {
	if (equalsHashCode160(&buffer->hash,
			      &messages[i].query)) {
	  if (SYSERR == decryptContent(&buffer->content,
				       &keywords[i],
				       &result)) {
#ifdef PRINT_WARNINGS
	    printf("ERROR: decrypt failed!?\n");
#endif
	  continue;
	  }
	  rootNode = (RootNode*) &result;
	  if ( (htons(rootNode->major_formatVersion) != ROOT_MAJOR_VERSION) ||
	       (htons(rootNode->minor_formatVersion) != ROOT_MINOR_VERSION) ) {
	    printf("ERROR: content has unsupported version: %d.%d (or is pre-GNUnet 0.4.0)\n",
		   rootNode->major_formatVersion,
		   rootNode->minor_formatVersion);
	    continue; /* bah! broken! */
	  }
	  filterResult(rootNode, i, keyCount,
		       &rc);
	}
      }
      break;
    default:
      /* ignore */
      break;
    }
  }
}

/**
 * Repeatedly send out the queries to GNUnet.
 * @param keyCount the number of queries/keys
 * @param timeout how long to do it (in seconds)
 * @param messages the initialized queries (will be modified as priorities/ttl increase)
 * @return 0
 **/
int sendQueries(int keyCount,
		int timeout,
		TCP_Query_Request messages[]) {
  time_t start;
  time_t now;
  int c;  
  int remTime;

  time(&start);

  /* issue requests forever, increasing ttl & priority. */
  c=0;
  while (1) {
    time(&now);
    if (timeout != 0) {
      remTime = start-now + timeout;
      if (remTime <= 0) 
	return 0;      
    } else
      remTime = 0xFFFFFF; /* infty */
    if (c == keyCount)
      c = 0;
    if (OK == writeToSocket(&SEARCH_socket_,
			    (TCP_SOCKET_BUFFER*)&messages[c])) {
      /* successful transmission to GNUnet,
	 increase ttl/priority for the next time */
      messages[c].ttl = htonl(2*ntohl(messages[c].ttl));
      messages[c].priority = htonl(2*ntohl(messages[c].priority));
    }
    if (ntohl(messages[c].ttl)/keyCount < remTime)
      sleep(ntohl(messages[c].ttl)/keyCount);
    else 
      sleep(remTime);
    c++;
  }
  return 0;
}

/**
 * Build an initial set of query messages
 * from the list of keywords.
 * @param keyCount the number of keywords
 * @param keywords the keywords (or keys)
 * @param messages the resulting query messages
 **/
void buildMessages(int keyCount,
		   HashCode160 keywords[],
		   TCP_Query_Request * messages[]) {
  int i;

  *messages = (TCP_Query_Request*) xmalloc(keyCount *
					   sizeof(TCP_Query_Request),
					   "buildMessages: messages");  
  for (i=0;i<keyCount;i++) {
    (*messages)[i].header.size = htons(sizeof(TCP_Query_Request));
    (*messages)[i].header.tcpType = htons(TCP_REQUEST_QUERY);
    (*messages)[i].ttl = htonl(keyCount*10+60);
    (*messages)[i].priority = htonl(10);
    hash(&keywords[i],
	 sizeof(HashCode160),
	 &((*messages)[i].query));
  }
}

/**
 * Parse the keywords (join at spaces, separate at AND).
 * @param num_keywords the number of ascii-keywords
 * @param keywords the list of ascii-keywords
 * @param keys the hashs of the keywords to set (= the keys, not the queries!)
 * @return -1 on error, 0 if we should exit without error, number of keys if we actually are going to do something
 **/
int parseKeywords(int num_keywords,
		  char ** keywords,
		  HashCode160 * keys[]) { 
  int keyCount;
  int i;
  char * tmp;

  keyCount = 0;
  *keys = (HashCode160*) xmalloc(sizeof(HashCode160) * (num_keywords+1),
				 "parseKeywords: result");
  for (i=0;i<num_keywords;i++) {
    if ( (i == num_keywords-1) ||
	 (0 == strcmp(keywords[i+1],"AND")) ) {
      keywords[keyCount] = keywords[i];
      hash(keywords[i],
	   strlen(keywords[i]),
	   &((*keys)[keyCount++]));
      i++; /* skip the "AND" */
    } else {
      tmp = xmalloc(strlen(keywords[i])+
		    strlen(keywords[i+1])+2,
		    "parseKeywords: tmp");
      tmp[0] = '\0';
      strcat(tmp, keywords[i]);
      strcat(tmp," ");
      strcat(tmp, keywords[i+1]);      
      keywords[i+1] = tmp;
    }
  }
  return keyCount;
}

/* end of searchutil.c */
