/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * ContentDatabase (directory based implementation). 
 * @author Christian Grothoff
 * @file util/contentdatabase_directory.c
 **/

#include "config.h"
#include "util/contentdatabase_directory.h"
#include "util/xmalloc.h"
#include "util/semaphore.h"

#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <dirent.h>
#include <errno.h>

/**
 * The (expanded) directory name (i.e. "/home/usr/.gnunet/data/content/")
 **/
static char * directory_name;

/**
 * Initialize the Directory module, expand filename
 * @param dir the directory where content is configured to be stored (e.g. ~/.gnunet/data/content).
 **/
static char * getDirectory(char * dir) {
  char * result;
  char * tmp;

  fprintf(stderr,
	  "Database (Directory): %s\n", 
	  dir);
  tmp = xmalloc(strlen(dir) + 2,
		"getDirectory: filename (tmp)");  
  tmp[0] = '\0';
  strcat(tmp, dir);
  strcat(tmp,"\\");
  result = expandFileName(tmp);
  xfree(tmp, "getDirectory: filename (tmp)");
  return result;
}

void initContentDatabase_Directory(char * dir) {  
  directory_name = getDirectory(dir);
  if (directory_name == NULL) {
    fprintf(stderr,
	    "FATAL: could not open directory %s!\n",
	    (char*) dir);
    exit(-1);
  }
}

/**
 * Clean shutdown of the storage module (not used at the moment)
 **/
void doneContentDatabase_Directory() {
  xfree(directory_name,
	"doneContentDatabase_Directory");
}

/**
 * Call a method for each entry in the database and
 * call the callback method on it. 
 * @return the number of items stored in the content database
 **/
int forEachEntryInDatabase_Directory(void (*callback)(HashCode160*)) {
  DIR * dinfo;
  struct dirent *finfo;
  struct stat istat;
  int count;
  HashCode160 hash;

  stat(directory_name,&istat);
  if (!S_ISDIR(istat.st_mode)) {
#ifdef PRINT_WARNINGS
    fprintf(stderr,
            "scanDirectory must be invoked on a directory (%s)!\n",
            dirName);
#endif
    return -1;
  }
  errno = 0;
  dinfo = opendir(directory_name);
  if ((errno == EACCES) || (dinfo == NULL)) {
    fprintf(stderr,
            "scanDirectory: access denied (%s)\n",
            directory_name);
    return -1;
  }
  count = 0;
  while ((finfo = readdir(dinfo)) != NULL)
    if (strlen(finfo->d_name) == sizeof(HashCode160)*2) {
      hex2hash((HexName*) finfo->d_name,
	       &hash);
      callback(&hash);
      count++;
    }

  closedir(dinfo);
  return count;
}

/**
 * Read the contents of a bucket to a buffer. 
 * @param name hashcode representing the name of the file, not freed!
 * @param len the maximum number of bytes to read
 * @param result the buffer to write the result to
 * @return the number of bytes read on success, -1 on failure
 **/ 
int readContent_Directory(HashCode160 * name,
			  int  len,
			  void * result) {
  /* open file, must exist, open read only */
  int handle;
  int size;
  HexName fn;
  FileName fil;

  if ((name == NULL) || (result == NULL))
    return -1;
  hash2hex(name, &fn);
  fil = xmalloc(strlen(directory_name) + strlen((char*)&fn) + 1,
		"readContent_Directory: filename");
  buildFileName(directory_name, &fn, fil);
  handle = open(fil,O_RDONLY,S_IRUSR);
  xfree(fil,"readContent: filename");

  if (handle < 0)
    return -1;
  size = read(handle, result, len);
  close(handle);
  return size;
}

/**
 * Write content to a file. Check for reduncancy and eventually
 * append.
 * @param name the hashcode representing the name of the file 
 *        (without directory)
 * @param block the CONTENT_SIZE bytes long block (no header!)
 **/
void writeContent_Directory(HashCode160 * name, 
			    CONTENT_Block * block) {
  CONTENT_Block preBlock[MAX_FSIZE];
  HexName fn;
  FileName fil;
  int i;
  int j;
  int blocks;
  int handle;

  hash2hex(name, &fn);
  fil = xmalloc(strlen((char*)directory_name) + strlen((char*)&fn) + 1,
		"writeContent_Directory: filename");
  buildFileName(directory_name, &fn, fil);
  handle = open(fil, O_RDWR|O_CREAT, S_IRUSR|S_IWUSR);
  if (handle == -1) {
    fprintf(stderr,
            "Warning: failed to open file %s\n", fil);
    xfree(fil,"writeContent: filaname (1)");
    return; /* failed! */
  }
  xfree(fil,"writeContent_Directory: filename");
  j = read(handle, preBlock, MAX_FSIZE*CONTENT_SIZE);
  if ((j == 0)||(j % CONTENT_SIZE != 0)) {
    write(handle, block, CONTENT_SIZE);
    close(handle);
    return;
  }
  blocks = j / CONTENT_SIZE;
  if (blocks >= MAX_FSIZE) {
    fprintf(stderr,
            "Attention: MAX_FSIZE (%d) collisions reached for a keyword!\n",
            MAX_FSIZE);
    close(handle);
    return; /* full! */
  }
  /* collision! check if duplicate and if not, append */
  for (i=0;i<blocks;i++) {
    if (0 == memcmp(&preBlock[i].content[0],
		    &block->content[0],
		    CONTENT_SIZE)) {
      close(handle);
      return; /* collision! */
    }
  }
  write(handle, block, CONTENT_SIZE);
  close(handle);
}

/**
 * Free space in the database by removing one file
 * @param name the hashcode representing the name of the file 
 *        (without directory)
 **/
void unlinkFromDB_Directory(HashCode160 * name) {
  HexName fn;
  FileName fil;

  if (name == NULL)
    return;
  hash2hex(name, &fn);
  fil = xmalloc(strlen(directory_name) + strlen((char*)&fn) + 1,
		"unlinkFromDB_Directory: filename");
  buildFileName(directory_name, &fn, fil);
  unlink(fil);
  xfree(fil,"unlinkFromDB_Directory: filename");
}


/* end of contentdatabase_gdbm.c */
