/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * ContentDatabase (gdbm implementation). 
 * @author Christian Grothoff
 * @author Uli Luckas
 * @file util/contentdatabase_gdbm.c
 **/

#include "config.h"
#include "util/contentdatabase_gdbm.h"
#include "util/xmalloc.h"
#include "util/semaphore.h"

#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <dirent.h>
#include <errno.h>

static GDBM_FILE dbf;

/**
 * Mutex for locking on requestmanager shared state
 **/
static Mutex DATABASE_Lock_;


/**
 * Open a gdbm database (for content)
 * @param dir the directory where content is configured
 *         to be stored (e.g. data/content). A file 
 *         called ${dir}.gdb is used instead
 **/
static GDBM_FILE getDatabase(char *dir) {
  FileName fin, ff;
  GDBM_FILE result;

  char *ext = ".gdb";
  int fnSize = strlen(dir) + strlen(ext);
  
  fprintf(stderr,
	  "Database (GDBM): %s\n", 
	  dir);

  ff = xmalloc(fnSize+1,
	       "getDatabase: filaname");  
  strcpy(ff, dir);
  if (ff[strlen(ff)-1] == '/')
    ff[strlen(ff) - 1] = 0; /* eat the '/' at the end */
  else 
    ff[strlen(ff)] = 0; /* no '/' to eat */
  strcat(ff, ext); /* add the extention */
  fin = expandFileName(ff); /* expand ~/.gnunet if present */
  xfree(ff,
	"getDatabase: filaname");  
  result = gdbm_open(fin, 0, GDBM_WRCREAT, S_IRUSR|S_IWUSR, 0);  
  if (!result) {
    fprintf(stderr,
	    "getDatabase: failed to open database file %s with error: %s",
	    (char*)fin,
	    gdbm_strerror(gdbm_errno));
    if (errno) {
      perror(", system error ");
    } else {
      fprintf(stderr,
	      "\n");
    }
  }
  xfree(fin,
	"getDatabase: filaname (2)");
  return result;
}

void initContentDatabase_GDBM(char * dir) {  
  dbf = getDatabase(dir);
  if (dbf == NULL) {
    fprintf(stderr,
	    "FATAL: could not open database!\n");
    exit(-1);
  }
  create_recursive_mutex(&DATABASE_Lock_);
}

/**
 * Clean shutdown of the storage module (not used at the moment)
 **/
void doneContentDatabase_GDBM() {
  gdbm_close(dbf);
  destroy_mutex(&DATABASE_Lock_);
}

/**
 * Call a method for each entry in the database and
 * call the callback method on it. 
 * @return the number of items stored in the content database
 **/
int forEachEntryInDatabase_GDBM(void (*callback)(HashCode160*)) {
  datum prevkey, key;
  HashCode160 doubleHash;
  int count;

  count = 0;
  /* scan database data/content.dbf and add entries to database 
     if not already present */
  MUTEX_LOCK(&DATABASE_Lock_);
  key = gdbm_firstkey(dbf);
  while (key.dptr) {
    if (strlen(key.dptr) == sizeof(HashCode160)*2) {	
      hex2hash((HexName*)key.dptr,
	       &doubleHash);
      callback(&doubleHash);
      count++; /* one more file */
    }
    prevkey = key;
    key = gdbm_nextkey(dbf, prevkey);
    xfree(prevkey.dptr, "database dptr (allocation in DB code)");
  }
  MUTEX_UNLOCK(&DATABASE_Lock_);
  key = gdbm_firstkey(dbf);
  return count;
}

/**
 * Read the contents of a bucket to a buffer. 
 * @param name hashcode representing the name of the file, not freed!
 * @param len the maximum number of bytes to read
 * @param result the buffer to write the result to
 * @return the number of bytes read on success, -1 on failure
 **/ 
int readContent_GDBM(HashCode160 * name,
		     int  len,
		     void * result) {
  HexName fn;
  datum key, buffer;

  if ((name == NULL) || (result == NULL))
    return -1;
  hash2hex(name, &fn);

  
  key.dptr = fn.data;
  key.dsize = strlen(key.dptr) + 1;

  MUTEX_LOCK(&DATABASE_Lock_);
  buffer = gdbm_fetch(dbf, key);
  MUTEX_UNLOCK(&DATABASE_Lock_);
  if (!buffer.dptr) 
    return -1;
  if (len > buffer.dsize) 
    len = buffer.dsize;
  memcpy(result, buffer.dptr, len);
  free(buffer.dptr);
  return buffer.dsize;  
}

/**
 * Write content to a file. Check for reduncancy and eventually
 * append.
 * @param name hashcode representing the name of the file (without directory)
 * @param block the CONTENT_SIZE bytes long block (no header!)
 **/
void writeContent_GDBM(HashCode160 * name, 
		       CONTENT_Block * block) {
  CONTENT_Block preBlock[MAX_FSIZE];
  HexName fn;
  datum buffer, key;
  int i;
  int j;
  int blocks;

  hash2hex(name, &fn);
  key.dptr = fn.data;
  key.dsize = strlen(key.dptr) + 1;
  MUTEX_LOCK(&DATABASE_Lock_);
  buffer = gdbm_fetch(dbf, key);
  MUTEX_UNLOCK(&DATABASE_Lock_);
  if (!buffer.dptr) {
    j = 0;
  } else {
    j = buffer.dsize;
    memcpy(preBlock->content, buffer.dptr, j);
    xfree(buffer.dptr,"freeing dptr (allocated by gdbm)");
  }
  
  if ((!j) || (j % CONTENT_SIZE != 0)) {
    if (!j) {
#ifdef VERY_VERBOSE
      fprintf(stderr,
	      "Info: New Bucket created (%d)\n",
	      MAX_FSIZE);
#endif
    } else {
      fprintf(stderr,
	      "WARNING: bucket in content database is corrupted!\n");
    }
    buffer.dptr = block->content;
    buffer.dsize = CONTENT_SIZE;
    MUTEX_LOCK(&DATABASE_Lock_);
    gdbm_store(dbf, key, buffer, GDBM_REPLACE);
    MUTEX_UNLOCK(&DATABASE_Lock_);
    return;
  }
  blocks = j / CONTENT_SIZE;
#ifdef VERY_VERBOSE
  fprintf(stderr,
	  "Info: Found bucket %s with %d blocks\n",
	  fn->data, blocks);
#endif
  if (blocks >= MAX_FSIZE) {
    fprintf(stderr,
	    "Attention: MAX_FSIZE (%d) collisions reached for a keyword!\n",
	    MAX_FSIZE);
    return; /* full! */
  }
  /* collision! check if duplicate and if not, append */    
  for (i=0;i<blocks;i++) {
    if (0 == memcmp(&preBlock[i].content[0],
		    &block->content[0],
		    CONTENT_SIZE)) {
      return; /* collision! */
    }
  }
#ifdef VERY_VERBOSE
  fprintf(stderr,
	  "Info: Appended Data as block %d to bucket.\n",
	  blocks + 1);
#endif
  memcpy(preBlock[blocks].content , block->content, CONTENT_SIZE);
  buffer.dptr = preBlock->content;
  buffer.dsize += CONTENT_SIZE;
  MUTEX_LOCK(&DATABASE_Lock_);
  gdbm_store(dbf, key, buffer, GDBM_REPLACE);
  MUTEX_UNLOCK(&DATABASE_Lock_);
}

/**
 * Free space in the database by removing one file
 * @param name hashcode representing the name of the file (without directory)
 **/
void unlinkFromDB_GDBM(HashCode160 * name) {
  datum key;
  HexName fn;

  hash2hex(name, &fn);
  key.dptr = fn.data;
  key.dsize = strlen(key.dptr) + 1;
  MUTEX_LOCK(&DATABASE_Lock_);
  gdbm_delete(dbf, key);
  MUTEX_UNLOCK(&DATABASE_Lock_);
}


/* end of contentdatabase_gdbm.c */
