/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Module for cron jobs.
 * @author Christian Grothoff
 * @file util/cron.c
 **/

#include "config.h"
#include "util/cron.h"

/** the initial size of the cron-job table **/
#define INIT_CRON_JOBS 64

/**
 * Granularity of the cron jobs (how precise is the timer,
 * in seconds). 1 is highest, do not set above 120.
 **/
#define CRON_GRANULARITY 1

/**
 * The Delta-list for the cron jobs.
 **/
typedef struct {
  /** The time to wait for this event (in seconds). */
  int delta;
  /** The method to call at that point. */
  void (*method)(void*);
  /** for cron-jobs: when this should be repeated
      automatically, 0 if this was a once-only job */
  int deltaRepeat;
  /** The index of the next entry in the delta list
      after this one */
  int next;
  /** data ptr (argument to the method) **/
  void * data;
} DeltaEntry;

/**
 * The delta-list of waiting tasks.
 **/
static DeltaEntry * deltaList_;

/**
 * The current size of the DeltaList.
 **/
static unsigned int deltaListSize_;

/** 
 * The lock for the delta-list.
 **/
static Mutex deltaListLock_;

/**
 * The first empty slot in the delta-list.
 **/
static int firstFree_;

/**
 * The first empty slot in the delta-list.
 **/
static int firstUsed_;

/**
 * The cron thread.
 **/
static pthread_t cron_;

/**
 * The main-method of cron, will NEVER terminate.
 **/
static void cron();

/**
 * Initialize the cron service.
 **/
void initCron() {
  int i;

  deltaListSize_ = INIT_CRON_JOBS;
  deltaList_ 
    = xmalloc(sizeof(DeltaEntry) * deltaListSize_,
	      "initCron: deltalist");
  for (i=0;i<deltaListSize_;i++)
    deltaList_[i].next = i-1;
  firstFree_ = deltaListSize_-1;
  create_mutex(&deltaListLock_);
  firstUsed_  = -1;
}

/**
 * Start the cron jobs.
 **/
void startCron() {
  pthread_create(&cron_, NULL,
		 (void * (*)()) cron, NULL);
}

/**
 * Add a cron-job to the delta list.
 * @param method which method should we run
 * @param delta how many seconds until we run the method
 * @param deltaRepeat if this is a periodic, the time between
 *        the runs, otherwise 0.
 **/
void addCronJob(void (*method)(void*),
		int delta,
		int deltaRepeat,
		void * data) {
  DeltaEntry * entry;
  DeltaEntry * pos;
  int last;
  int current;

  while (delta % CRON_GRANULARITY != 0)
    delta++;
  while (deltaRepeat % CRON_GRANULARITY != 0)
    deltaRepeat++;
  mutex_lock(&deltaListLock_);
  if (firstFree_ == -1) {
    /* need to grow */
    DeltaEntry * tmp;
    int i;

    tmp = xmalloc(sizeof(DeltaEntry) * deltaListSize_ * 2,
		  "addCronJob: grow deltaList");
    memcpy(tmp, 
	   deltaList_,
	   deltaListSize_*sizeof(DeltaEntry));
    xfree(deltaList_,
	  "addCronJob: old deltaList");
    deltaList_ = tmp;
    for (i=deltaListSize_;i<deltaListSize_*2;i++)
      deltaList_[i].next = i-1;
    deltaListSize_ = deltaListSize_ * 2;
    firstFree_ = deltaListSize_-1;
  }
  entry = &deltaList_[firstFree_];
  entry->method = method;
  entry->data = data;
  entry->deltaRepeat = deltaRepeat;
  if (firstUsed_ == -1) {
    firstUsed_ 
      = firstFree_;
    firstFree_
      = entry->next;
    entry->next = -1; /* end of list */
    entry->delta = delta;
    mutex_unlock(&deltaListLock_); 
 
    return;
  }
  /* no, there are jobs waiting */
  last = -1;
  current = firstUsed_;
  pos = &deltaList_[current];
  while (delta > pos->delta) {
    delta -= pos->delta;
    if (pos->next != -1) {
      last = current;
      current = pos->next;
      pos = &deltaList_[current];
    }
    else { /* append */
      pos->next = firstFree_;
      entry->delta = delta;
      firstFree_
	= entry->next;
      entry->next = -1;
      mutex_unlock(&deltaListLock_); 
      return;
    }      
  }
  /* insert before pos */
  if (last == -1)
    firstUsed_ = firstFree_;
  else 
    deltaList_[last].next = firstFree_;
  entry->delta = delta;
  pos->delta -= delta;
  firstFree_
    = entry->next;
  entry->next = current;
  mutex_unlock(&deltaListLock_); 
}

#ifdef HAVE_PRNIT_CRON_TAB
/**
 * Print the cron-tab.
 **/
static void printCronTab() {
  int jobId;
  DeltaEntry * tab;

  mutex_lock(&deltaListLock_);
  fprintf(PRINTTO,
	  "Start: %d Free: %d\n", 
	firstUsed_,firstFree_);
  for (jobId = 0;jobId<deltaListSize_;jobId++) {
    tab = &deltaList_[jobId];
    fprintf(PRINTTO,
	    "%d: delta %d --- method %d --- repeat %d --- next %d\n",
	    jobId, tab->delta, (int)tab->method, tab->deltaRepeat, 
	    tab->next);
  }
  mutex_unlock(&deltaListLock_);
}
#endif

/**
 * Process the cron-job at the beginning of the waiting
 * queue, that is, remove, invoke, and re-insert if
 * it is a periodical job. Make sure the sync is down
 * while the job is running (it may add other jobs!)
 **/
static void runJob() {
  DeltaEntry * job;
  int jobId;
  void (* method)(void*);
  void * data;
  int repeat;

  mutex_lock(&deltaListLock_);
  jobId = firstUsed_;
  if (jobId == -1) {
    mutex_unlock(&deltaListLock_); 
    return; /* no job to be done */
  }
  job = &deltaList_[jobId];
  method = job->method;
  data = job->data;
  repeat = job->deltaRepeat;
  /* remove from queue */
  firstUsed_
    = job->next;
  job->next
    = firstFree_;
  firstFree_ = jobId;
  mutex_unlock(&deltaListLock_); 
  /* run */
  method(data);
  /* re-insert */
  if (repeat > 0) 
    addCronJob(method, repeat, repeat, data);
}

/**
 * The main-method of cron.
 **/
static void cron() {
  int delta;
  while (1) {
    mutex_lock(&deltaListLock_);
    if (firstUsed_ != -1) 
      deltaList_[firstUsed_].delta
	-= CRON_GRANULARITY;
    delta = -1;
    while (delta <= 0) {
      if (delta == 0)
	mutex_lock(&deltaListLock_);
      if (firstUsed_ != -1) {
	delta = deltaList_[firstUsed_].delta; 
#ifdef CHECK_ASSERTS
	if (delta < 0) {
#ifdef PRINT_WARNINGS
	  fprintf(PRINTTO,
		  "WARNING: crontab inconsistend, cron exits!\n");
#endif
	  mutex_unlock(&deltaListLock_);     
	  return;
	}
#endif
      } else
	delta = 1; /* end loop */
      mutex_unlock(&deltaListLock_);     
      if (delta == 0)
	runJob();
    }
    sleep(CRON_GRANULARITY); /* granularity here! */
  }
}

/**
 * Remove all matching cron-jobs from the list.
 * @param method which method is listed?
 * @param repeat which repeat factor was chosen? 
 * @param data what was the data given to the method
 **/
void delCronJob(void (*method)(void *),
		int repeat,
		void * data) {
  DeltaEntry * job;
  DeltaEntry * last;
  int jobId;

  mutex_lock(&deltaListLock_);
  jobId = firstUsed_;
  if (jobId == -1) {
    mutex_unlock(&deltaListLock_); 
    return; 
  }
  last = NULL;
  job = &deltaList_[jobId];
  while ( (job->method != method) || 
	  (job->data != data) ||
	  (job->deltaRepeat != repeat) ) {
    last = job;
    if (job->next == -1) {
      mutex_unlock(&deltaListLock_); 
      return; 
    }
    jobId = job->next;
    job = &deltaList_[jobId];    
  }
  if (last != NULL) 
    last->next = job->next;
  else
    firstUsed_ = job->next;
  if (job->next != -1) {
    last = &deltaList_[job->next]; 
    last->delta += job->delta;
  }
  job->next
    = firstFree_;
  firstFree_ = jobId;
  mutex_unlock(&deltaListLock_);
  /* ok, there may be more matches, go again! */
  delCronJob(method, repeat, data);
}

/* end of cron.c */
