/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * hashcode management
 * @author Christian Grothoff
 * @file util/hashing.c
 **/

#include "config.h"
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include "util/hashing.h"
#include "util/xmalloc.h"

/**
 * Hash block of given size.
 * @param block the data to hash, length is given as a second argument
 * @param size the length of the data to hash
 * @param ret pointer to where to write the hashcode
 **/
void hash(void * block,
	  int size,
	  HashCode160 * ret) {
  if (block == NULL)
    return;
  RIPEMD160(block, size, (unsigned char*) ret);
}

static unsigned char * encoding__ = "0123456789ABCDEF";

/**
 * Convert (hash) block to hex (= filename)
 * @param block the sequence to convert
 * @param result where to store thestring (\0-terminated), hex-encoding
 **/
void hash2hex(HashCode160 * block,
	      HexName * result) {
  int i;
  int j;
  unsigned char c;
  unsigned char clow;

  if ((block == NULL) || (result == NULL)) {
    fprintf(stderr,
	    "hash2hex called with block or result NULL!");
    exit(-1);
  }

  result->data[sizeof(HashCode160)*2]=0;
  j=0;
  for (i=0;i<sizeof(HashCode160);i++) {
    c = ((unsigned char *)block)[i]; 
    clow = c & 15; /* get lower nibble */
    result->data[j++] = encoding__[clow];
    clow = c >> 4; /* get higher nibble */
    result->data[j++] = encoding__[clow];
  }  
}

/**
 * Convert hex (filename) to the hostIdentity
 * @param hex the filename
 * @param hash is set to the correspoinding host identity
 **/
void hex2hash(HexName * hex,
	      HashCode160 * hash) {
  int i;
  int j;
  unsigned char c;
  unsigned char clow;
  unsigned char chigh;

  if ((hex == NULL) || (hash == NULL)) {
    fprintf(stderr,
	    "hex2hash called with hax or hash NULL!");
    exit(-1);
  }
  if (strlen((char*)hex) != sizeof(HashCode160)*2) {
    fprintf(stderr,
	    "assertion failed: strlen(hex) is not 32");
    exit(-1);
  }
  j=0;
  i=0;
  while (i<sizeof(HashCode160)*2) {
    clow = hex->data[i++];
    if (clow >= 'A')
      clow = clow - 'A' + 10;
    else
      clow = clow - '0';
    chigh = hex->data[i++];    
    if (chigh >= 'A')
      chigh = chigh - 'A' + 10;
    else
      chigh = chigh - '0';
    c = clow + (chigh << 4);
    ((unsigned char *)hash)[j++] = c;
  }  
}


/**
 * Compute the distance between 2 hashcodes.
 * The computation must be fast, not involve
 * a.a or a.e (they're used elsewhere), and
 * be somewhat consistent. And of course, the
 * result should be a positive number.
 * @returns a positive number which is a measure for 
 *  hashcode proximity.
 **/
int distanceHashCode160(HashCode160 * a, 
			HashCode160 * b) {
  int x = (a->b - b->b)>>16;
  return ((x*x)>>16);
}

/**
 * Compare two hashcodes.
 * @return 1 if they are equal, 0 if not.
 **/
int equalsHashCode160(HashCode160 * a, 
		      HashCode160 * b) {
  return (0 == memcmp(a,b,sizeof(HashCode160)));
}

/* end of hashing.c */
