/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Hostkey - public key cryptography routines 
 * @author Christian Grothoff
 * @file util/hostkey.c
 **/

#include "config.h"
#include "util/hostkey.h"
#include "util/xmalloc.h"
#include "util/returnvalues.h"
#include "debugging.h"
#include <netinet/in.h>

#define HOSTKEY_LEN 2048

/**
 * Initialize Random number generator.
 **/
void initRAND() {
  srand((unsigned int)time(NULL));
  /* this is probably not the right random generator for openssl ... */
}

/**
 * This HostKey implementation uses RSA.
 **/
HOSTKEY makeHostkey() {
  return RSA_generate_key(HOSTKEY_LEN,65535,NULL,0);
}

/**
 * Free memory occupied by hostkey
 **/
void freeHostkey(HOSTKEY hostkey) {
  RSA_free(hostkey);
}

/**
 * Extract the public key of the host.
 * @param hostkey the hostkey to extract into the result.
 * @param result where to write the result.
 **/
void getPublicKey(HOSTKEY hostkey,
		  PublicKey * result) {
  BLOCK_LENGTH sizen;
  BLOCK_LENGTH sizee;
  BLOCK_LENGTH size;

  sizen = BN_num_bytes(hostkey->n);
  sizee = BN_num_bytes(hostkey->e);
  size = sizen + sizee+2*BLOCK_LENGTH_SIZE;
  if (size != sizeof(PublicKey)-sizeof(result->padding)) {
    fprintf(stderr,
	    "oops: sizeof public key does not match size (%d!=%d)\n",
	    size, sizeof(PublicKey)-sizeof(result->padding));
    exit(-1);
  }
  if (RSA_KEY_LEN != sizen+sizee) {
    fprintf(stderr,
	    "PublicKey datastructure wrong (%d+%d!=%d)!\n",
	    sizen, sizee, RSA_KEY_LEN);
    exit(-1);
  }
  result->len = htons(size);
  result->sizen = htons(sizen);
  result->padding = 0;
  BN_bn2bin(hostkey->n,
	    &result->key[0]);
  BN_bn2bin(hostkey->e,
	    &result->key[sizen]);
}

/**
 * Internal: publicKey => RSA-Key
 **/
static HOSTKEY public2Hostkey(PublicKey * publicKey) {
  HOSTKEY result;
  int sizen;
  int sizee;

  result = RSA_new();
  if (ntohs(publicKey->len) != sizeof(PublicKey)-sizeof(publicKey->padding)) {
#ifdef PRINT_WARNINGS
    print("public2Hostkey: received invalid publicKey (size=%d)\n",
	  ntohs(publicKey->len));
#endif
    return NULL;
  }
  sizen = ntohs(publicKey->sizen);
  sizee = ntohs(publicKey->len) - sizen - 2*BLOCK_LENGTH_SIZE;
  if ( (sizen != RSA_ENC_LEN) || 
       (sizee + sizen != RSA_KEY_LEN)) {
#ifdef PRINT_WARNINGS
    print("public2Hostkey: received invalid publicKey (sizee=%d, sizen=%d)\n",
	  sizee,sizen);
#endif
    return NULL;
  }
  result->n = BN_bin2bn(&publicKey->key[0], sizen, NULL);
  result->e = BN_bin2bn(&publicKey->key[sizen],sizee, NULL);
  return result;
}

/**
 * Encode the private key in a format suitable for
 * storing it into a file.
 * @returns encoding of the private key.
 *    The first 4 bytes give the size of the array, as usual.
 **/
HostKeyEncoded * encodeHostkey(HOSTKEY hostkey) {
  /*
               BIGNUM *n;               public modulus
               BIGNUM *e;               public exponent
               BIGNUM *d;               private exponent
               BIGNUM *p;               secret prime factor
               BIGNUM *q;               secret prime factor
               BIGNUM *dmp1;            d mod (p-1)
               BIGNUM *dmq1;            d mod (q-1)
               BIGNUM *iqmp;            q^-1 mod p
  */
  BLOCK_LENGTH sizen;
  BLOCK_LENGTH sizee;
  BLOCK_LENGTH sized;
  BLOCK_LENGTH sizep;
  BLOCK_LENGTH sizeq;
  BLOCK_LENGTH sizedmp1;
  BLOCK_LENGTH sizedmq1;
  BLOCK_LENGTH sizeiqmp;
  BLOCK_LENGTH size;
  HostKeyEncoded * retval;

  sizen = BN_num_bytes(hostkey->n);
  sizee = BN_num_bytes(hostkey->e);
  sized = BN_num_bytes(hostkey->d);
  sizep = BN_num_bytes(hostkey->p);
  sizeq = BN_num_bytes(hostkey->q);
  sizedmp1 = BN_num_bytes(hostkey->dmp1);
  sizedmq1 = BN_num_bytes(hostkey->dmq1);
  sizeiqmp = BN_num_bytes(hostkey->iqmp);
  size = sizen+sizee+sized+sizep+sizeq+sizedmp1+sizedmq1+sizeiqmp+sizeof(HostKeyEncoded);
  retval = (HostKeyEncoded *) xmalloc(size,
				      "encodeHostkey: hostkey");
  retval->len = htons(size);
  retval->sizen = htons(sizen);
  retval->sizee = htons(sizee);
  retval->sized = htons(sized);
  retval->sizep = htons(sizep);
  retval->sizeq = htons(sizeq);
  retval->sizedmp1 = htons(sizedmp1);
  retval->sizedmq1 = htons(sizedmq1);
  BN_bn2bin(hostkey->n,&retval->key[0]);
  BN_bn2bin(hostkey->e,&retval->key[0+sizen]);
  BN_bn2bin(hostkey->d,&retval->key[0+sizen+sizee]);
  BN_bn2bin(hostkey->p,&retval->key[0+sizen+sizee+sized]);
  BN_bn2bin(hostkey->q,&retval->key[0+sizen+sizee+sized+sizep]);
  BN_bn2bin(hostkey->dmp1,&retval->key[0+sizen+sizee+sized+sizep+sizeq]);
  BN_bn2bin(hostkey->dmq1,&retval->key[0+sizen+sizee+sized+sizep+sizeq+sizedmp1]);
  BN_bn2bin(hostkey->iqmp,&retval->key[0+sizen+sizee+sized+sizep+sizeq+sizedmp1+sizedmq1]);
  return retval;
}

/**
 * Decode the private key from the file-format back
 * to the "normal", internal format.
 **/
HOSTKEY decodeHostkey(HostKeyEncoded * encoding) {
  BLOCK_LENGTH sizen;
  BLOCK_LENGTH sizee;
  BLOCK_LENGTH sized;
  BLOCK_LENGTH sizep;
  BLOCK_LENGTH sizeq;
  BLOCK_LENGTH sizedmp1;
  BLOCK_LENGTH sizedmq1;
  BLOCK_LENGTH size;
  BLOCK_LENGTH sum;

  HOSTKEY result = (HOSTKEY) RSA_new();
  size    = ntohs(encoding->len) - sizeof(HostKeyEncoded);
  sizen   = ntohs(encoding->sizen);
  sizee   = ntohs(encoding->sizee);
  sized   = ntohs(encoding->sized);
  sizep   = ntohs(encoding->sizep);
  sizeq   = ntohs(encoding->sizeq);
  sizedmp1= ntohs(encoding->sizedmp1);
  sizedmq1= ntohs(encoding->sizedmq1);
  sum = 0;
  result->n= BN_bin2bn(&encoding->key[sum], sizen, NULL); sum += sizen;
  result->e= BN_bin2bn(&encoding->key[sum], sizee, NULL); sum += sizee;
  result->d= BN_bin2bn(&encoding->key[sum], sized, NULL); sum += sized;
  result->p= BN_bin2bn(&encoding->key[sum], sizep, NULL); sum += sizep;
  result->q= BN_bin2bn(&encoding->key[sum], sizeq, NULL); sum += sizeq;
  result->dmp1= BN_bin2bn(&encoding->key[sum], sizedmp1, NULL); sum += sizedmp1;
  result->dmq1= BN_bin2bn(&encoding->key[sum], sizedmq1, NULL); sum += sizedmq1;
  result->iqmp= BN_bin2bn(&encoding->key[sum], size-sum, NULL);
  return result;
}

/**
 * Encrypt a block with the public key of another
 * host that uses the same cyper.
 * @param block the block to encrypt
 * @param size the size of block
 * @param publicKey the encoded public key used to encrypt
 * @param target where to store the encrypted block
 * @returns SYSERR on error, OK if ok
 **/
int encryptHostkey(void * block, 
		   BLOCK_LENGTH size,
		   PublicKey * publicKey,
		   RSAEncryptedData * target) {
  HOSTKEY foreignkey;
  int rs;
  int len;

  foreignkey = public2Hostkey(publicKey);
  if (foreignkey == NULL)
    return SYSERR;
  rs = RSA_size(foreignkey);
  /* now encrypt. First get size of the block */
  if (size > (rs - 41)) {
#ifdef PRINT_WARNINGS
    print("HostKey::encryptHostkey() called with %d bytes where foreignkey allows only %d\n",
	  size, 
	  rs-41);
#endif
    freeHostkey(foreignkey);
    return SYSERR;
  }
  len = RSA_public_encrypt(size, 
			   block, 
			   &target->encoding[0], 
			   foreignkey,
			   RSA_PKCS1_OAEP_PADDING);
  if (len != RSA_ENC_LEN) {
#ifdef PRINT_WARNINGS
    fprintf(stderr,"RSA-Encoding has unexpected length (%d)!",
	    len);
#endif
    freeHostkey(foreignkey);
    return SYSERR;
  }
  freeHostkey(foreignkey);
  return OK;
}

/**
 * Decrypt a given block with the hostkey. 
 * @param hostkey the hostkey with which to decrypt this block
 * @param block the data to decrypt, encoded as returned by encrypt
 * @param result pointer to a location where the result can be stored
 * @param max the maximum number of bits to store for the result, if
 *        the decrypted block is bigger, an error is returned
 * @returns the size of the decrypted block, -1 on error
 **/
int decryptHostkey(HOSTKEY hostkey, 
		   RSAEncryptedData * block,
		   void * result,
		   unsigned int max) {
  RSAEncryptedData tmp; /* this is as big as the result can possibly get */
  int size;

  if (block == NULL)
    return -1;
  size = RSA_private_decrypt(sizeof(RSAEncryptedData), 
			     &block->encoding[0],
			     &tmp.encoding[0], 
			     hostkey,
			     RSA_PKCS1_OAEP_PADDING);
  if ((size == -1) || (size > max))
    return -1;
  memcpy(result,
	 &tmp.encoding[0],
	 size);
  return size;
}

/**
 * Sign a given block.
 * @param hostkey the hostkey with which to sign this block
 * @param size how many bytes to sign
 * @param block the data to sign
 * @param result where to write the signature
 * @return SYSERR on error, OK on success
 **/
int sign(HOSTKEY hostkey, 
	 BLOCK_LENGTH size,
	 void * block,
	 Signature * result) {
#if EXTRA_CHECKS
  PublicKey pkey;
#endif
  int rs = RSA_size(hostkey);
  int sigSize;

  if (block == NULL)
    return SYSERR;
  if (rs != sizeof(Signature)) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "sign: signature length (RSA_size) has unexpected value (%d)!",
	    rs);
#endif
    return SYSERR;
  }
  sigSize = RSA_private_encrypt(size, /* size */
				(unsigned char *)block, /* data to sign */
				&result->sig[0],
				hostkey,
				RSA_PKCS1_PADDING); /* or: RSA_NO_PADDING? */
  if (sigSize != sizeof(Signature)) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "sign: sigSize wrong (%d)!",sigSize);
#endif
    return SYSERR;
  }
#if EXTRA_CHECKS
  getPublicKey(hostkey, &pkey);
  if (SYSERR == verifySig(block, size, result, &pkey)) {
    fprintf(stderr,
	    "FATAL: sign: generated signature does not pass verification!\n");
    exit(-1);
  }
#endif
  return OK;
}

/**
 * Verify signature.
 * @param block the signed data
 * @param len the length of the block 
 * @param sig signature
 * @param publicKey public key of the signer
 * @returns OK if ok, SYSERR if invalid
 **/
int verifySig(void * block,
	      BLOCK_LENGTH len,
	      Signature * sig,	      
	      PublicKey * publicKey) {
  HOSTKEY hostkey;
  int rs;
  int size;
  Signature tmp;
 
  hostkey = public2Hostkey(publicKey);
  if ((hostkey == NULL) || (sig == NULL) || (block == NULL))
    return SYSERR; /* hey, no data !? */
  rs = RSA_size(hostkey);
  if (rs != RSA_ENC_LEN) {
    fprintf(stderr,
	    "verifySig: rs != RSA_ENC_LEN (%d)!",rs);
    exit(-1);
  }
  size = RSA_public_decrypt(RSA_ENC_LEN, /* size */
			    &sig->sig[0], /* signature */
			    &tmp.sig[0],
			    hostkey,
			    RSA_PKCS1_PADDING);
  if (rs < size) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "verifySig: rs < size (%d - %d). Error (in OpenSSL?).\n",
	    rs,size);
#endif
    freeHostkey(hostkey);
    return SYSERR;
  }
  freeHostkey(hostkey);

  if (size != len) {
#if PRINT_WARNINGS
    print("VerifySig: signature mismatch: %d - %d\n",
	  rs, len); 
#endif
    return SYSERR;
  }
  if ((size = memcmp((unsigned char*)block,
		     &tmp.sig[0],
		     size)) != 0) {
#if PRINT_WARNINGS
    print("verifySig: signature mismatch: %d!\n",size); 
#endif
    return SYSERR;   
  }       
  return OK;
}


/* end of hostkey.c */
