/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Module for ip verification
 * @author Christian Grothoff
 * @file util/ipcheck.c
 **/

#include "config.h"
#include "util/returnvalues.h"
#include "util/ipcheck.h"
#include <netinet/in.h>


/**
 * Parse a network specification. The argument specifies
 * a list of networks. The format is
 * <tt>[network/netmask;]*</tt> (no whitespace, must be terminated
 * with a semicolon). The network must be given in dotted-decimal
 * notation. The netmask can be given in CIDR notation (/16) or
 * in dottet-decumal (255.255.0.0).
 * <p>
 * @param routeList a string specifying the forbidden networks
 * @return the converted list, NULL if the synatx is flawed
 **/
CIDRNetwork * parseRoutes(char * routeList) {
  int count;
  int i;
  int j;
  int len;
  int cnt;
  int pos;
  unsigned int temps[8];
  unsigned char * ip;
  unsigned char * mask;
  int slash;
  CIDRNetwork * result;

  len = strlen(routeList);
  count = 0;
  for (i=0;i<len;i++)
    if (routeList[i] == ';')
      count++;
  result = xmalloc(sizeof(CIDRNetwork) * (count+1),
		      "ipCheck: result table (persistent)");
  /* add termination */
  result[count].network.addr = 0;
  result[count].netmask.addr = 0;
  i=0;
  pos = 0;
  while (i < count) {
    ip = (char*) &result[i].network;
    mask = (char*) &result[i].netmask;
    cnt = sscanf(&routeList[pos],
		 "%u.%u.%u.%u/%u.%u.%u.%u;",
		 &temps[0],
		 &temps[1],
		 &temps[2],
		 &temps[3],
		 &temps[4],
		 &temps[5],
		 &temps[6],
		 &temps[7]);
    if (cnt == 8) {
      for (j=0;j<8;j++)
	if (temps[j] > 0xFF) {
	  fprintf(stderr,
		  "ERROR: wrong format for IP: %s\n",
		  &routeList[pos]);
	  return NULL;
	}
      for (j=0;j<4;j++)
	ip[j] = temps[j];
      for (j=4;j<8;j++)
	mask[j-4] = temps[j];
      result[i].network.addr = htonl(result[i].network.addr);
      result[i].netmask.addr = htonl(result[i].netmask.addr);
      while (routeList[pos] != ';')
	pos++;
      pos++;
      i++;
      continue;
    } 
    /* try second notation */
    cnt = sscanf(&routeList[pos],
		 "%u.%u.%u.%u/%u;",
		 &temps[0],
		 &temps[1],
		 &temps[2],
		 &temps[3],
		 &slash);
    if (cnt == 5) {
      for (j=0;j<4;j++)
	if (temps[j] > 0xFF) {
	  fprintf(stderr,
		  "ERROR: wrong format for IP: %s\n",
		  &routeList[pos]);
	  return NULL;
	}
      for (j=0;j<4;j++)
	ip[j] = temps[j];
     if ( (slash <= 32) && (slash > 0) ) {
	result[i].netmask.addr = 0;
	while (slash > 0) {
	  result[i].netmask.addr
	    = 0x80000000 | (result[i].netmask.addr >> 1);
	  slash--;
	}
	while (routeList[pos] != ';')
	  pos++;
	pos++;
	result[i].network.addr = htonl(result[i].network.addr);
	result[i].netmask.addr = htonl(result[i].netmask.addr);
 	i++;
	continue;       
      } else {
	fprintf(stderr,
		"invalid network notation (/%d is not legal in IPv4 CIDR!)",
		slash);
	return NULL; /* error */
      }
    } 
    fprintf(stderr,
	    "invalid network notation: >>%s<<",
	    &routeList[pos]);
    return NULL; /* error */
  }
  if (pos < strlen(routeList))
    return NULL; /* oops */
  return result; /* ok */
}

/**
 * Check if the given IP address is in the list of 
 * IP addresses.
 * @param list a list of networks
 * @param ip the IP to check
 * @return NO if the IP is not in the list, YES if it it is
 **/
int checkIPListed(CIDRNetwork * list,
		  struct in_addr ip) {
  int i;
  IPaddr add;
  
  add.addr = ntohl( *(unsigned int*) &ip);
  i=0;
  while ( (list[i].network.addr != 0) ||
	  (list[i].netmask.addr != 0) ) {
    if ( (add.addr & list[i].netmask.addr) == 
	 (list[i].network.addr & list[i].netmask.addr) ) {
      return YES;
    }
    i++;
  }
  return NO;
}


/* end of ipcheck */
