/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * A simple counting semaphore, implemented with pthreads.
 * @file util/semaphore.c
 **/

#include "config.h"
#include "util/semaphore.h"

extern int pthread_mutexattr_setkind_np(pthread_mutexattr_t *attr, int kind);

/**
 * function just prints an error message and exits 
 **/
static void do_error (char *msg) {
  perror (msg);
#if DEBUG_LOCKING /* Want to be able to examine stack if debugging */
  kill(getpid(), 9);
#endif
  exit (1);
}

/* -------------------------------------------------------------------- */
/* The following functions replace standard library functions in that   */
/* they exit on any error returned from the system calls.  Saves us     */
/* from having to check each and every call above.                      */
/* -------------------------------------------------------------------- */


static int tw_pthread_mutex_unlock (pthread_mutex_t * m) {
  int return_value;

  if ((return_value = pthread_mutex_unlock (m)) != 0)
    do_error ("pthread_mutex_unlock");
  
  return (return_value);
}

static int tw_pthread_mutex_lock(pthread_mutex_t * m) {
  int return_value;
  
  if ((return_value = pthread_mutex_lock (m)) != 0)
    do_error ("pthread_mutex_lock");  
  return (return_value);
}
static int tw_pthread_cond_wait(pthread_cond_t * c, 
				pthread_mutex_t * m) {
  int  return_value;
  
  if ((return_value = pthread_cond_wait (c, m)) != 0)
    do_error ("pthread_cond_wait");
  return (return_value);
}

static int tw_pthread_cond_signal (pthread_cond_t * c) {
  int return_value;

  if ((return_value = pthread_cond_signal (c)) != 0)
    do_error ("pthread_cond_signal");  
  return (return_value);
}


void create_mutex(Mutex * mutex) {
  if ( pthread_mutex_init(&mutex->pthreadMutex, NULL) != 0)
    do_error("pthread_mutex_init");
}

void create_recursive_mutex(Mutex * mutex) {
  pthread_mutexattr_t attr;

  pthread_mutexattr_init(&attr);
#ifndef SOMEBSD
  pthread_mutexattr_setkind_np(&attr,  
			       PTHREAD_MUTEX_RECURSIVE_NP);
#else
  pthread_mutexattr_setkind_np(&attr,
			       PTHREAD_MUTEX_RECURSIVE);
#endif
  if ( pthread_mutex_init(&mutex->pthreadMutex, &attr) != 0)
    do_error("pthread_mutex_init");
}
void destroy_mutex(Mutex * mutex) {
  pthread_mutex_destroy(&mutex->pthreadMutex);
}

void mutex_lock(Mutex * mutex) {
  if ( pthread_mutex_lock(&mutex->pthreadMutex) != 0)
    do_error("pthread_mutex_lock");
}

void mutex_unlock(Mutex * mutex) {
  if ( pthread_mutex_unlock(&mutex->pthreadMutex) != 0)
    do_error("pthread_mutex_unlock");
}




/**
 * function must be called prior to semaphore use -- handles
 * setup and initialization.  semaphore destroy (below) should
 * be called when the semaphore is no longer needed.
 **/
Semaphore * new_semaphore (int value) {
  Semaphore * s = xmalloc(sizeof(Semaphore),
			  "new_semaphore: semaphore");
  s->v = value;
  if (pthread_mutex_init (&(s->mutex), NULL) != 0)
    do_error ("pthread_mutex_init");

  if (pthread_cond_init (&(s->cond), NULL) != 0)
    do_error ("pthread_cond_init");
  return s;
}

/**
 * function increments the semaphore and signals any threads that
 * are blocked waiting a change in the semaphore.
 **/
int semaphore_up (Semaphore * s) {
  int value_after_op;

  tw_pthread_mutex_lock (&(s->mutex));  
  (s->v)++;
  value_after_op = s->v;  
  tw_pthread_mutex_unlock (&(s->mutex));
  tw_pthread_cond_signal (&(s->cond));  
  return (value_after_op);
}


/**
 * function decrements the semaphore and blocks if the semaphore is
 * <= 0 until another thread signals a change.
 **/
int semaphore_down (Semaphore * s) {
  int value_after_op;
  
  tw_pthread_mutex_lock (&(s->mutex));
  while (s->v <= 0) {
    tw_pthread_cond_wait (&(s->cond), &(s->mutex));
  }
  (s->v)--;
  value_after_op = s->v;
  tw_pthread_mutex_unlock (&(s->mutex));  
  return (value_after_op);
}


