/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/** 
 * Status calls implementation.
 * @author Tzvetan Horozov
 * @author Christian Grothoff
 * @file util/statuscalls.c
 **/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "util/statuscalls.h"

/**
 * Define the maximum allowed network interfaces 
 **/
#define MAX_INTERFACES 20


static char * interfacePtrs[MAX_INTERFACES]; /* Pointer to each interface */
static int numInterfaces = 0;
static int maxNetLoad;

/**
 * The following method is called in order to initialize the status calls
 * routines. After that it is safe to call each of the status calls separately
 * @return 1 on success and -1 on error.
 **/
int initStatusCalls(Interfaces interfaces,
		    int maxNet) {
  char *ch=interfaces;
  int start=-1, done=-1;
#ifdef PRINT_UDP
  int i;
#endif
  maxNetLoad = maxNet;

  /* fail gracefully if config-file is incomplete */
  if (interfaces == NULL) {
    fprintf(stderr,
	    "no network interfaces defined!\n");
    exit(-1);
  }

  /* The string containitn the interfaces is formatted in the following way: 
   * each comma is replaced by '\0' and the pointers to the beginning of every 
   * interface are stored
   */
  while (done<0){
    if (*ch=='\0'){
      done=1;      
      if (start>0){	
	numInterfaces++;
      }	      
    }
    else if ((*ch>='a'&&*ch<='z') || (*ch>='A'&&*ch<='Z') || (*ch>='0'&&*ch<='9')){
      if (start<0){
	start=1;   
	interfacePtrs[numInterfaces]=ch;	
      }
    }
    else {
      if (start>0){
	start=-1;
	*ch='\0';	
	numInterfaces++;
      }    
    }
    ch++;
  }
  
  if (numInterfaces<=0) {
    fprintf(stderr,
	    "ERROR: No network interfaces specified in he configuration file\n");
    exit(-1);
  }
  if (numInterfaces> MAX_INTERFACES) {
    fprintf(stderr,
	    "ERROR: Too many network interfaces specified in the configuration file. Must be less than %d\n",
	    MAX_INTERFACES);
    exit(-1);
  }


#ifdef PRINT_UDP  
  fprintf(stderr,
	  "Total interfaces=%d\n",numInterfaces);    
  for (i=0;i<numInterfaces;i++){
    fprintf(stderr,
	    " -> %s\n",(char*)interfacePtrs[i]);
  }
#endif

  /* Call the three methods once to initialize the sequence */
  networkUsage(); 
  cpuUsage();
  sleep(3);
  return 1;  
}

/**
 * The following routine returns the percentage of
 * available used bandwidth.
 * Example: If 81 is returned this means that 81% of the network 
 * bandwidth of the host is consumed.
 * The method initStatusCalls() should be called before this routine is 
 * invoked
 * If there is an error the method returns -1
 **/
int networkUsage() {
#define MAX_PROC_LINE 5000
  static time_t lastnettime=0; /* In seconds */
  static int lastnetresult=-1;
  static int firstnetcall=0;
  static unsigned int last_net_results[MAX_INTERFACES][4];
  char line[MAX_PROC_LINE];
  unsigned int rxnew, txnew, collnew, found=0, rxdiff=0, txdiff=0, i=0;
  int ifnum=0;
  time_t now, elapsedtime;
  char curInterface[255];
  char *data;

#ifdef LINUX
  FILE* proc_net_dev;
#else
  FILE *command;
#endif

  /* Get the current time in microseconds */
  time(&now);  
  elapsedtime=now-lastnettime;
  if ((elapsedtime < 2) && (lastnetresult != -1))
    return lastnetresult;
  lastnettime=now;

#ifdef LINUX
  proc_net_dev = fopen("/proc/net/dev", "r");
  /* Try to open the file*/
  if (!proc_net_dev) {
    fprintf(stderr,
	    "ERROR: Could not open /proc/net/dev (%d)\n", 
	    (int)proc_net_dev);
    return -1;
  }
 
  /* Parse the line with 'eth0' */
  while (!feof(proc_net_dev)&& ifnum<numInterfaces) {    
    fgets(line, MAX_PROC_LINE, proc_net_dev);
    
    for (i=0;i<numInterfaces;i++){
      strcpy(curInterface,interfacePtrs[i]);
      strcat(curInterface,":");   
      found=(int)strstr(line,curInterface);
      if (found){
	data=(char*)strchr(line,':');
	data++;	
	if(sscanf(data,"%u %u %u %u %u %u %u %u %u",
		  &rxnew,&txnew,&collnew,&txnew,&txnew,&txnew,&txnew ,&txnew ,&txnew)<=0){
	  fclose(proc_net_dev);
	  fprintf(stderr,
		  "ERROR: reading interface data \n");
	  exit(-1);
	}
        /*
	  fprintf(stderr,"%s:Rx=%d, Tx=%d, Col=%d\n",
	  curInterface,rxnew, txnew, collnew);
	*/
	if (rxnew != last_net_results[ifnum][0]){
	  rxdiff+=rxnew-last_net_results[ifnum][0];	  
	  last_net_results[ifnum][0]=rxnew;
	}
	
	if (txnew != last_net_results[ifnum][1]){
	  txdiff+=txnew-last_net_results[ifnum][1];
	  last_net_results[ifnum][1]=txnew;
	}	
	ifnum++;
      	break;
      }
    }
  }
  fclose(proc_net_dev);
#else
  if ( ( command = popen("netstat -n -f inet -i", "r") ) == NULL ) {
    fprintf(stderr,
            "ERROR: could not open netstat\n");
    return -1;
  }

  while ( !feof(command) && ifnum < numInterfaces ) {
    fgets(line, MAX_PROC_LINE, command);

    for ( i = 0 ; i < numInterfaces ; i++ ) {

      strcpy(curInterface, interfacePtrs[i]);
      found=(int)strstr(line, curInterface);

      if ( found ) {
          if(sscanf(line,"%s %*s %*s %*s %u %*s %u %*s %*s", 
		    curInterface, &rxnew, &txnew) <=0 ) {
            pclose(command);
            fprintf(stderr,
                    "ERROR: reading interface data\n");
            exit(-1);
          }
          if ( rxnew != last_net_results[ifnum][0]) {
            rxdiff+=rxnew-last_net_results[ifnum][0];
            last_net_results[ifnum][0]=rxnew;
          }

          if ( txnew != last_net_results[ifnum][1] ) {
            txdiff+=txnew-last_net_results[ifnum][1];
            last_net_results[ifnum][1]=txnew;
          }
          ifnum++;
          break;
      }
    }
  }
  pclose(command);
#endif

  /* If this is the first invocation to that function then return */
  if (firstnetcall==0){
    firstnetcall=1;
    return -1;
  }

  lastnetresult = ((100*(rxdiff+txdiff))/elapsedtime)/maxNetLoad;
  /*fprintf(stderr,
    "NETWORK:current %d Bps, rxdiff=%d %d, txdiff=%d %d, maxnet=%d,
    elapsedtime=%d\n",(rxdiff+txdiff)/elapsedtime,rxdiff,rxnew,txdiff,
    txnew,maxNetLoad,elapsedtime);
   */
  return lastnetresult;  
}


/**
 * The following routine returns a number between 0-100 (can be larger than 100
 * if the load is > 1) which indicates the percentage CPU usage.  
 * 
 * Before its first invocation the method initStatusCalls() must be called.
 * If there is an error the method returns -1
 **/
int cpuUsage(){
#ifndef HAVE_GETLOADAVG
#ifdef LINUX
  static long lastcputime=0; /* In seconds */
  static int lastcpuresult=-1;
  static int firstcpucall=0;
  int ret=-1;
  struct timeval t1;
  long now, elapsedtime;
  static int last_cpu_results[4] = { 0, 0, 0, 0 };
  FILE* proc_stat;
  char line[128];
  int user_read, system_read, nice_read, idle_read;
  int user, system, nice, idle;
  int usage_time=0, total_time=1;
#endif
#else
  double loadavg;
#endif

 
#ifndef HAVE_GETLOADAVG
#ifdef LINUX
  /* Get the current time in microseconds */
  gettimeofday(&t1,NULL);
  now=t1.tv_sec;
  elapsedtime=now-lastcputime;
  if (elapsedtime < 2)
    return lastcpuresult;
  lastcputime=now;

  proc_stat = fopen("/proc/stat", "r");
  /* Try to open the file*/
  if (! proc_stat) {
    fprintf(stderr,
	    "ERROR: Could not open /proc/stat");
    return -1;
  }
  /* Get the first line with the data */
  if(fgets(line, 128, proc_stat)==NULL) {
    fprintf(stderr,
	    "ERROR: reading /proc/stat file");
    fclose(proc_stat);
    return -1;
  }
  if(sscanf(line, "%*s %i %i %i %i", &user_read, &system_read, &nice_read,
	    &idle_read)<=0) {
    fclose(proc_stat);
    fprintf(stderr,
	    "ERROR: decoding /proc/stat file");
    return -1;
  }

  /* Store the current usage*/
  user = user_read - last_cpu_results[0];
  system = system_read - last_cpu_results[1];
  nice = nice_read - last_cpu_results[2];
  idle = idle_read - last_cpu_results[3];
  
  /* Calculate the % usage */
  if ((user + system + nice + idle) > 0) {
    usage_time = user + system + nice;
    total_time = usage_time + idle;
  }
  ret= (100 * usage_time)/ total_time;

  /* Store the values for the next calculation*/
  last_cpu_results[0] = user_read;
  last_cpu_results[1] = system_read;
  last_cpu_results[2] = nice_read;
  last_cpu_results[3] = idle_read;
  fclose(proc_stat);
  /* If this is the first invocation to that function then return */
  if (firstcpucall==0){
    firstcpucall=1;    
    return -1;
  }
  /* If the method was called less that 1 second ago then return the old value
   */
  if (elapsedtime<=1){
    if (lastcpuresult >= 0)  
      return lastcpuresult;
    /* If no last time was ever obtained, retunr ERROR */
    else 
      return -1;
  }
  lastcpuresult=ret;
  return ret;
#else
  /* loadaverage not defined and not linux: error */
  return -1;
#endif
#else
  if ( getloadavg(&loadavg, 1) == -1 ) {
    fprintf(stderr,
            "ERROR: getting load average\n");
    return -1;
  }
  return (int) (100 * loadavg);
#endif
}

/* end of statusCalls.c */
