/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Storage management - IO abstraction
 * @author Christian Grothoff
 * @file util/storage.c
 **/

#include "config.h"
#include "util/storage.h"
#include "util/xmalloc.h"

#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <dirent.h>
#include <errno.h>

static int atoo(char *s) {
  int n = 0;
  while ( '0' <= *s && *s < '8' ) {
    n <<= 3;
    n += *s++ - '0';
  }
  return n;
}

/**
 * Assert that fil corresponds to a filename
 * (of a file that exists and that is not a directory).
 * @returns 1 if yes, 0 if not (will print an error
 * message in that case, too).
 **/
int assertIsFile(FileName fil) {
  struct stat filestat;
  int ret;

  ret = stat(fil, &filestat);
  if (ret != 0) {
    perror("stat");
    fprintf(stderr,
	    "Can not stat %s.\n",
	    fil);
    return 0;
  }
  if (!S_ISREG(filestat.st_mode)) {
    fprintf(stderr,
	    "%s is not a regular file.\n",
	    fil);
    return 0;
  }  

  if ( access(fil, R_OK) < 0 ) {
    perror("access");
    return 0;
  }

  return 1;
}

/**
 * Complete filename (a la shell) from abbrevition.
 * @param fil the name of the file, may contain ~/ or 
 *        other shell stuff. Will NOT be freed!
 * @returns the full file name, expanded with wordexp,
 *          NULL is returned on error
 **/
FileName expandFileName(FileName fil) {
  FileName fm;
  FileName fn;

  if (fil == NULL)
    return NULL;
  
  if (fil[0] == '~') {
    fm = getenv("HOME");
    if (fm == NULL)
      fm = "$HOME"; /* keep it symbolic to show error to user! */
    fn = xmalloc(strlen(fm) + strlen(fil) + 1,
		 "expandFileName: fn");
    fn[0] = 0;
    strcat(fn, fm);
    strcat(fn, "/");
    strcat(fn, &fil[1]); /* do not copy '~' */
    return fn;
  }
  if (fil[0] == '/') {
    /* absolute path, just copy */
    fn = xmalloc(strlen(fil) + 1,
		 "expandFileName: fn (2)");
    strcpy(fn, fil);
    return fn;
  }
  fm = getenv("PWD");
  if (fm == NULL)
    fm = "$PWD";
  fn = xmalloc(strlen(fm) + 1 + strlen(fil) + 1,
	       "expandFileName: fn (3)");
  fn[0] = 0;
  strcat(fn, fm);
  strcat(fn, "/");
  strcat(fn, fil);
  return fn;
}

/**
 * implementation of "mkdir -p"
 * @param dir the directory to create
 **/
void mkdirp(char * dir) {
  FileName rdir;
  int len;
  int pos;

  rdir = expandFileName((FileName)dir); /* expand directory */
  len = strlen(rdir);
  pos = 0;
  while (pos < len) {
    if (rdir[pos] == '/') {
      rdir[pos] = 0;
      mkdir(rdir,S_IRUSR | S_IWUSR | S_IXUSR);      
      rdir[pos] = '/';       
    }      
    pos++;
  }   
  xfree(rdir,"mkdirp: expandedDirName");
}

/**
 * Build a filename from directory and filename, completing like the shell does
 * @param dir the name of the directory, may contain ~/ or other shell stuff. Will 
 *        NOT be freed!
 * @param fil the name of the file, will NOT be deallocated anymore!
 * @param result where to store the full file name (must be large enough!)
 **/
void buildFileName(char * dir,
		   HexName * fil,
		   FileName result) {
  if ((dir == NULL) || (fil == NULL) || (result == NULL)) {
    fprintf(stderr,
	    "buildFileName called with NULL in arguments");
    exit(-1);
  }
  result[0]=0;
  strcat(result,(char*)dir);
  strcat(result,(char*)fil);
}

/**
 * Read the contents of a binary file into a buffer.
 * @param fileName the name of the file, not freed,
 *        must already be expanded!
 * @param len the maximum number of bytes to read
 * @param result the buffer to write the result to
 * @return the number of bytes read on success, -1 on failure
 **/ 
int readFile(FileName fileName,
	     int  len,
	     void * result) {
  /* open file, must exist, open read only */
  int handle;  
  int size;

  if ((fileName == NULL) || (result == NULL))
    return -1;
  handle = open(fileName,O_RDONLY,S_IRUSR);
  if (handle < 0)
    return -1; 
  size = read(handle, result, len);
  close(handle);
  return size;  
}

/**
 * Write a buffer to a file.
 * @param fileName the name of the file, NOT freed!
 * @param buffer the data to write
 * @param n number of bytes to write
 * @param mode permissions to set on the file
 **/ 
void writeFile(FileName fileName, 
	       void * buffer,
	       int n,
	       char *mode) {
  int handle;
  /* open file, open with 600, create if not 
     present, otherwise overwrite */  
  if ((fileName == NULL) || (buffer == NULL))
    return;
  handle = open(fileName,O_CREAT|O_WRONLY,S_IRUSR|S_IWUSR);
  /* write the buffer take length from the beginning */
  if (n != write(handle,buffer,n)) 
    fprintf(stderr,
	    "Writing %d bytes to file %s failed!\n",
	    n, fileName);
  chmod(fileName, atoo(mode));
  close(handle);
}

/**
 * Scan a directory for files. The name of the directory
 * must be expanded first (!).
 * @param dirName the name of the directory
 * @param callback the method to call for each file,
 *        can be NULL, in that case, we only count
 * @param data argument to pass to callback
 * @return the number of files found, -1 on error
 **/
int scanDirectory(FileName dirName,
		  void (*callback)(HexName *, void *),
		  void * data) {
  DIR * dinfo;
  struct dirent *finfo;
  struct stat istat;
  int count = 0;
 
  if (dirName == NULL) 
    return -1; 
  stat((char*)dirName,&istat);
  if (!S_ISDIR(istat.st_mode)) {
#ifdef PRINT_WARNINGS
    fprintf(stderr,
	    "scanDirectory must be invoked on a directory (%s)!\n",
	    dirName);
#endif
    return -1;
  }
  errno = 0;
  dinfo = opendir((char*)dirName);
  if ((errno == EACCES) || (dinfo == NULL)) {
    fprintf(stderr,
	    "scanDirectory: access denied (%s)\n",
	    (char*)dirName);
    return -1;
  }
  while ((finfo = readdir(dinfo)) != NULL)
    if (strlen(finfo->d_name) == sizeof(HashCode160)*2) {	
      if (callback != NULL)
	callback((HexName*) finfo->d_name,
		 data);
      count++;
    }
  closedir(dinfo);
  return count;
}

/* end of storage.c */
