/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Generic TCP code.
 * @author Christian Grothoff
 **/

#include "config.h"
#include "util/tcpio.h"
#include "util/xmalloc.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <netinet/in.h>

#define DEBUG_TCPIO 0

/**
 * Catch sigpipe on send, and do nothing
 **/
static void catcher(int sig) {
#if PRINT_INFO
  fprintf(stderr, "signal %d caught\n", sig);
#endif
  signal(sig, catcher);
}

/**
 * Initialize TCPIO module.
 * Installs a signal handler for SIGPIPE because read/write
 * operations on sockets can send that signal and we would
 * like to just ignore it.
 **/
void initTCPIO() {
  if (SIG_ERR == signal(SIGPIPE, SIG_IGN)) {
    /* retry: install 'method'  as signal handler */
    if (SIG_ERR == signal(SIGPIPE, catcher))
      fprintf(stderr,
	      "WARNING: could not install handler for SIGPIPE!\nAttempting to continue anyway.");
  }
}

/**
 * Shutdown TCPIO module.
 **/
void doneTCPIO() {
  /* should we restore the signal handler here? */
}

/**
 * Initialize a GNUnet client socket.
 * @param port the portnumber in host byte order
 * @param hostname the name of the host to connect to
 * @param result the SOCKET (filled in)
 * @return OK if successful, SYSERR on failure
 **/
int initGNUnetClientSocket(unsigned short port,
			   char * hostname,
			   GNUNET_TCP_SOCKET * result) {
  result->ip_info = gethostbyname(hostname);
  if (result->ip_info == NULL) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "ERROR: could not find IP of %s\n",
	    hostname);
#endif
    return SYSERR;
  }
  result->port = port;
  result->socket = -1; /* closed */
  return OK;
}

/**
 * Initialize a GNUnet server socket.
 * @param sock the open socket
 * @param result the SOCKET (filled in)
 * @return OK (always successful)
 **/
int initGNUnetServerSocket(int sock,
			   GNUNET_TCP_SOCKET * result) {
  result->ip_info = NULL;
  result->port = 0;
  result->socket = sock;
  return OK;
}

/**
 * Check if a socket is open. Will ALWAYS return 'true'
 * for a valid client socket (even if the connection is
 * closed), but will return false for a closed server socket.
 * @return 1 if open, 0 if closed
 **/
int isOpenConnection(GNUNET_TCP_SOCKET * sock) {
  return ( (sock->socket != -1) ||
	   (sock->ip_info != NULL) ); 
}

/**
 * Check a socket, open and connect if it is closed and it is a client-socket.
 **/
static int checkSocket(GNUNET_TCP_SOCKET * sock) {
  int res;
  struct sockaddr_in soaddr;

  if (sock->socket != -1)
    return OK;
  if (sock->ip_info == NULL)
    return SYSERR;
  sock->socket = socket(PF_INET,SOCK_STREAM,6);/* 6: TCP */
  if (sock->socket == -1) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "Can not open socket. Is the server running?\n");
#endif
    return SYSERR;
  }

  soaddr.sin_family=AF_INET;
  soaddr.sin_addr.s_addr=((struct in_addr*)(sock->ip_info->h_addr))->s_addr;
  soaddr.sin_port=htons(sock->port);
  res = connect(sock->socket, 
		(struct sockaddr*)&soaddr,
		sizeof(soaddr));
  if (res < 0) {
#if PRINT_WARNINGS
    fprintf(stderr,
	    "Can not connect: %d\n",
	    res);
#endif
    close(sock->socket);
    sock->socket = -1;
    return SYSERR;
  }
  return OK;
}

/**
 * Write to a GNUnet TCP socket.
 * @param sock the socket to write to
 * @param buffer the buffer to write
 * @return OK if the write was sucessful, otherwise SYSERR.
 **/
int writeToSocket(GNUNET_TCP_SOCKET * sock,
		  TCP_SOCKET_BUFFER * buffer) {
  int res;
  int pos;
  int size;

  if (SYSERR == checkSocket(sock))
    return SYSERR;
  size = ntohs(buffer->size);
#if DEBUG_TCPIO
  fprintf(stderr,
	  "writing %d bytes to socket %d\n",
	  size, 
	  sock->socket);
#endif
  pos = 0;

  while (pos < size) {
#ifndef LINUX
    res = send(sock->socket,
	       &((char*)buffer)[pos],
	       size-pos,
	       0 /*MSG_DONTWAIT*/);
#else
    res = send(sock->socket,
	       &((char*)buffer)[pos],
	       size-pos,
	       MSG_NOSIGNAL/*|MSG_DONTWAIT*/);
#endif
      if (res < 0) {
	closeSocketTemporarily(sock);
	return SYSERR;
      }
      pos += res;
  }  
#if DEBUG_TCPIO
  fprintf(stderr,
	  "writing %d bytes to socket %d DONE\n",
	  size, 
	  sock->socket);
#endif

  return OK;
}

/**
 * Read from a GNUnet TCP socket.
 * @param sock the socket
 * @param buffer the buffer to write data to
 * @return OK if the read was successful, SYSERR if the socket
 *         was closed by the other side (if the socket is a
 *         client socket and is used again, tcpio will attempt
 *         to re-establish the connection [temporary error]).
 **/
int readFromSocket(GNUNET_TCP_SOCKET * sock,
		   TCP_SOCKET_BUFFER * buffer) {
  int res;
  int pos;
  char * buf;
  int size;

  if (SYSERR == checkSocket(sock))
    return SYSERR;

  buf = (char*) buffer;
  pos = 0;
  res = 0;
  while (pos < sizeof(TCP_HEADER)) {
#ifndef LINUX
    res = recv(sock->socket,
	       &((char*)buffer)[pos],
	       sizeof(TCP_HEADER)-pos,
	       MSG_WAITALL);
#else
    res = recv(sock->socket,
	       &((char*)buffer)[pos],
	       sizeof(TCP_HEADER)-pos,
	       MSG_WAITALL|MSG_NOSIGNAL);
#endif
    if (res <= 0)
      break;    
    pos += res;
  }
  size = ntohs(buffer->size);
  if ( (pos != sizeof(TCP_HEADER)) ||
       (size > MAX_TCP_RECORD_SIZE) ) {
#if DEBUG_TCPIO
    fprintf(stderr,
	    "invalid read on socket (%d, %d, %d, %d), closing\n",
	    size, pos, res, sock->socket);
#endif
    closeSocketTemporarily(sock);
    return SYSERR; /* other side closed socket or invalid header */
  }
#if DEBUG_TCPIO
  fprintf(stderr,
	  "reading %d bytes from socket\n",
	  size);
#endif
  while (pos < size) {
#ifndef LINUX
    res = recv(sock->socket,
	       &((char*)buffer)[pos],
	       size-pos,
	       MSG_WAITALL);
#else
    res = recv(sock->socket,
	       &((char*)buffer)[pos],
	       size-pos,
	       MSG_WAITALL|MSG_NOSIGNAL);
#endif
    if (res < 0) {  /* error, abort */
      closeSocketTemporarily(sock);
      return SYSERR;
    }    
    pos += res;
    if (pos == size)
      break;
    if (res == 0) { /* other side closed connection, but we're not done yet */
      closeSocketTemporarily(sock);
      return SYSERR;
    }
  }
#if DEBUG_TCPIO
  fprintf(stderr,
	  "successfully received %d bytes from socket\n",
	  size);
#endif  
  return OK; /* success */
}

/**
 * Close a GNUnet TCP socket for now (use to temporarily close
 * a TCP connection that will probably not be used for a long
 * time; the socket will still be auto-reopened by the
 * readFromSocket/writeToSocket methods if it is a client-socket).
 **/
void closeSocketTemporarily(GNUNET_TCP_SOCKET * sock) {
  if (sock->socket != -1) {
#if DEBUG_TCPIO
    fprintf(stderr,
	    "closing socket\n");
#endif
    close(sock->socket);
    sock->socket = -1;
  }
}

/**
 * Destroy a socket for good. If you use this socket afterwards,
 * you must first invoke initializeSocket, otherwise the operation
 * will fail.
 **/
void destroySocket(GNUNET_TCP_SOCKET * sock) {
  closeSocketTemporarily(sock);
  if (sock->ip_info != NULL) {
    sock->ip_info = NULL; /* do NOT free, will segfault, was obtained from gethostbyname! */
  }
}


/*  end of tcpio.c */
