#!/usr/bin/perl -w
package Control::Curve;
use strict;

=head NAME - Control::Curve

Implements user-defined control curves

=head $Id: Curve.pm,v 1.12 2001/11/02 06:04:24 tramm Exp $

=cut


sub new
{
	my $class		= shift;
	my $name		= shift;

	my %points		= (
		-128		=> -128,
		127		=> 127,
		128		=> 127,
		@_,
	);


	my $curve		= bless {
		name		=> $name,
		current		=> 0,
		y		=> 0,
		points		=> \%points,
	}, $class;

	$curve->recalc;
	return $curve;
}


#
# Rebuilds the internal precomputed points for each servo
# position.
#
sub recalc
{
	my $curve		= shift;
	my @data;
	my ($x1,@x)		= sort { $a <=> $b } keys %{ $curve->{points} };
	my $y1			= $curve->{points}->{$x1};

	while( @x )
	{
		my $x2		= shift @x;
		my $y2		= $curve->{points}->{$x2};
		my $m		= ($y1 - $y2) / ($x1 - $x2);

		for( ; $x1 < $x2 ; $x1++ ) {
			$y1 = $m * ($x1 - $x2) + $y2;
			push @data, int $y1;
		}
	}

	$curve->{data} = \@data;
}


sub draw
{
	my $curve		= shift;
	my $mw			= shift;
	my $width		= shift || $curve->{width};
	my $height		= shift || $curve->{height};
	my $f;

	# Store these; we'll need them later
	$curve->{width}		= $width;
	$curve->{height}	= $height;

	$f			= $mw->Frame(
		-relief		=> 'sunken',
		-border		=> 3,
	);

	$f->Label(
		-text		=> $curve->{name},
		-width		=> 8,
	)->pack(
		-side		=> 'top',
		-expand		=> 0,
	);

	$curve->{canvas}	= $f->Canvas(
		-width		=> $width,
		-height		=> $height,
		-background	=> 'black',
	)->pack(
		-side		=> 'top',
		-anchor		=> 'nw',
		-expand		=> 0,
		-fill		=> 'both',
	);

	$f->Label(
		-textvariable	=> \$curve->{y},
		-width		=> 4,
		-justify	=> 'right',
		-anchor		=> 'e',
	)->pack(
		-side		=> 'right',
		-expand		=> 0,
		-anchor		=> 'e',
	);

	$f->Label(
		-textvariable	=> \$curve->{current},
		-width		=> 4,
		-justify	=> 'right',
		-anchor		=> 'e',
	)->pack(
		-side		=> 'right',
		-expand		=> 0,
		-anchor		=> 'e',
	);

	$curve->draw_points;
	$curve->draw_line;
	return $f;
}


sub draw_points
{
	my $curve		= shift;
	my $c			= $curve->{canvas}
		or return;
	my $width		= $curve->{width};
	my $height		= $curve->{height};

	# Delete any if we have already drawn them
	$c->delete( $_ )
		for @{ $curve->{tk_points} };

	# Draw the control points
	my @dots;
	for my $x (keys %{ $curve->{points} } )
	{
		my $y = $curve->{points}->{$x};

		my ($sx,$sy) = $curve->scale( $x, $y );

		my $dot = $c->createRectangle(
			$sx-3, $sy-3,
			$sx+3, $sy+3,
			-fill		=> 'green',
		);

		$c->bind( $dot, '<B1-Motion>', sub {
			my $mw		= shift;
			my $ev		= $mw->XEvent;
			my $nx		= $ev->x;
			my $ny		= $ev->y;

			# Only go vertical with this one
			# Don't calc a new x value, and set dx to 0
			$y		= 128 - $ny * 256 / $height;
			# $x		= 128 + $nx * 256 / $width;

			$curve->set_point( $x, $y );
			$curve->draw_line;

			my $dy		= $ny - $sy;
			#my $dx		= $nx - $sx;
			my $dx		= 0;

			$sy		= $ny;
			#sx		= $nx;

			$c->move( $dot, $dx, $dy );
		});

		push @dots, $dot;
	}

	$curve->{dots} = \@dots;
}
	
sub draw_line
{
	my $curve		= shift;
	my $c			= $curve->{canvas}
		or return;
	my $width		= $curve->{width};
	my $height		= $curve->{height};

	# Delete one if it is already there
	$c->delete( $curve->{line} )
		if $curve->{line};

	my $x = -128;
	$curve->{line} = $c->createLine(
		map(
			$curve->scale( $x++, $_ ),
			@{ $curve->{data} }
		),
		-fill		=> 'white',
	);
}


sub draw_current
{
	my $curve		= shift;
	my $c			= $curve->{canvas}
		or return;
	my $x			= $curve->{current};
	my $y			= $curve->y( $x );
	my ($sx,$sy)		= $curve->scale( $x, $y );

	$c->delete( $curve->{cur_dot} )
		if $curve->{cur_dot};

	$curve->{cur_dot} = $c->createRectangle(
		$sx-2, $sy-2,
		$sx+2, $sy+2,
		-fill		=> 'red',
	);

	return $y;
}

sub set_point
{
	my $curve		= shift;
	my $x			= shift;
	my $y			= shift;

	return if $curve->{points}->{$x} == $y;

	$curve->{points}->{$x} = $y;
	$curve->recalc;
}

sub y
{
	my $curve		= shift;
	my $x			= shift;
	$x = -128 if $x < -128;
	$x =  127 if $x >  127;

	return $curve->{data}->[int $x + 128];
}

sub scale
{
	my $curve		= shift;
	my $x			= shift;
	my $y			= shift;

	return (
		(128 + $x) * $curve->{width}  / 256,
		(128 - $y) * $curve->{height} / 256,
	);
}

sub current
{
	my $curve		= shift;
	my $x			= shift;

	$curve->{current} = int $x;
	$curve->draw_current;

	return $curve->{y} = $curve->y( $x );
}


"0, but true";
__END__
