# 
# This file is licensed under the GPL.  Please see the file "LICENSE" in
# the root directory of this project for terms and conditions.
# (c) 2001 by Trammell Hudson <hudson@swcp.com>
#
use strict;

# $Id: EFIS.pm,v 1.4 2001/10/28 20:33:09 tramm Exp $

=head NAME - Hover::EFIS

A Tk widget that displays a simulated EFIS (Electronic Flight
Information System) for remote control aircraft.

=head METHOD - $efis->set_roll()

=head METHOD - $efis->set_pitch()

=head METHOD - $efis->set_yaw()

=head METHOD - $efis->set_speed()

=head METHOD - $efis->set_altitude()

=cut


=head NAME - Hover::EFIS::AI

The Attitude Indicator or "Aritifical Horizon" is the primary flight
instrument for maintaining level flight.

=cut

package EFIS::AI;
sub new
{
	my $class		= shift;
	my $c			= shift;
	my $x			= shift;
	my $y			= shift;
	my $h			= shift;
	my $w			= shift;

	my $ai			= bless {
		canvas		=> $c,

		sky		=> $c->createArc(
			$x, $y, $x+$w, $y+$h,
			-start		=> 90,
			-extent		=> 180,
			-fill		=> 'blue',
			-outline	=> 'white',
			-style		=> 'chord',
		),

		ground		=>  $c->createArc(
			$x, $y, $x+$w, $y+$h,
			-start		=> 90,
			-extent		=> -180,
			-fill		=> 'brown',
			-outline	=> 'white',
			-style		=> 'chord',
		),

		level		=>  $c->createArc(
			$x, $y, $x+$w, $y+$h,
			-start		=> 90,
			-extent		=> 180,
			-outline	=> 'white',
			-style		=> 'chord',
		),

		# Attitude attributes
		roll		=> 0,
		pitch		=> 0,
	}, $class;

	$ai->refresh;

	return $ai;
}


sub roll	{ $_[0]->{roll} }
sub pitch	{ $_[0]->{pitch} }

sub set_pitch
{
	my $ai			= shift;
	my $new_angle		= shift;

	$ai->{pitch}		= EFIS::rel2abs( $ai->pitch, $new_angle );

	$ai->refresh;
}

sub set_roll
{
	my $ai			= shift;
	my $new_angle		= shift;

	$ai->{roll}		= $new_angle; #EFIS::rel2abs( $ai->roll, $new_angle );
	$ai->refresh;
}

sub refresh
{
	my $efis		= shift;

	$efis->{canvas}->itemconfigure(
		$efis->{sky},
		-start		=> $efis->roll - $efis->pitch,
		-extent		=> + ( 180 + 2 * $efis->pitch ),
	);

	$efis->{canvas}->itemconfigure(
		$efis->{level},
		-start		=> $efis->roll,
	);

	$efis->{canvas}->itemconfigure(
		$efis->{ground},
		-start		=> $efis->roll - $efis->pitch,
		-extent		=> - ( 180 - 2 * $efis->pitch ),
	);

}


package EFIS::HSI;
sub new
{
	my $class		= shift;
	my $c			= shift;
	my $x			= shift;
	my $y			= shift;
	my $w			= shift;
	my $h			= shift;

	bless {
		canvas		=> $c,

		ring		=> $c->createArc(
			$x, $y, $x+$w, $y+$h,
			-start		=> 90,
			-extent		=> 359,
			-fill		=> 'gray30',
			-outline	=> 'green',
			-style		=> 'chord',
		),

		# Attitude attributes
		heading		=> 0,
		direction	=> 0,
	}, $class;
}


package EFIS::Ticker;
sub new
{
	my $class		= shift;
	my $c			= shift;
	my $x			= shift;
	my $y			= shift;
	my $w			= shift;
	my $h			= shift;
	my $justify		= shift;
	my $value		= shift || 0;

	my $pad_x		= 10;

	# Fix up justify
	if( $justify eq 'left' ) {
		$justify = 0;
	} else {
		$justify = 1;
	}

	# Draw the large rectangle (ticker)
	$c->createRectangle(
		$x + $pad_x, $y,
		$x + $w - $pad_x, $y + $h,
		-fill		=> 'blue',
		-outline	=> 'white',
	);

	# Draw the small rectangle (current)
	$c->createRectangle(
		$x - ( $justify ? 0 : 20 ),
		$y + $h/2 - 15,
		$x + $w + ( $justify ? 20 : 0 ),
		$y + $h/2 + 15,
		-fill		=> 'gray30',
		-outline	=> 'white',
	);

	# Put the current value into the small box
	my $current = $c->createText(
		$x + $w/2 + ( $justify ? 10 : -10 ),
		$y + $h/2,
		-anchor		=> 'center',
		-fill		=> 'white',
		-justify	=> 'center',
		-text		=> $value,
		-font		=> '-*-fixed-*-*-*-*-24-*-*-*-*-*-*-*',
	);

	bless {
		canvas		=> $c,

		value		=> $value,
		current		=> $current,
	}, $class;
}

sub value { $_[0]->{value} }
sub set_value
{
	my $self		= shift;
	my $new_value		= shift;

	$self->{value} = EFIS::rel2abs( $self->value, $new_value );
	$self->refresh;
}

sub refresh
{
	my $self		= shift;

	$self->{canvas}->itemconfigure(
		$self->{current},
		-text		=> $self->value,
	);
}


package EFIS::AS;
use base 'EFIS::Ticker';
sub airspeed { $_[0]->value }
sub set_speed {
	my $as			= shift;
	$as->set_value( @_ );
}

package EFIS::ALT;
use base 'EFIS::Ticker';
sub altitude { $_[0]->value }
sub set_altitude {
	my $alt			= shift;
	$alt->set_value( @_ );
}


package EFIS;
sub new
{
	my $class		= shift;
	my $mw			= shift;

	my $width		= shift || 400;
	my $height		= shift || 640;

	my $ticker_width	= 40;
	my $ticker_height	= $height;

	my $gauge_width		= $width - 2 * $ticker_width;
	my $gauge_height	= $height / 2;


	my $c		= $mw->Canvas(
		-background	=> 'black',
		-width		=> $width,
		-height		=> $height,
	)->pack(
		-side		=> 'top',
		-fill		=> 'both',
		-expand		=> 0,
	);


	bless {
		canvas		=> $c,

		ai		=> EFIS::AI->new(
			$c,
			$ticker_width, 1,
			$gauge_width, $gauge_height,
		),

		hsi		=> EFIS::HSI->new(
			$c,
			$ticker_width, $gauge_height,
			$gauge_width, $gauge_height,
		),

		alt		=> EFIS::ALT->new(
			$c,
			1, 1,
			$ticker_width, $ticker_height,
			'left',
		),

		as		=> EFIS::AS->new(
			$c,
			$ticker_width + $gauge_width, 1,
			$ticker_width, $ticker_height,
			'right',
		),
	}, $class;
}


sub AI		{ $_[0]->{ai} }
sub HSI		{ $_[0]->{hsi} }
sub AS		{ $_[0]->{as} }
sub ALT		{ $_[0]->{alt} }

sub roll	{ $_[0]->AI->roll }
sub pitch	{ $_[0]->AI->pitch }
sub yaw		{ $_[0]->HSI->yaw }

sub set_pitch
{
	my $efis		= shift;
	$efis->AI->set_pitch( @_ );
}

sub set_roll
{
	my $efis		= shift;
	$efis->AI->set_roll( @_ );
}

sub set_yaw
{
	my $efis		= shift;
	$efis->HSI->set_yaw( @_ );
}

sub set_speed
{
	my $efis		= shift;
	$efis->AS->set_speed( @_ );
}

sub set_altitude
{
	my $efis		= shift;
	$efis->ALT->set_altitude( @_ );
}

sub rel2abs
{
	my $base		= shift;
	my $rel			= shift;

	if( $rel =~ /^[+-]/ )
	{
		return $base + $rel;
	} else {
		return $rel;
	}
}


"0, but true";
__END__
