#!/usr/bin/perl -w
my $version = q$Id: Needle.pm,v 1.13 2001/11/04 01:48:26 tramm Exp $;

=head1 NAME EFIS::Needle

The EFIS::Needle widget draws a two-needle gauge suitable for N1/N2
tachs, fuel gauges, or other instruments that require "analog" feedback.
It also draws labels for the two needles and a colored bar graph on
the face of the gauge.  It is similar to the Dial widget, but
resembles the instruments on aircraft rather than a car.

Usage:

use Tk;
use EFIS::Needle;

my $mw = MainWindow->new();
my $gauge = $mw->Needle(
	-width		=> $width,
	-label0		=> 'N1',
	-label1		=> 'N2',
)->pack;


=cut

package EFIS::Needle;
use strict;
use base 'Tk::Frame';
use Data::Dumper;


Tk::Widget->Construct('Needle');

sub Populate
{
	my $cw		= shift;
	my $args	= shift;

	my $width	= $cw->{width} = delete $args->{-width}		|| 128;

	my $value0	= $cw->{value0} = delete $args->{-value0}	|| 0;
	my $text0	= $cw->{text0} = delete $args->{-label0}	|| "";

	my $value1	= $cw->{value1} = delete $args->{-value1}	|| 0;
	my $text1	= $cw->{text1} = delete $args->{-label1}	|| "";
	my $ranges	= $cw->{ranges} = delete $args->{-ranges}	|| [
		gold		=>  20,
		darkgreen	=>  80,
		gold		=>  90,
		darkred		=> 100,
	];

	$cw->configure(
		-border		=> '3',
		-relief		=> 'sunken',
	);

	my $canvas = $cw->{canvas} = $cw->Canvas(
		-width		=> $width,
		-height		=> $width + 20,
	)->pack(
		-side		=> 'left',
		-expand		=> 0,
		-fill		=> 'both',
	);

	# Background is bunk

	# Draw the outer ring
	$canvas->createArc(
		0, 0,
		$width, $width,
		-fill		=> 'black',
		-outline	=> 'white',
		-start		=> 0,
		-extent		=> 359,
		-style		=> 'chord',
	);

	# Draw the two warning areas
	$cw->{area0} = $canvas->createArc(
		0, 0,
		$width, $width,
		-fill		=> 'darkblue',
		-outline	=> 'white',
		-start		=> 135,
		-extent		=> 90,
		-style		=> 'chord',
	);

	$cw->{area1} = $canvas->createArc(
		0, 0,
		$width, $width,
		-fill		=> 'darkblue',
		-outline	=> 'white',
		-start		=> -45,
		-extent		=> 90,
		-style		=> 'chord',
	);

	# Draw the two labels at the bottom
	$cw->{label0} = $canvas->createText(
		10,
		$width + 2,
		-anchor		=> 'nw',
		-fill		=> 'black',
		-text		=> "$text0: $value0",
	);

	$cw->{label1} = $canvas->createText(
		$width - 10,
		$width + 2,
		-anchor		=> 'ne',
		-fill		=> 'black',
		-text		=> "$value1 :$text1",
	);



	# Draw the ranges
	my $start	= $width;
	my $x1		= $width / 2 - 6;
	my $x2		= $width / 2 + 6;
	

	my @ranges	= @$ranges;
	while( @ranges )
	{
		my $color	= shift @ranges;
		my $val		= shift @ranges;
		my $end		= $width - $val * $width / 100;

		$canvas->createRectangle(
			$x1, $end,
			$x2, $start,
			-fill		=> $color,
		);

		$start = $end;
	}
		

	# Draw the labels and needles
	$cw->redraw();

	return $cw;
}


sub redraw
{
	my $gauge	= shift;
	my $canvas	= $gauge->{canvas};

	$canvas->delete( $gauge->{needle0} )
		if $gauge->{needle0};
	$canvas->delete( $gauge->{needle1} )
		if $gauge->{needle1};

	my $width	= $gauge->{width};
	my $half	= $width / 2;
	my $angle0	= 3.14 * (50 - $gauge->{value0}) / 180;
	my $angle1	= 3.14 * (50 - $gauge->{value1}) / 180;

	$gauge->{needle0} = $canvas->createLine(
		    0,
		$half,
		    0 + 1.3 * $half * cos( $angle0 ),
		$half + 1.3 * $half * sin( $angle0 ),
		-fill		=> 'white',
		-width		=> 2,
	);

	$gauge->{needle1} = $canvas->createLine(
		$width,
		$half,
		$width - 1.3 * $half * cos( $angle1 ),
		$half  + 1.3 * $half * sin( $angle1 ),
		-fill		=> 'white',
		-width		=> 2,
	);

	$canvas->itemconfigure( $gauge->{label0},
		-text		=> "$gauge->{text0}: $gauge->{value0}",
	);

	$canvas->itemconfigure( $gauge->{label1},
		-text		=> "$gauge->{value1} :$gauge->{text1}",
	);

	# Find the range that matters
	my $start	= 0;
	my @ranges	= @{ $gauge->{ranges} };

	while( @ranges )
	{
		my $color	= shift @ranges;
		my $val		= shift @ranges;

		if( $start < $gauge->{value0} && $gauge->{value0} < $val )
		{
			$canvas->itemconfigure(
				$gauge->{"area0" },
				-fill		=> $color
			);
		}

		if( $start < $gauge->{value1} && $gauge->{value1} < $val )
		{
			$canvas->itemconfigure(
				$gauge->{"area1" },
				-fill		=> $color
			);
		}

		$start = $val;
	}
}

sub warning
{
	my $gauge	= shift;
	my $which	= shift;
	my $color	= shift;

	$gauge->{canvas}->itemconfigure(
		$gauge->{"area$which"},
		-fill		=> $color,
	);
}


sub set
{
	my $gauge	= shift;
	my $which	= shift;
	my $value	= shift;

	$value =   0 if $value <   0;
	$value = 100 if $value > 100;

	$gauge->{"value$which"} = int $value;
	$gauge->redraw;

	
}
"0, but true";
__END__
