# 
# This file is licensed under the GPL.  Please see the file "LICENSE" in
# the root directory of this project for terms and conditions.
# (c) 2001 by Trammell Hudson <hudson@swcp.com>
#
package Joystick;
use strict;

=head NAME - Joystick

Interfaces to the Linux joystick API.  Very simple filedescriptor
manipulation.

I would use joystick.ph, but there seems to be a bug with h2ph in this
distribution of Linux.  I'm not sure would is wrong.  Anyway, don't be
scared of unpack!


$Id: Joystick.pm,v 1.3 2001/10/05 16:21:18 tramm Exp $

=cut

# If this worked, it would be great!
#require "sys/ioctl.ph";
#require "linux/joystick.ph";

use constant JSIOCGAXES		=> 0x80016a11;
use constant JSIOCGBUTTONS	=> 0x80016a12;

=head METHOD - Joystick::new

Opens the default joystick device and returns an joystick
object.  Returns undef if the open fails.

=cut

sub new
{
	my $class		= shift;
	my $device		= shift || "/dev/js0";

	use FileHandle;
	my $fh			= new FileHandle "<$device"
		or return;

	my $js			= bless {
		dev		=> $fh,
	}, $class;

	return $js;
}

sub ioctl
{
	my $js			= shift;
	my $function		= shift;
	my $char_ptr		= 'PADDING';

	ioctl $js->{dev}, $function, $char_ptr
		or warn "Ioctl failed: $!\n"
		and return;

	return ord( substr( $char_ptr, 0, 1 ) );
}


=head METHOD - axes

Returns the number of axes supported by the joystick.  If the
internal ioctl fails, returns undef.

=cut

sub axes
{
	my $js			= shift;
	return $js->ioctl( JSIOCGAXES );
}


=head METHOD - buttons

Returns the number of buttons supported by the joystick.  If the
internal ioctl fails, returns undef.

=cut

sub buttons
{
	my $js			= shift;
	return $js->ioctl( JSIOCGBUTTONS );
}


=head METHOD - watch_axis

Calls the function when ever the numbered axis changes.

=cut

sub watch_axis
{
	my $js			= shift;
	my $axis		= shift;
	my $handler		= shift;

	push @{ $js->{axis}->[$axis] }, $handler;
}

sub watch_button
{
	my $js			= shift;
	my $button		= shift;
	my $handler		= shift;

	push @{ $js->{button}->[$button] }, $handler;
}


=head METHOD - update

Check for file activity, call any handlers, etc.

=cut

sub is_axis	{ $_[0] & 0x02 }
sub is_button	{ $_[0] & 0x01 }
sub is_event	{ $_[0] & 0x80 }

sub update
{
	my $js			= shift;
	my $buf			= '';
	my $sizeof_js_event	= 8;

	sysread $js->{dev}, $buf, $sizeof_js_event, 0
		or warn "Joystick: read failed: $!\n"
		and return;

# We're unpacking one of these:
#	struct js_event {
#		__u32 time;	/* event timestamp in miliseconds */
#		__s16 value;	/* value */
#		__u8 type;	/* event type */
#		__u8 number;	/* axis/button number */
#	};

	my ($time,$value,$type,$num) = unpack( 'LsCC', $buf );

=for nobody
	printf "%08x: type=%s num=%d value=%d\n",
		$time,
		is_axis( $type ) ? 'axis' :
		is_button( $type) ? 'button' :
		'unknown',
		$num,
		$value,
	;
=cut

	if( is_axis( $type ) )
	{
		$js->{axis_value}->[$num] = $value;
		$_->( $js, $value ) for @{ $js->{axis}->[$num] };
	}

	if( is_button( $type ) )
	{
		$js->{button_value}->[$num] = $value;
		$_->( $js, $value ) for @{ $js->{button}->[$num] };
	}

	return 1;
}


=head METHOD - axis

my $last_axis_value = $js->axis( $axis_number );

=cut

sub axis
{
	my $js			= shift;
	my $axis		= shift;

	return $js->{axis_value}->[$axis];
}


=head METHOD - button

my $last_button_value = $js->button( $button_number );

=cut

sub button
{
	my $js			= shift;
	my $button		= shift;

	return $js->{button_value}->[$button];
}


"0, but true";
__END__
