/**
 *  $Id: pwmeter.c,v 1.29 2001/11/04 05:53:00 tramm Exp $
 *
 * This is a simple attempt to sample the incoming gyros and
 * safety pilot receiver.  It also drives outputs to servos
 * and now reads the simple commands from the serial port.
 *
 * We are now using the uart library to read and write using
 * interrupt driven IO.  Since we are only doing IO during the
 * quiescent period, the interrupt overhead should be minimal.
 *
 */
#include <io.h>
#include <sig-avr.h>
#include <interrupt.h>
#include "avr.h"

/**
 *  We use interrupts to count our few pulse counters.
 * Right now that is N1 and N2.
 */
static uint16_t n1, n2;

SIGNAL( SIG_INTERRUPT7 )
{
	n1++;
}

SIGNAL( SIG_INTERRUPT6 )
{
	n2++;
}

static inline void
pulse_init( void )
{
	/* Set PORTE as input for edge triggered interrupts */
	outp( 0x00, DDRE );
	outp( 0xFF, PORTE );
	
	sbi( EICR, ISC71 );
	cbi( EICR, ISC70 );
	sbi( EIMSK, INT7 );
}


/**
 *  CHT and EGT are thermisters on the cylinder head and exhaust
 * manifolds.  We measure the resistance by the voltage drop.
 * 
 */
static uint8_t		cht;
static uint8_t		egt;

static inline void
process_sensors( void )
{
	cht = volts[0] - 0x150;
	egt = volts[2] >> 2;
}


/**
 *  Autopilot code goes here...
 *
 * Only some servos are given to the autopilot...  A true value
 * here means that the autopilot will not send its commands to
 * the servos for that servo.
 */
static const uint16_t	CLOSE_ENOUGH	= 32;

static uint8_t		autopilot[ 8 ] =
{
	1,		/* 0: Pitch */
	1,		/* 1: Roll */
	0,		/* 2: Throttle or collective */
	0,		/* 3: Anti-torque */
	0,		/* 4: Throttle */
	0,		/* 5: Unused */
	0,		/* 6: Unused */
	0,		/* 7: Unused */
};

static void inline
fix_axis(
	int16_t			pos,
	uint8_t			servo
)
{
	/* Are we controlling this servo? */
	if( !autopilot[servo] )
		return;

	/* Level? */
	if( -CLOSE_ENOUGH < pos && pos < CLOSE_ENOUGH )
		return;

	/* Righting? */
	/* Steady? */
	/* Tipping? */

	if( servo == 0 )
		led_set_mask( 128 + pos / 16 );
	servo_set( servo, 128 + pos / 16 );
}


static void
process_attitude( void )
{
	int16_t roll, pitch, yaw;

	attitude( &roll, &pitch, &yaw );

	/* Check for "level" */
	fix_axis( roll, 1 );
	fix_axis( pitch, 0 );
}

/**
 * Command formats:
 *
 * 0xFF servo position		Quickly move servo to that position
 * 0xFE servo position		Slowly move servo to that position
 * 0xFD servo auto/manual	Give/Take servo to the autopilot
 * 0xFC xxx xxx			Reset to level
 *
 */
static inline void
process_uart( void )
{
	static uint8_t		command		= 0x00;
	static uint8_t		servo		= 0xFF;
	uint8_t			c;

	while( getchar( &c ) >= 0 )
	{
/*
		puts( "read=" );
		puthexs( c );
		puts( "\r\n" );
*/

		if( !command )
		{
			/* Check for a command word */
			if( (c & 0xF0) != 0xF0 )
				puts( "?\r\n" );
			else
				command = c;
			continue;
		}

		if( servo == 0xFF )
		{
			servo = c;
			continue;
		}


		/* We have three words */
		switch( command )
		{
			case 0xFF:
				/* Fast servo motion */
				if( !autopilot[servo] )
					servo_set( servo, c );
				break;
			case 0xFE:
				/* Slow servo motion */
				if( !autopilot[servo] )
					servo_set( servo, c );	
				break;
			case 0xFD:
				/* Autopilot take over or give up */
				autopilot[servo] = c;
				break;
			case 0xFC:
				/* Reset to level */
				reset_level();
				break;
			default:
				/* Huh? */
				puts( "?\r\n" );
				break;
		}

		command	= 0x00;
		servo	= 0xFF;
	}
}



static void
do_output( void )
{
	int16_t roll, pitch, yaw;
	attitude( &roll, &pitch, &yaw );

	putchar( 'R' );	puthex( roll );  putchar( ' ' );
	putchar( 'P' );	puthex( pitch ); putchar( ' ' );
	putchar( 'H' );	puthex( yaw );   putchar( ' ' );
#if 0
	puts( "\r\n" );

	/* Output the RPM of the N1 counter */
	putchar( 'N' ); puthex( n1 ); putchar( ' ' );
	putchar( 'T' ); puthex( n2 ); putchar( ' ' );
	puts( "\r\n" );
	n1 = n2 = 0;
#endif

	putchar( 'C' ); puthexs( cht ); putchar( ' ' );
	putchar( 'E' ); puthexs( egt ); putchar( ' ' );

	putchar( 'S' ); puthexs( out_servos[0] ); putchar( ' ' );

	puts( "\r\n" );
}




/**
 *  Setup everything and go into the mainloop...
 */
int main( void )
{
	uint16_t	last_bits	= 0;
	uint8_t		cycle_count	= 0;

	/**
	 *  Configure the ports and timers
	 */
	avr_init();
	pulse_init();
	adc_init( 0x0F );	/* CHT, Vg1, EGT, Vg2 */

	/* Turn on interrupts */
	sbi( SREG, 7 );

	puts( "$Id: pwmeter.c,v 1.29 2001/11/04 05:53:00 tramm Exp $\r\n" );


	/**
	 *  Now go into the main loop
	 */
	while(1)
	{
		/* Check the A->D every time through. */
		if( ready_adc() )
			process_adc();

		/* If it hasn't been more than 1/60 second */
		if( high_bits == last_bits )
			continue;

		/* Deal with all our inputs */
		process_sensors();
		rate_integration();
		process_attitude();
		process_uart();

		/* Backup for the level reset */
		if( (inp( PIND ) & 1) == 0 )
			reset_level();

		/* Every 32 * 1/60 ~= 1/2 second we do a status output */
		if( ++cycle_count % 32 == 0 )
			do_output();

		/* Don't run until another 60 Hz cycle */
		last_bits = high_bits;
	}

	/* Not reached */
	return 0;
}
