/**
 * $Id: servo.c,v 1.37 2001/11/04 05:15:50 tramm Exp $
 *
 * Servo and gyro management code.  The servos are to be hooked
 * to port C.
 */
#include <io.h>
#include <sig-avr.h>
#include <interrupt.h>
#include "servo.h"
#include "timer.h"

#ifdef __timer
#undef __timer
#endif
#define __timer( f )	f ## 0

#define SERVO_MASK	(MAX_SERVOS-1)
#define  IN_PORT	PINA	/* Input from the gyros or receiver */
#define  IN_PORT_DIR	DDRA	/* Input from the gyros or receiver */
#define  IN_PORT_CFG	PORTA	/* Configuration for the gyro port */
#define OUT_PORT	PORTC	/* Output to the servos */
#define OUT_PORT_DIR	DDRC	/* Output to the servos */


/**
 *  We have to define a weighted average for sampling the incoming
 * servo data.  If we do not average the data, the actual values
 * are too "noisy" to be useful.
 *
 * This is equivilant to:
 *             (W-1) * x'  +  x
 *      x' =  -------------------
 *                      W
 *
 * Obviously, WEIGHT should be a power of two.  The actual code is
 * optimized to avoid multipies; it does a shift followed by a subtract.
 */

/**
 *  Output servos values are precomputed into usecond pulse widths to
 * save time in the servo_out() task.  Input servos are scaled to be
 * within the -128 .. 127 range.
 */

uint16_t		 in_servos[ MAX_SERVOS ];
uint8_t			out_servos[ MAX_SERVOS ];


INTERRUPT( __timer( SIG_OUTPUT_COMPARE ) )
{
	static uint8_t		current_servo		= 0;
	static uint8_t		current_servo_bit	= 1;

	/* Clear the output to avoid glitches */
	outp( 0, OUT_PORT );

	/* Rotate to the next output bit */
	current_servo_bit <<= 1;
	if( current_servo_bit == 0 )
		current_servo_bit = 1;

	/* Find the next servo */
	current_servo = ( current_servo + 1 ) & SERVO_MASK;

	outp( out_servos[current_servo], __timer( OCR ) );
	outp( 0x00, __timer( TCNT ) );
	outp( current_servo_bit, OUT_PORT );
}


/**
 *  Initialize our ports to be going in the right direction.
 * Otherwise we couldn't read from them...
 */
void
servo_init( void )
{
	int servo;
	outp( 0x00, IN_PORT_DIR );
	outp( 0x00, IN_PORT_CFG );

	/* Port C is always output */
	/* outp( 0xFF, OUT_PORT_DIR ); */

	/* Zero any outputs that we might have */
	outp( 0x00, OUT_PORT );

	/* Position all the servos in a reasonable location */
	for( servo = 0 ; servo < MAX_SERVOS ; servo++ )
	{
		servo_set( servo, 0 );
		out_servos[servo] = pos2ticks( 128 );
	}

	/*
	 * Setup timer0 (an 8-bit one) to serve as our PWM
	 * counter at Clock / 32.  It'll call servo_out when it hits the
	 * right value.
	 *
	 * This lets us have up to 2048 ms of servo pulse width
	 * with the 4 Mhz clock.
	 */
	sbi( TIMSK, __timer( OCIE ) );
	outp( 0x03, __timer( TCCR ) );
}


/**
 *  Sample the servo input lines and compute the position of
 * each servo input.  Since the output servos are now interrupt
 * driven, this is the only periodic servo task remaining.
 *
 * The other option is to only poll one servo pin at a time.
 * The code in pwmeter.c version 1.15 does that quite well,
 * we'll add it back in here once the need arrises again.
 */
void
servo_task( void )
{
	static uint16_t		start[ MAX_SERVOS ] = {};
	static uint8_t		last;
	uint8_t			bits;
	uint8_t			tmp_bits;
	uint8_t			servo;
	uint16_t		now;

	/* Read all the servo bits */
	bits = inp( IN_PORT );

	/* Check for any change.  No change and we return */
	if( bits == last )
		return;

	now		= time();
	tmp_bits	= bits;
	servo		= 0;

	while( bits != last )
	{
		/*
		 * Rising edge; mark the start time and keep looking.
		 */
		if( !(last & 1) &&  (bits & 1) ) {
			start[servo] = now;
		} else

		/*
		 * Falling edge; compute the time and store the position.
		 * Rate integration happens in another task.
		 */
		if(  (last & 1) && !(bits & 1) ) {
			/* Underflow doesn't matter */
			uint16_t	len	= now - start[servo];
			uint16_t	avg	= in_servos[servo];
			uint8_t		weight	= 8;

			/* Average it */
			in_servos[servo] =
				(weight * avg - avg + len) / weight;
		}

		bits >>= 1;
		last >>= 1;
		servo++;
	}

	last = tmp_bits;
}

