#ifndef _SERVO_H_
#define _SERVO_H_

/**
 *  $Id: servo.h,v 1.12 2001/11/04 04:19:07 tramm Exp $
 *
 * Servo/gyro management routines.
 */
#include <inttypes.h>
#include "timer.h"
#include "uart.h"


#define SERVO_FUTABA

#ifdef SERVO_FUTABA
# define LOW		usec2pulse( 1000)
# define HIGH		usec2pulse( 2000)
# define FRAME		usec2pulse(11000)
#else
# error "No servo type defined"
#endif

#define MAX_SERVOS		8
#define SERVO_MASK		(MAX_SERVOS-1)

extern uint16_t		 in_servos[ MAX_SERVOS ];
extern uint8_t		out_servos[ MAX_SERVOS ];


/**
 *  We precompute how many ticks the PWM timer should wait
 * before bringing the servo's output line low.  The PWM
 * timer runs at Clk / 32, so each tick is 8 usec with a
 * 4 Mhz clock.  We could compute this with the constants
 * in "timer.h", but it would be difficult to do and not
 * flow into 32-bit math.  So we use the constant 8.
 */
#define usec2pulse(u)		((u) / 8)

static inline uint8_t
pos2ticks( uint8_t pos )
{
	return ((uint16_t) pos) * (HIGH - LOW) / 256l + LOW;
}


/**
 *  The servo input timer uses the system clock, which runs at Clk / 8,
 * so each tick is 2 usec.  Here we use "timer.h", because it is defined
 * as a nice 8-bit value.
 */
static inline int
ticks2pos( uint16_t ticks )
{
	return (ticks - LOW) * 256 / (HIGH-LOW) - 128;
}


/**
 *  servo_set positions the specified servo at the position -128 .. 127
 * required.  If the servo is out of range, nothing happens.  If the
 * position is out of range it will be truncated to fit in the servo's
 * range.
 */
static inline void
servo_set(
	uint8_t		servo,
	uint8_t		pos
)
{
	out_servos[ servo ] = pos2ticks( pos );
/*
	puts( "servo " );
	puthexs( servo );
	putchar( '=' );
	puthexs( pos );
	putchar( ' ' );
	puthexs( out_servos[servo] );
	puts( "\r\n" );
*/
}

/**
 *  servo_set_ticks sets the specified servo pulse to be ticks system
 * clock ticks wide.  It scales it to servo clock ticks, which are 4 times
 * longer than the system clock.
 */
static inline void
servo_set_ticks(
	uint8_t		servo,
	uint16_t	ticks
)
{
	out_servos[ servo ] = ticks / 4;
}


/**
 *  servo_get retrieves the instantaneous value of a servo input.  It
 * is -not- the attitude or anything else very useful, except to the
 * rate integration code or perhaps a servo mixing program.
 */
static inline int8_t
servo_get(
	uint8_t		servo
)
{
	return ticks2pos( in_servos[ servo ] );
}


/* Tasks */
extern void servo_task( void );
extern void servo_init( void );

#endif
