/**
 *  $Id: uart.c,v 1.8 2001/10/25 00:44:49 tramm Exp $
 *
 * Serial port access routines
 */
#include <io.h>
#include <sig-avr.h>
#include <interrupt.h>
#include "uart.h"

#define RX_BUF_SIZE		256
#define RX_BUF_MASK		(RX_BUF_SIZE-1)

#define TX_BUF_SIZE		256
#define TX_BUF_MASK		(TX_BUF_SIZE-1)

#if ( RX_BUF_SIZE & RX_MASK )
#	error "UART Receiver buffer is not a power of 2"
#endif
#if ( TX_BUF_SIZE & TX_MASK )
#	error "UART Transmit buffer is not a power of 2"
#endif

static uint8_t rx_buf[ RX_BUF_SIZE ];
volatile uint8_t rx_head;
volatile uint8_t rx_tail;

static uint8_t tx_buf[ TX_BUF_SIZE ];
static volatile uint8_t tx_head;
static volatile uint8_t tx_tail;

/**
 *  Setup the serial port for 19.2, no handshaking.
 * Also initialize our queues.
 */
void
uart_init( void )
{
	/* Baudrate = 19.2 for 4 Mhz clock */
	outp( 12, UBRR );

	/* Enable UART Rx, Tx, plus interrupts for incoming data */
	sbi( UCR, RXEN );
	sbi( UCR, TXEN );
	sbi( UCR, RXCIE );

	enable_external_int( INT0 );
	sei();

	rx_head = rx_tail = 0;
	tx_head = tx_tail = 0;
}


/**
 *  Get a character from the incoming buffer.  Returns -1 if the
 * buffer is empty.
 */
int8_t getchar( uint8_t *data )
{
	uint8_t tmp_tail;

	if( rx_head == rx_tail )
		return -1;
	tmp_tail = ( rx_tail + 1 ) & RX_BUF_MASK;
	rx_tail = tmp_tail;
	*data = rx_buf[ tmp_tail ];
	return 0;
}

/**
 *  Put a character onto the outgoing buffer.  Returns -1 if the
 * buffer is full.
 */
int8_t putchar( uint8_t c )
{
	uint8_t tmp_head;

	tmp_head = ( tx_head + 1 ) & TX_BUF_MASK;

	/* Check for overflow of the buffer */
	if( tmp_head == tx_tail )
		return -1;

	tx_buf[ tmp_head ] = c;
	tx_head = tmp_head;

	/* Signal that we have data */
	sbi( UCR, UDRIE );

	return 0;
}
		
SIGNAL( SIG_UART_RECV )
{
	uint8_t data = inp( UDR );
	uint8_t tmp_head = (rx_head + 1) & RX_BUF_MASK;
	rx_head = tmp_head;
	
	if( tmp_head == rx_tail ) {
		/* Out of buffer space! */
	}

	rx_buf[ tmp_head ] = data;
}


/**
 *  This interrupt is called whenever the UART data register is empty.
 * If we have no data remaining we disable the interrupt.  Otherwise
 * push another byte into the buffer and wait.
 */

SIGNAL( SIG_UART_DATA )
{
	uint8_t tmp_tail;

	/* Disable this interrupt if we've flushed the data */
	if( tx_head == tx_tail ) {
		cbi( UCR, UDRIE );
	} else {
		/* Send a byte of the buffer */
		tmp_tail = ( tx_tail + 1 ) & TX_BUF_MASK;
		tx_tail = tmp_tail;
		outp( tx_buf[tmp_tail], UDR );
	}
}


/**
 *  Random output functions
 */
static const char hex_digits[] = "0123456789abcdef";

	
void
puthexs( uint8_t i )
{
	putchar( hex_digits[ (i >>  4) & 0xF ] );
	putchar( hex_digits[ (i >>  0) & 0xF ] );
}

void
puthex( uint16_t i )
{
	putchar( hex_digits[ (i >> 12) & 0xF ] );
	putchar( hex_digits[ (i >>  8) & 0xF ] );
	putchar( hex_digits[ (i >>  4) & 0xF ] );
	putchar( hex_digits[ (i >>  0) & 0xF ] );
}

void
puthexl( uint32_t i )
{
	putchar( hex_digits[ (i >> 28) & 0xF ] );
	putchar( hex_digits[ (i >> 24) & 0xF ] );
	putchar( hex_digits[ (i >> 20) & 0xF ] );
	putchar( hex_digits[ (i >> 16) & 0xF ] );
	putchar( hex_digits[ (i >> 12) & 0xF ] );
	putchar( hex_digits[ (i >>  8) & 0xF ] );
	putchar( hex_digits[ (i >>  4) & 0xF ] );
	putchar( hex_digits[ (i >>  0) & 0xF ] );
}

void
puts( const char *s )
{
	while( *s )
		putchar( *s++ );
}

void
putnl( void )
{
	putchar( '\n' );
	putchar( '\r' );
}
