/*
 *  $Id: SQLGetFunctions.c,v 1.2 1999/12/01 10:26:35 martin Exp martin $ 
 *
 *  $Log: SQLGetFunctions.c,v $
 *  Revision 1.2  1999/12/01 10:26:35  martin
 *  Run through indent.
 *
 *  Revision 1.1  1999/11/24 16:34:19  steve
 *  Initial revision
 * 
 *
 */

/* 
 *
 * SQLGetFunctions returns information about whether a driver supports a
 * specific ODBC function. This function is implemented in the Driver Manager;
 * it can also be implemented in drivers. If a driver implements
 * SQLGetFunctions, the Driver manager calls the function in the driver.
 * Otherwise, it executes the function itself.
 *
 * The program looks on the command line for the Connection String
 *
 * SQLGetFunctions InConnectionStr 
 *
 * e.g. 
 *
 * SQLGetFunctions DSN=easysoft
 *
 * It performs three passes on the ODBC Driver as an:
 *
 *  1) ODBC 2.x Application
 *  2) ODBC 3.x Application with ODBC 2.x behaviour
 *  3) ODBC 3.x Application with ODBC 3.x behaviour
 *
 *
 *
 *
 *
 *
 */

#include <stdio.h>
/*
 *
 * Exclude rarely-used stuff from Windows headers
 *
 */
#ifdef WIN32
# define WIN32_LEAN_AND_MEAN		
#include <windows.h>
#else
# include <unistd.h>
#endif

#include <sql.h>
#include <sqlext.h>

#define NUMBER_BEHAVIOURS                       3
#define ODBC2					0
#define ODBC3_ODBC2				1
#define ODBC3_ODBC3				2

struct driver_block
{
    int                 behaviour;
    SQLHENV             henv;
    SQLHDBC             hdbc;
    SQLHSTMT		hstmt;
    SQLRETURN		retcode;
    char                outstr[1024];
    SQLSMALLINT         outstr_len;
    SQLHANDLE		handle;
    SQLSMALLINT		type;	
    SQLRETURN		funcs_retcode;
    SQLUSMALLINT	funcs2[100];
    SQLSMALLINT		funcs3[SQL_API_ODBC3_ALL_FUNCTIONS_SIZE];
};

#define CORE	1
#define LEVEL_1 2
#define LEVEL_2 3

#define ISO_92	1
#define DEPRECATED 2
#define ODBC 3
#define X_OPEN 4

#define ODBC_1	1
#define ODBC_2	2
#define ODBC_3	3

struct func 
{
    char                *name;
    SQLUSMALLINT	func_id;
    int                 odbc2_conform;
    int                 odbc3_conform;
    int                 standard;
    int                 version;
} func_table[] = 	
{
    { "SQLAllocConnect", SQL_API_SQLALLOCCONNECT, CORE, 0, DEPRECATED, ODBC_1 },
    { "SQLAllocEnv", SQL_API_SQLALLOCENV, CORE, 0, DEPRECATED, ODBC_1 }, 
    { "SQLAllocHandle", SQL_API_SQLALLOCHANDLE, 0, CORE, ISO_92, ODBC_3 },
    { "SQLAllocHandleStd", SQL_API_SQLALLOCHANDLESTD, 0, CORE, ISO_92, ODBC_3 },   
    { "SQLAllocStmt", SQL_API_SQLALLOCSTMT, CORE, 0, DEPRECATED, ODBC_1 },
    { "SQLBindCol", SQL_API_SQLBINDCOL, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLBindParam", SQL_API_SQLBINDPARAM, 0, CORE, DEPRECATED, ODBC_3 },
    { "SQLBindParameter", SQL_API_SQLBINDPARAMETER, LEVEL_1, CORE, ODBC, ODBC_2 },
    { "SQLBrowseConnect", SQL_API_SQLBROWSECONNECT, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLBulkOperations", SQL_API_SQLBULKOPERATIONS, 0, LEVEL_1, ODBC, ODBC_3 },
    { "SQLCancel", SQL_API_SQLCANCEL, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLCloseCursor", SQL_API_SQLCLOSECURSOR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLColAttribute", SQL_API_SQLCOLATTRIBUTE, 0, CORE, ISO_92, ODBC_3 },
    { "SQLColAttributes", SQL_API_SQLCOLATTRIBUTES, CORE, 0, DEPRECATED, ODBC_1 }, 
    { "SQLColumnPrivileges", SQL_API_SQLCOLUMNPRIVILEGES, LEVEL_2, LEVEL_2, ODBC, ODBC_1 },
    { "SQLColumns", SQL_API_SQLCOLUMNS, LEVEL_1, CORE, X_OPEN, ODBC_1 },
    { "SQLConnect", SQL_API_SQLCONNECT, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLCopyDesc", SQL_API_SQLCOPYDESC, 0, CORE, ISO_92, ODBC_3 },
    { "SQLDataSources", SQL_API_SQLDATASOURCES, LEVEL_2, CORE, ISO_92, ODBC_1 },
    { "SQLDescribeCol", SQL_API_SQLDESCRIBECOL, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLDescribeParam", SQL_API_SQLDESCRIBEPARAM, LEVEL_2, LEVEL_2, ODBC, ODBC_1 },
    { "SQLDisconnect", SQL_API_SQLDISCONNECT, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLDriverConnect", SQL_API_SQLDRIVERCONNECT, LEVEL_1, CORE, ODBC, ODBC_1 },
    { "SQLDrivers", SQL_API_SQLDRIVERS, LEVEL_2, CORE, ODBC, ODBC_2 },
    { "SQLEndTran", SQL_API_SQLENDTRAN, 0, CORE, ISO_92, ODBC_3 },
    { "SQLError", SQL_API_SQLERROR, CORE, 0, DEPRECATED, ODBC_1 },
    { "SQLExecDirect", SQL_API_SQLEXECDIRECT, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLExecute", SQL_API_SQLEXECUTE, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLExtendedFetch", SQL_API_SQLEXTENDEDFETCH, LEVEL_2, 0, DEPRECATED, ODBC_1 },
    { "SQLFetch", SQL_API_SQLFETCH, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLFetchScroll", SQL_API_SQLFETCHSCROLL, 0, CORE, ISO_92, ODBC_3 },
    { "SQLForeignKeys", SQL_API_SQLFOREIGNKEYS, LEVEL_2, LEVEL_2, ODBC, ODBC_1 },
    { "SQLFreeConnect", SQL_API_SQLFREECONNECT, CORE, 0, DEPRECATED, ODBC_1 },
    { "SQLFreeEnv", SQL_API_SQLFREEENV, CORE, 0, DEPRECATED, ODBC_1 },
    { "SQLFreeHandle", SQL_API_SQLFREEHANDLE, 0, CORE, ISO_92, ODBC_3 },
    { "SQLFreeStmt", SQL_API_SQLFREESTMT, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLGetConnectAttr", SQL_API_SQLGETCONNECTATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetConnectOption", SQL_API_SQLGETCONNECTOPTION, LEVEL_1, 0, DEPRECATED, ODBC_1 },
    { "SQLGetCursorName", SQL_API_SQLGETCURSORNAME, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLGetData", SQL_API_SQLGETDATA, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLGetDescField", SQL_API_SQLGETDESCFIELD, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetDescRec", SQL_API_SQLGETDESCREC, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetDiagField", SQL_API_SQLGETDIAGFIELD, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetDiagRec", SQL_API_SQLGETDIAGREC, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetEnvAttr", SQL_API_SQLGETENVATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetFunctions", SQL_API_SQLGETFUNCTIONS, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLGetInfo", SQL_API_SQLGETINFO, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLGetStmtAttr", SQL_API_SQLGETSTMTATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLGetStmtOption", SQL_API_SQLGETSTMTOPTION, LEVEL_1, 0, DEPRECATED, ODBC_1 },
    { "SQLGetTypeInfo", SQL_API_SQLGETTYPEINFO, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLMoreResults", SQL_API_SQLMORERESULTS, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLNativeSQL", SQL_API_SQLNATIVESQL, LEVEL_2, CORE, ODBC, ODBC_1 },
    { "SQLNumParams", SQL_API_SQLNUMPARAMS, LEVEL_2, CORE, ISO_92, ODBC_1 },
    { "SQLNumResultCols", SQL_API_SQLNUMRESULTCOLS, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLParamData", SQL_API_SQLPARAMDATA, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLParamOptions", SQL_API_SQLPARAMOPTIONS, LEVEL_2, 0, DEPRECATED, ODBC_1 },
    { "SQLPrepare", SQL_API_SQLPREPARE, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLPrimaryKeys", SQL_API_SQLPRIMARYKEYS, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLProcedureColumns", SQL_API_SQLPROCEDURECOLUMNS, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLProcedures", SQL_API_SQLPROCEDURES, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLPutData", SQL_API_SQLPUTDATA, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLRowCount", SQL_API_SQLROWCOUNT, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLSetConnectAttr", SQL_API_SQLSETCONNECTATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLSetConnectOption", SQL_API_SQLSETCONNECTOPTION, LEVEL_1, 0, DEPRECATED, ODBC_1 },
    { "SQLSetCursorName", SQL_API_SQLSETCURSORNAME, CORE, CORE, ISO_92, ODBC_1 },
    { "SQLSetDescField", SQL_API_SQLSETDESCFIELD, 0, CORE, ISO_92, ODBC_3 },
    { "SQLSetDescRec", SQL_API_SQLSETDESCREC, 0, CORE, ISO_92, ODBC_3 },
    { "SQLSetEnvAttr", SQL_API_SQLSETENVATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLSetParam", SQL_API_SQLSETPARAM, LEVEL_1, 0, DEPRECATED, ODBC_1 },
    { "SQLSetPos", SQL_API_SQLSETPOS, LEVEL_2, LEVEL_1, ODBC, ODBC_1 },
    { "SQLSetScrollOptions", SQL_API_SQLSETSCROLLOPTIONS,  LEVEL_2, 0, DEPRECATED, ODBC_1 },
    { "SQLSetStmtAttr", SQL_API_SQLSETSTMTATTR, 0, CORE, ISO_92, ODBC_3 },
    { "SQLSetStmtOption", SQL_API_SQLSETSTMTOPTION, LEVEL_1, 0, DEPRECATED, ODBC_1 },
    { "SQLSpecialColumns", SQL_API_SQLSPECIALCOLUMNS, LEVEL_1, CORE, X_OPEN, ODBC_1 },
    { "SQLStatistics", SQL_API_SQLSTATISTICS, LEVEL_1, CORE, ISO_92, ODBC_1 },
    { "SQLTablePrivileges", SQL_API_SQLTABLEPRIVILEGES, LEVEL_2, LEVEL_2, ODBC, ODBC_1 },
    { "SQLTables", SQL_API_SQLTABLES, LEVEL_1, CORE, X_OPEN, ODBC_1 },
    { "SQLTransact", SQL_API_SQLTRANSACT, CORE, 0, DEPRECATED, ODBC_1 }
};

#define NUM_FUNCS (sizeof func_table / sizeof (struct func))


static SQLRETURN initialise(struct driver_block *);
static SQLRETURN connect(struct driver_block *, char *);
static SQLRETURN getfunctions(struct driver_block *);
static SQLRETURN report(struct driver_block *);
static SQLRETURN print_function (struct func *, struct driver_block *,
                                 struct driver_block *, struct driver_block *);

static SQLRETURN disconnect(struct driver_block *);
static void display_error(struct driver_block *, char *);

int main(int argc, char *argv[])
{
    SQLRETURN           retcode;
    char				connstr[1024];
    int					i;
    struct driver_block	driver[NUMBER_BEHAVIOURS];
	
    if (argc == 1)
    {
        strcpy (connstr, "DSN=demo");	
    }
    else
    {
        strcpy (connstr, argv[1]);
    }
    
    printf("\nSQLGetFunctions on %s\n\n", connstr);
    
    retcode = initialise(&driver[0]);
    
    for (i=0; i < NUMBER_BEHAVIOURS; i++)
    {
        retcode = connect(&driver[i], connstr);
    	if (SQL_SUCCEEDED(retcode))
        {
            retcode = getfunctions(&driver[i]);
        }
        else
        {
            driver[i].funcs_retcode = SQL_ERROR;
        }
        retcode = disconnect(&driver[i]);
    }
	
    retcode = report(&driver[0]);
	
    return 0;
}

static SQLRETURN initialise(struct driver_block *driver)
{
    struct driver_block *p;
	
    p = driver;
    memset (p, 0, sizeof(struct driver_block));
    p->behaviour = ODBC2;
    p->henv = SQL_NULL_HENV;
    p->hdbc = SQL_NULL_HDBC;
    p->hstmt = SQL_NULL_HSTMT;
	
    p++;
    memset (p, 0, sizeof(struct driver_block));
    p->behaviour = ODBC3_ODBC2;
    p->henv = SQL_NULL_HENV;
    p->hdbc = SQL_NULL_HDBC;
    p->hstmt = SQL_NULL_HSTMT;
	
    p++;
    memset (p, 0, sizeof(struct driver_block));
    p->behaviour = ODBC3_ODBC3;
    p->henv = SQL_NULL_HENV;
    p->hdbc = SQL_NULL_HDBC;
    p->hstmt = SQL_NULL_HSTMT;

    return 0;
}

static SQLRETURN connect(struct driver_block *driver, char *connstr)
{
    /* 
     *
     * Allocate environment handle
     *
     */
    if (driver->behaviour == ODBC2)
    {
        driver->retcode = SQLAllocEnv(
            &driver->henv);			/* EnvironmentHandle */
    }
    else
    {
        driver->retcode = SQLAllocHandle(			
            SQL_HANDLE_ENV,			/* HandleType */ 
            SQL_NULL_HANDLE,                    /* InputHandle */
            &driver->henv);			/* OutputHandlePtr */
    }
    if (!SQL_SUCCEEDED(driver->retcode))
    {
        fprintf(stderr, "Failed to allocate environment handle\n");
        return driver->retcode;
    }
    
    /*
     *
     * Set to ODBC 2.x behaviour 
     *
     */
    if (driver->behaviour == ODBC3_ODBC2)
    {
        driver->retcode = SQLSetEnvAttr(
            driver->henv,			/* EnvironmentHandle */
            SQL_ATTR_ODBC_VERSION,              /* Attribute */
            (void *) SQL_OV_ODBC2,              /* ValuePtr */
            0);                                 /* StringLength */
        if (!SQL_SUCCEEDED(driver->retcode))
    	{
            fprintf(stderr, "Failed to set ODBC 2.x behaviour\n");
            return driver->retcode;
    	}
    }


    /*
     *
     * Set to ODBC 3.x behaviour 
     *
     */
    if (driver->behaviour == ODBC3_ODBC3)
    {
        driver->retcode = SQLSetEnvAttr(
            driver->henv,			/* EnvironmentHandle */
            SQL_ATTR_ODBC_VERSION,              /* Attribute */
            (void *) SQL_OV_ODBC3,              /* ValuePtr */
            0);                                 /* StringLength */
        if (!SQL_SUCCEEDED(driver->retcode))
    	{
            fprintf(stderr, "Failed to set ODBC 3.x behaviour\n");
            return driver->retcode;
    	}
    }


    /*
     *
     * Allocate connection handle
     *
     */    
    if (driver->behaviour == ODBC2)
    {
        driver->retcode = SQLAllocConnect(
            driver->henv,			/* Environment Handle */
            &driver->hdbc);			/* Connection Handle */
    }
    else
    {
        driver->retcode = SQLAllocHandle(
            SQL_HANDLE_DBC,			/* HandleType */ 
            driver->henv,			/* InputHandle */
            &driver->hdbc);			/* OutputHandlePtr */
    }
    if (!SQL_SUCCEEDED(driver->retcode))
    {
       	fprintf(stderr, "Failed to allocate connection handle\n");
       	return driver->retcode;
    }
	
	
    /*
     *
     * Connect to driver
     *
     */
    driver->retcode = SQLDriverConnect(
        driver->hdbc,                           /* ConnectionHandle */ 
        NULL, 					/* WindowHandle */
        connstr,				/* InConnectionString */
        SQL_NTS,				/* StringLength1 */
        driver->outstr,                         /* OutConnectionString */
        sizeof(driver->outstr),                 /* BufferLength */
        &driver->outstr_len,                    /* StringLength2Ptr */
        SQL_DRIVER_COMPLETE);                   /* DriverCompletion */
    if (driver->retcode != SQL_SUCCESS)
    {
        driver->handle = driver->hdbc;
        driver->type = SQL_HANDLE_DBC;
        display_error(driver, "SQLDriverConnect");
    }
    if (!SQL_SUCCEEDED(driver->retcode)) return driver->retcode;
    

    return SQL_SUCCESS;
}

static SQLRETURN getfunctions(struct driver_block *driver)
{
    /*
     *
     * Check function support
     *
     */
    if (driver->behaviour == ODBC2 || driver->behaviour == ODBC3_ODBC2)
    {
        driver->retcode = SQLGetFunctions(
            driver->hdbc,			/* ConnectionHandle */
            SQL_API_ALL_FUNCTIONS,              /* FunctionId */
            driver->funcs2);			/* SupportedPtr */
    }
    else
    {
        driver->retcode = SQLGetFunctions(
            driver->hdbc,			/* ConnectionHandle */
            SQL_API_ODBC3_ALL_FUNCTIONS, 	/* FunctionId */
            driver->funcs3);			/* SupportedPtr */
    }
    if (driver->retcode != SQL_SUCCESS)
    {
        driver->handle = driver->hdbc;
        driver->type = SQL_HANDLE_DBC;
        display_error(driver, "SQLGetFunctions");
        memset (driver->funcs2, 0, sizeof(driver->funcs2));
        memset (driver->funcs3, 0, sizeof(driver->funcs3));
    }
    driver->funcs_retcode = driver->retcode;
    if (!SQL_SUCCEEDED(driver->retcode)) return driver->retcode;

    return SQL_SUCCESS;
}

static SQLRETURN report(struct driver_block *driver)
{
    int i;
    struct driver_block *o2;
    struct driver_block *o3_o2;
    struct driver_block *o3;
	
    o2 = driver;
    o3_o2 = driver;
    o3 = driver;
    o3_o2++;
    o3++;
    o3++;
	
    if ( (!SQL_SUCCEEDED(o2->funcs_retcode)) &&
         (!SQL_SUCCEEDED(o3_o2->funcs_retcode)) &&
         (!SQL_SUCCEEDED(o3->funcs_retcode)) )
    {
        return 0;
    }
	
    printf("Function Conformance      Version ODBC2   ODBC3   Standards  Applic. Behaviour\n");
    printf("                          Intro.                  Compliance ODBC2 ODBC3 ODBC3\n");
    printf("                                                                   ODBC2      \n");
    printf("------------------------------------------------------------------------------\n");

    for (i=0; i < NUM_FUNCS; i++)
    {
        print_function(
            &func_table[i],
            o2,
            o3_o2,
            o3);
    }
	
    printf("\n\n");
	
    return 0;
}

static SQLRETURN print_function (struct func * func, struct driver_block *odbc2,
                                 struct driver_block *odbc3_odbc2, struct driver_block *odbc3)
{

    printf("%-25s ",func->name);

    switch (func->version)
    {
      case ODBC_1:
        printf("ODBC 1  ");
        break;
      case ODBC_2:
        printf("ODBC 2  ");
        break;
      case ODBC_3:
        printf("ODBC 3  ");
        break;
      default:
        printf("        ");
        break;
    };

    switch (func->odbc2_conform)
    {
      case CORE:
        printf("Core    ");
        break;
      case LEVEL_1:
        printf("Level 1 ");
        break;
      case LEVEL_2:
        printf("Level 2 ");
        break;
      default:
        printf("        ");
        break;
    };
	
    switch (func->odbc3_conform)
    {
      case CORE:
        printf("Core    ");
        break;
      case LEVEL_1:
        printf("Level 1 ");
        break;
      case LEVEL_2:
        printf("Level 2 ");
        break;
      default:
        printf("        ");
        break;
    };
	
    switch (func->standard)
    {
      case ISO_92:
        printf("ISO 92     ");
        break;
      case DEPRECATED:
        printf("Deprecated ");
        break;
      case ODBC:
        printf("ODBC       ");
        break;
      case X_OPEN:
        printf("X/OPEN     ");
        break;
      default:
        printf("           ");
        break;
    };
	

    if ((func->func_id <= 100) && (odbc2->funcs2[func->func_id]))
        printf("Yes   ");
    else
        printf("      ");
    if ((func->func_id <= 100) && (odbc3_odbc2->funcs2[func->func_id]))
        printf("Yes   ");
    else
        printf("      ");
    if (SQL_FUNC_EXISTS(odbc3->funcs3,func->func_id) == SQL_TRUE)
        printf("Yes  ");
    else
        printf("     ");

    printf("\n");

    return 0;
}
					

static SQLRETURN disconnect(struct driver_block *driver)
{

    /*
     *
     * Close the connection
     *
     */
    driver->retcode = SQLDisconnect(
        driver->hdbc);		/* ConnectionHandle */
	
    if (!SQL_SUCCEEDED(driver->retcode))
    {
       	fprintf(stderr, "Failed to disconnect connection handle\n");
       	return driver->retcode;
    }

    /*
     *
     * Free connection handle
     *
     */
    if (driver->behaviour == ODBC2)
    {
        driver->retcode = SQLFreeConnect(
            driver->hdbc);	/* Connection Handle */
    }
    else
    {
        driver->retcode = SQLFreeHandle(
            SQL_HANDLE_DBC,	/* HandleType */
            driver->hdbc);	/* Handle */
    }
    if (!SQL_SUCCEEDED(driver->retcode))
    {
       	fprintf(stderr, "Failed to free connection handle\n");
       	return driver->retcode;
    }

    /*
     *
     * Free environment handle
     *
     */
    if (driver->behaviour == ODBC2)
    {
        driver->retcode = SQLFreeEnv(
            driver->henv);	/* EnvironmentHandle */
    }
    else
    {
        driver->retcode = SQLFreeHandle(
            SQL_HANDLE_ENV,	/* HandleType */
            driver->henv);	/* Handle */
    }
    if (!SQL_SUCCEEDED(driver->retcode))
    {
       	fprintf(stderr, "Failed to free environment handle\n");
       	return driver->retcode;
    }

    driver->hdbc = NULL;
    driver->henv = NULL;
	
    return SQL_SUCCESS;
}

static void display_error(
    struct driver_block *driver,
    char *fn)
{
    SQLSMALLINT 	rec_number;
    SQLINTEGER	native;
    char state[ 7 ];
    SQLCHAR text[1024];
    SQLSMALLINT len;
    SQLRETURN retcode;
    int			number_recs;

    /*
     *
     * Display what sort of error
     *
     */
    printf("Function: %s returned ", fn);
    switch (driver->retcode)
    {
      case SQL_SUCCESS:
        printf("SQL_SUCCESS\n");
        break;
      case SQL_SUCCESS_WITH_INFO:
        printf("SQL_SUCCESS_WITH_INFO\n");
        break;
      case SQL_ERROR:
        printf("SQL_ERROR\n");
        break;
      default:
        printf("RETCODE %d\n",driver->retcode);
        break;
    }
	
    number_recs = 0;
    rec_number = 1;
    do 
    {
	/*
	 *
	 * Get error text
	 *
	 */
        if (driver->behaviour == ODBC2)
        {
            retcode = SQLError(
                driver->henv,			/* EnvironmentHandle */
                driver->hdbc,			/* ConnectionHandle */
                driver->hstmt,			/* StatementHandle */	
                state,                          /* SqlState */
                &native,                        /* NativeErrorPtr */
                text,                           /* ErrorMsg */
                sizeof(text),			/* BufferLength */
                &len);                          /* TextLengthptr */
        }
        else
        {
            retcode = SQLGetDiagRec(
                driver->type,			/* HandleType */ 	
                driver->handle, 		/* Handle */
                rec_number,                     /* RecNumber */
                state,                          /* Sqlstate*/
                &native,                        /* NativeErrorPtr */
                text,                           /* MessageText */
                sizeof(text), 			/* BufferLength */
                &len );                         /* TextLengthPtr */
        }
        if (SQL_SUCCEEDED(retcode))
        {
            printf( "%s:%ld:%ld:%s\n", state, rec_number, native, text );
            number_recs++;
        }
        rec_number++;
    }
    while( retcode == SQL_SUCCESS );

    if (number_recs == 0)
        if (driver->behaviour == ODBC2)
            printf("SQLError did not return any error information");
        else
            printf("SQLGetDiagRec did not return any diagnostics");

    printf("\n\n");

}
