/*\
||| Written By Fredrik Hbinette <hubbe@lysator.liu.se>
||| All rights reserved. No warrenties, use at your own risk.
\*/

/*
 * Note to self:
 *
 * Filter functionality for next version:
 *   filter: pnmtogif "$file" "$outfile"
 *   
 * implementation:
 *   o let plugger register a property notification
 *     handler for the plugin window.
 *   o the child will process the data into temporary files
 *     and send the url back to plugger by setting the property
 *   o the child then waits around for a while and deletes
 *     the files afterwards. 
 *   o Make sure that plugger doesn't kill the child, then it
 *     cannot cleanup the temporary files.
 *
 *
 * Note #2:
 *   o When swallowing windows, use SubstructureRedirectMask to
 *     play window manager for that window. (We want to detect
 *     and/or prevent the window from moving and resizing itself)
 *    
 * other things to implement:
 *   $xsize, $ysize, $url
 */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sysexits.h>
#include <signal.h>
#include <stdarg.h>
#include <npapi.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <sys/time.h>
#include <X11/X.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>

#include <sys/socket.h>

#include "plugger.h"

/* UGLY */
#include "plugger-common.c"
#define WINDOW ((Window) THIS->windata.window)

struct data
{
  Display *display;
  char *displayname;
  int pid1;
  int pid2;

  NPWindow windata;

  char *mimetype;
  int repeats;

  int flags;
  char *command;
  char *winname;

#ifdef STREAMER
  int fd;
  int peekfd;
  int waitfd;

  struct timeval start_buffering;
  int buffering;
  int buffered;
  int offset;
  char *buffer;
#endif
};



#define THIS (*(struct data **) &(instance->pdata))
#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))
#define xalloc malloc
#define xfree free


/*
 * Helper functions
 */

/* Check weather the first word in 'file' is somewhere to be found in
 * the users PATH. If so, return 1 otherwise 0.
 */
static int inpath(char *file)
{
  struct stat buf;
  char tmp[16384];
  char *path=getenv("PATH");
  char *pos;
  if(!path) return 0;

  D("inpath(%s)\n",file);
  if(file[0]=='/')
    return !stat(file, &buf);

  if(!strncmp(file,"internal:",9)) return 1;

  /* Ugly ugly ugly */
  if(file[0]=='i' && file[1]=='f' && isspace(file[2]))
    file+=3;

  /* Insert magical shell-script tests here */

  D("Hmm? PATH=%s\n",path);
  pos=path;
  while(1)
  {
    char *next;
    next=FEND(pos,':');
    if(next!=pos)
    {
      char *ptr=tmp;

      memcpy(ptr=tmp,pos,next-pos);
      ptr+=next-pos;
      *(ptr++)='/';
      memcpy(ptr,
	     file,
	     FEND(file,' ')-file);
      ptr+=FEND(file,' ')-file;
      *ptr=0;
      D("stat(%s)\n",tmp);
      if(!stat(tmp, &buf)) return 1;
      D("nope\n");
    }
    if(!*next) return 0;
    pos=next+1;
  }
  D("GAAHHH!\n");
}

#ifndef MAX_FD
#define MAX_FD 1024
#endif

/* Shell for fork which handles some magic needed to prevent
 * netscape from freaking out. It also prevents interferrance
 * from signals.
 */

static int my_fork(NPP instance, int fd1, int fd2)
{
  int pid;
  sigset_t set,oset;
  sigfillset(& set);
  D(">>>>>>>>Forking<<<<<<<<,\n");
  sigprocmask(SIG_SETMASK,&set,&oset);
  pid=fork();
  if(pid==-1) return pid;
  if(!pid)
  {
    int f;
    int signum;
    alarm(0);
    if(!(THIS->flags & H_DAEMON))
      setsid();

    for(signum=0;signum<NSIG;signum++) signal(signum, SIG_DFL);

    for(f=3;f<MAX_FD;f++)
    {
      if(f == fd1) continue;
      if(f == fd2) continue;
      close(f);
    }
#ifdef DEBUG
    debug_output=0;
    D("Chilid re-opened ndebug\n");
#endif
    THIS->display=0;
  }else{
#ifdef STREAMER
    if(THIS->peekfd>=0)
      close(THIS->peekfd);
#endif
    D("Child running with pid=%d\n",pid);
  }
  sigprocmask(SIG_SETMASK,&oset,&set);
  return pid;
}

static char *helper_fname=NULL;

static void run(NPP instance, const char *file)
{
  char buffer[1024];

  sprintf(buffer,"%d,%d,%d,%d,%d,%d,%d,%d",
	  THIS->flags,
	  THIS->repeats,
	  THIS->waitfd,
	  THIS->windata.window,
	  THIS->windata.x,
	  THIS->windata.y,
	  THIS->windata.width,
	  THIS->windata.height);
  D("Executing helper: %s %s %s %s %s %s %s\n",
    helper_fname,
    buffer,
    file,
    THIS->winname,
    THIS->displayname,
    THIS->command,
    THIS->mimetype);

  execlp(helper_fname,
	 helper_fname,
	 buffer,
	 file ? file : "",
	 THIS->winname ? THIS->winname : "",
	 THIS->displayname ? THIS->displayname : "",
	 THIS->command,
	 THIS->mimetype,
	 0);

  D("EXECLP FAILED!\n");
	 
  _exit(69); /* EX_UNAVAILABLE */
}


/* 
 * Functions below
 * are called from inside netscape/mozilla/opera
 */

struct mimetype
{
  struct mimetype *next;
  char *line;
};

struct command
{
  struct command *next;
  int flags;
  char *cmd;
  char *winname;
};

struct handle
{
  struct handle *next;
  struct mimetype *types;
  struct command *commands;
};

static struct handle *first_handle=NULL;

static int gobble(char *line, char *kw)
{
  return !strncasecmp(line,kw,strlen(kw)) && !isalnum(line[strlen(kw)]);
}

/* read configuration file into memory */
static void read_config(FILE *f)
{
  struct handle **handlep;
  struct command **commandp=0;
  struct mimetype **typep=0;
  char buffer[16384];
  int have_commands=1;
  D("read_config\n");

  handlep=&first_handle;

  while(!feof(f))
  {
    fgets(buffer,sizeof(buffer),f);
    D("::: %s",buffer);
    if(buffer[0]=='#' || !buffer[0] || buffer[0]=='\n') continue;

    if(buffer[strlen(buffer)-1]=='\n')
      buffer[strlen(buffer)-1]=0;

    if(isspace(buffer[0]))
    {
      char *x=buffer+1;
      while(isspace(*x)) x++;
      if(!*x)
      {
	D("Empty line.\n");
	continue;
      }

      D("New command\n");

      *commandp=(struct command *)xalloc(sizeof(struct command));
      if(!*commandp)
      {
	D("xalloc failed\n");
	return;
      }
      (*commandp)->flags=0;
      (*commandp)->cmd=0;
      (*commandp)->winname=0;
      (*commandp)->next=0;

      /* Command */
      have_commands++;
      while(*x!=':' && *x)
      {
/* 	D("Parsing %s\n",x); */
	switch(*x)
	{
	  case ',':
	  case ' ':
	  case '\t':
	    x++;
	    break;

	  default:
#define GOBBLE(X,Y) \
  if(gobble(x,X)) { x+=strlen(X); (*commandp)->flags|=Y; break; }
	    GOBBLE("repeat",H_REPEATCOUNT);
	    GOBBLE("loop",H_LOOP);
	    GOBBLE("stream",H_STREAM);
	    GOBBLE("preload",H_PRELOAD);
	    GOBBLE("many",H_MANY);
	    GOBBLE("ignore_errors",H_IGNORE_ERRORS);
	    GOBBLE("exits",H_EXITS);
	    GOBBLE("nokill",H_DAEMON);

	    GOBBLE("maxaspect",H_MAXASPECT);
	    GOBBLE("fill",H_FILL);
#ifdef H_NOISY
	    GOBBLE("noisy",H_NOISY);
#endif
	    if(gobble(x,"swallow"))
	    {
	      char *end;
	      (*commandp)->flags|=H_SWALLOW;
	      x+=7;
	      while(isspace(*x)) x++;
	      if(*x != '(')
	      {
		fprintf(stderr,"Expected '(' after 'swallow'\n");
		continue;
	      }
	      x++;
	      end=strchr(x,')');
	      if(end)
	      {
		(*commandp)->winname=xalloc(end - x +1);
		memcpy((*commandp)->winname, x, end-x);
		(*commandp)->winname[end-x]=0;
	      }
	      x=end+1;
	      break;
	    }
	    D("Unknown directive: %s\n",x);

	    /* Unknown directive */
	    fprintf(stderr,"Unknown directive: %s\n",x);
	    x+=strlen(x);
	}
      }
      if(*x==':')
      {
	x++;
	while(isspace(*x)) x++;
	(*commandp)->cmd=strdup(x);
      }else{
	D("No colon? (%s)\n",x);
      }
      if(!(*commandp)->cmd)
      {
	xfree(*commandp);
	*commandp=0;
	D("strdup failed %s\n",x);
	return;
      }
      commandp=&((*commandp)->next);
    }else{
      /* Mime type */

      if(have_commands)
      {
  	D("New handle\n");
	if(commandp)
	  D("Commandp=%p\n",*commandp);
	*handlep=(struct handle *)xalloc(sizeof(struct handle));
	if(!*handlep)
	  return;
	
	(*handlep)->commands=0;
	(*handlep)->types=0;
	(*handlep)->next=0;
	commandp=&((*handlep)->commands);
	typep=&((*handlep)->types);
	handlep=&((*handlep)->next);
	have_commands=0;
      }

      D("New mimetype\n");
      *typep=(struct mimetype *)xalloc(sizeof(struct mimetype));
      if(!*typep)
	return;

      (*typep)->next=0;
      (*typep)->line=strdup(buffer);

      if(!(*typep)->line)
      {
	xfree(*typep);
	*typep=0;
	return;
      }
      typep=&((*typep)->next);
    }
  }
}


int find_helper_file(char *basename,
		     int (*cb)(char *,void *data),
		     void *data)
{
  static char fname[8192];
  char *tmp;
  D("find_helper_file '%s'\n",basename);
  if((tmp=getenv("HOME")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/.netscape/%s",tmp,basename);
    if(cb(fname,data)) return 1;
    sprintf(fname,"%s/.mozilla/%s",tmp,basename);
    if(cb(fname,data)) return 1;
    sprintf(fname,"%s/.opera/%s",tmp,basename);
    if(cb(fname,data)) return 1;
  }

  if((tmp=getenv("MOZILLA_HOME")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/%s",tmp,basename);
    if(cb(fname, data)) return 1;
  }

  if((tmp=getenv("OPERA_DIR")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/%s",tmp,basename);
    if(cb(fname, data)) return 1;
  }

  sprintf(fname,"/usr/local/netscape/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/etc/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/usr/etc/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/usr/local/etc/%s",basename);
  if(cb(fname, data)) return 1;
  if(cb(basename, data)) return 1;
  
  return 0;
}

static int read_config_cb(char *fname, void *data)
{
  FILE *f=fopen(fname,"r");
  if(!f) return 0;
  read_config(f);
  fclose(f);
  return 1;
}

static int find_plugger_helper_cb(char *fname, void *data)
{
  struct stat buf;
  if(stat(fname, &buf)) return 0;
  helper_fname=strdup(fname);
  return 1;
}

/* Find configuration file and read it into memory */
static void do_read_config(void)
{
  if(first_handle) return;
  D("do_read_config\n");
  
  if(!find_helper_file("pluggerrc-" VERSION,read_config_cb,0) &&
     !find_helper_file("pluggerrc",read_config_cb,0))
  {
    fprintf(stderr,"Plugger: Unable to find pluggerrc file!\n");
    return;
  }
  D("do_read_config done\n");

  find_helper_file("plugger-" VERSION, find_plugger_helper_cb, 0);
  if(inpath("plugger-" VERSION))
    helper_fname="plugger-" VERSION;

  if(!helper_fname)
    fprintf(stderr,"Plugger: Unable to find plugger-" VERSION "\n");
}

/* Construct a MIME Description string for netscape
 * so that netscape shall know when to call us back.
 */
char *NPP_GetMIMEDescription(void)
{
  char *x,*y;
  struct handle *h;
  struct mimetype *m;
  int size_required;

  do_read_config();

  D("Getmimedescription\n");

  size_required=0;
  for(h=first_handle;h;h=h->next)
    for(m=h->types;m;m=m->next)
      size_required+=strlen(m->line)+1;

  D("Size required=%d\n",size_required);
  x=(char *)xalloc(size_required+1);
  if(!x) return 0;

  D("Malloc did not fail\n");

  y=x;

  for(h=first_handle;h;h=h->next)
  {
    D("Foo: %p\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
      char *tmp;
/*       D("appending: %s\n",m->line); */
      memcpy(y,m->line,strlen(m->line));
      y+=strlen(m->line);
      *(y++)=';';
    }
  }
  if(x!=y) y--;
  *(y++)=0;
  D("Getmimedescription done: %s\n",x);
  return x;
}

/* Go through the commands in the config file 
 * and find one that fits our needs.
 */
static int find_command(NPP instance,
			int streaming)
{
  struct handle *h;
  struct mimetype *m;
  struct command *c;

  D("find_command\n");

  do_read_config();

  D("find_command...\n");

  for(h=first_handle;h;h=h->next)
  {
    D("commands for this handle at (%p)\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
      char *tmp1=FEND(m->line,':');
      char tmp2;
      int tmp3;
      D("Checking '%s'\n",m->line);

      while(tmp1>m->line && isspace(tmp1[-1]))
	tmp1--;

      D("tmp1=%s\n",tmp1);

      tmp2=*tmp1;
      *tmp1=0;
      D("Checking '%s' ?= '%s'\n",m->line,THIS->mimetype);
      tmp3=strcasecmp(THIS->mimetype, m->line);
      *tmp1=tmp2;
      if(!tmp3)
      {
	D("Match found!\n");
	break;
      }else{
	D("Not same.\n");
      }
    }

    if(m)
    {
      for(c=h->commands;c;c=c->next)
      {
	D("Checking command: %s\n",c->cmd);
	if((c->flags & H_LOOP) && THIS->repeats != MAXINT) continue;
	if( (!!streaming) != (!!(c->flags & H_STREAM))) continue;
	if(!inpath(c->cmd)) continue;
	D("Match found!\n");
	THIS->command=c->cmd;
	THIS->flags=c->flags;
	THIS->winname=c->winname;
	return 1;
      }
    }
  }
  D("No match found\n");
  return 0;
}

/* Let netscape know things about plugger */
NPError NPP_GetValue(void *future, NPPVariable variable, void *value)
{
  NPError err = NPERR_NO_ERROR;

  D("Getvalue %d\n",variable);
  
  switch (variable)
  {
  case NPPVpluginNameString:
    D("GET Plugin name\n");
    *((char **)value) = "Plugger "
      VERSION
      ;

  break;

  case NPPVpluginDescriptionString:
    D("GET Plugin description\n");
    *((char **)value) =
      "<img width=40 height=40 border=0 align=left src=http://fredrik.hubbe.net/plugger/logo.gif>"
      "<a href=http://fredrik.hubbe.net/plugger.html>Plugger</a> "
      "version "
      VERSION
      ", written by "
      "<a href=http://fredrik.hubbe.net/>Fredrik Hbinette</a> "
      "<a href=mailto:hubbe@hubbe.net>&lt;hubbe@hubbe.net&gt</a>. "
      "For documentation on how to configure plugger, go to the plugger "
      " <a href=http://fredrik.hubbe.net/plugger.html>homepage</a> "
      "or check the man page. (type <tt>man&nbsp;plugger</tt>)"
      ;
  break;

  default:
    err = NPERR_GENERIC_ERROR;
  }
  return err;
}

/* Initiate another instance of plugger, it is important to know
 * that there might be several instances going at one time.
 */
NPError  NPP_New(NPMIMEType pluginType,
		 NPP instance,
		 uint16 mode,
		 int16 argc,
		 char* argn[],
		 char* argv[],
		 NPSavedData* saved)
{
  int e;
  NPError tmp;

  D("NEW (%s)\n",pluginType);

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  THIS = NPN_MemAlloc(sizeof(struct data));
  if(!THIS) return NPERR_OUT_OF_MEMORY_ERROR;
  memset((char *)THIS, 0, sizeof(*THIS));

  /* Change default to repeat once? */
  THIS->repeats=0x7fffffff;
  THIS->pid1=-1;
  THIS->pid2=-1;
  WINDOW=0;
  THIS->displayname=0;
  THIS->display=0;
  
#ifdef STREAMER
  THIS->fd=-1;
  THIS->waitfd=-1;
  THIS->peekfd=-1;
#endif

  if(!pluginType) return NPERR_GENERIC_ERROR;

  for(e=0;e<argc;e++)
  {
    if(!strcasecmp("loop",argn[e]))
    {
      switch(argv[e][0])
      {
      case 't': case 'T': case 'y': case 'Y':
	THIS->repeats=MAXINT;
	break;
      case 'f': case 'F': case 'n': case 'N':
	THIS->repeats=1;
	break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	THIS->repeats=atoi(argv[e]);
      }
    }
  }

  return NPERR_NO_ERROR;
}

/* Free data, kill processes, it is time for this instance to die */
NPError NPP_Destroy(NPP instance, NPSavedData** save)
{
  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  D("Destroy\n");

  if (THIS)
  {
    if(THIS->pid1 > 0)
    {
      my_kill(-THIS->pid1);
      my_kill(THIS->pid1);
    }
    
    if(THIS->pid2 > 0)
    {
      my_kill(-THIS->pid2);
      my_kill(THIS->pid2);
    }

    D("Freeing memory %p\n",THIS->mimetype);
    
    if(THIS->mimetype)
    {
      NPN_MemFree(THIS->mimetype);
      THIS->mimetype=0;
    }

#ifdef STREAMER
    D("Closing fds\n");
    if(THIS->fd >= 0)
    {
      close(THIS->fd);
      THIS->fd=-1;
    }
    
    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    
    if(THIS->waitfd >= 0)
    {
      close(THIS->waitfd);
      THIS->waitfd=-1;
    }
    
    if(THIS->buffer)
    {
      NPN_MemFree(THIS->buffer);
      THIS->buffer=0;
    }

    NPN_MemFree(THIS);
    THIS = 0;
#endif
  }
  D("Destroy finished\n");
  
  return NPERR_NO_ERROR;
}


/* Open a new stream,
 * each instance can only handle one stream at a time.
 */
NPError NPP_NewStream(NPP instance,
		      NPMIMEType type,
		      NPStream *stream, 
		      NPBool seekable,
		      uint16 *stype)
{
  int wantstream;
  D("Newstream ... \n");

  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;

  /* This is a stupid special case and should be coded into
   * pluggerc instead...
   */
  if(!strncasecmp("image/",type,6) ||
     !strncasecmp("x-image/",type,6))
    THIS->repeats=1;

  D("Mime type %s\n",type);

  if(THIS->mimetype)
  {
    NPN_MemFree(THIS->mimetype);
    THIS->mimetype=0;
  }
  THIS->mimetype = NPN_MemAlloc(strlen(type)+1);
  if(!THIS->mimetype) return NPERR_OUT_OF_MEMORY_ERROR;
  strcpy(THIS->mimetype, type);

  D("Url is %s (seekable=%d)\n",stream->url,seekable);
#ifdef STREAMER
  /* Hmm, this seems weird... */
  wantstream=!(seekable && !strncasecmp(stream->url,"file:",5));
#else
  wantstream=0;
#endif
  
  if(!find_command(instance, wantstream))
  {
    if(!find_command(instance, !wantstream))
    {
      if(!inpath("xmessage"))
      {
	NPN_Status(instance, "No approperiate application found!");
	return NPERR_GENERIC_ERROR;
      }else{
	D("Using xmessage!!!\n");
	/* Hard-coded default */
	THIS->command="xmessage -buttons '' \"Plugger: No approperiate application for type $mimetype found!\"";
	THIS->flags = H_REPEATCOUNT | H_FILL | H_SWALLOW | H_IGNORE_ERRORS;
	THIS->winname="Xmessage";
      }
    }
  }

#ifdef STREAMER
  if((THIS->flags & H_STREAM) && strncasecmp(stream->url,"file:",5))
  {
    int foo[2],bar[2];

    if(THIS->repeats == 1 || 
      (THIS->flags & H_LOOP) ||
      (THIS->flags & H_DAEMON) ||
      (THIS->flags & H_REPEATCOUNT))
      *stype=NP_NORMAL;
    else
      *stype=NP_ASFILE;

    if(socketpair(AF_UNIX, SOCK_STREAM, 0, foo)<0 ||
       socketpair(AF_UNIX, SOCK_STREAM, 0, bar) < 0)
    {
      NPN_Status(instance, "Streamer: Failed to create a pipe!");
      return NPERR_GENERIC_ERROR;
    }
    D("SOCKETS: %d<->%d  &&  %d<->%d\n",foo[0],foo[1],bar[0],bar[1]);

    THIS->pid1=my_fork(instance, bar[1], foo[0]);
    if(THIS->pid1==-1)
    {
      NPN_Status(instance, "Streamer: My_Fork failed!");
      return;
    }
    
    if(!THIS->pid1)
    {
      D("Streaming child running\n");
      dup2(foo[0],0);
      close(foo[0]);

      THIS->repeats=1;
      THIS->waitfd=bar[1];
      D("CHILD RUNNING run() [1]\n");
      run(instance, 0);
      exit(69);
    }else{
      THIS->buffer=NPN_MemAlloc(BUFSIZE);
      if(!THIS->buffer)
	return NPERR_OUT_OF_MEMORY_ERROR;
      
      if(THIS->repeats < MAXINT)
	THIS->repeats--;
      THIS->fd=foo[1]; /* This is the FD we write data to */
      D("FD to parent = %d\n",THIS->fd);
      fcntl(THIS->fd, F_SETFL, O_NDELAY);

      THIS->waitfd=bar[0];
      close(bar[1]);
      if(THIS->flags & H_PRELOAD)
      {
	gettimeofday(& THIS->start_buffering,0);
	THIS->buffering=1;
	THIS->peekfd=foo[0];
      }else{
	close(foo[0]);
      }
    }
    
    D("Ok\n");
  }else
#endif
    *stype = NP_ASFILEONLY;
  
  return NPERR_NO_ERROR;
}


#ifdef STREAMER


/* Try to write data to our children */
static int trywrite(NPP instance)
{
  D("trywrite (%d bytes buffered) fd=%d\n",THIS->buffered,THIS->fd);

  if(THIS->buffering)
  {
    struct timeval t;
    int p, p1, p2, p3;
    gettimeofday(&t, 0);
    
    p3=p1=THIS->buffered * 100;
    p1/=PRELOAD;
    p3/=BUFSIZE;

    p2=((t.tv_sec - THIS->start_buffering.tv_sec) * 100 +
	(t.tv_usec - THIS->start_buffering.tv_usec) / (10000) )
      / PRELOAD_TIME;
    
    /* If we have less than PRELOAD bytes,
     * or, if we have been buffering for less than
     * PRELOAD_TIME seconds, we keep buffering
     */
    
    p=MAXIMUM(p3,MINIMUM(p1,p2));
    
    if(p<100)
    {
      char b[256];
      
      sprintf(b,"Buffering ... %2d%%",p);
      D("%s\n",b);
      NPN_Status(instance, b);
      return 1;
    }
  
    THIS->buffering=0;
  }

  if(THIS->peekfd>=0)
  {
    D("Checking waitfd\n");
    if(data_available(THIS->waitfd))
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
  }

  while(THIS->buffered)
  {
    int i;

    do {
      D("trywrite %d bytes (offset = %d)\n",
	MIN(THIS->buffered, BUFSIZE-THIS->offset),
	THIS->offset);
      i=write(THIS->fd,
	      THIS->buffer+THIS->offset,
	      MIN(THIS->buffered, BUFSIZE-THIS->offset) );
      D("Wrote %d bytes (errno = %d)\n",i, errno);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	return 1;
      default:
	D("trywrite: Errno = %d\n",errno);
	return 0;
      }
    }

    if(!i) return 1;

    THIS->offset+=i;
    THIS->buffered-=i;
    if(THIS->offset == BUFSIZE) THIS->offset=0;
  }

  D("Checking preload\n");

  /* Hmm, this will not work properly with 
   * programs which buffer more data than we do..
   * With luck it doesn't affect streaming though.
   */
  if((THIS->flags & H_PRELOAD) &&
     THIS->peekfd>=0 &&
     !data_available(THIS->peekfd))
  {
    D("(Re)-starting preload\n");
    THIS->buffering=1;
    gettimeofday(& THIS->start_buffering,0);
  }
  D("trywrite-exit: errno = %d\n",errno);
  return 1;
}
#endif

/* */
int32 NPP_WriteReady(NPP instance, NPStream *stream)
{
#ifdef STREAMER
  D("Writeready\n");
  if(!instance) return 0;

  trywrite(instance);

  D("Writeready returns: %d\n", BUFSIZE - THIS->buffered);

  /* Kluge to make netscape behave! */
  /* A small delay is invoked because our children has not
   * ingested all data we have fed them yet. Without this delay
   * netscape would use 100% cpu until all data is ingested.
   */

  if(!BUFSIZE - THIS->buffered)
  {
    struct timeval timeout;
    timeout.tv_sec=0;
    timeout.tv_usec=1000;
    select(0,0,0,0,&timeout);
  }
  return BUFSIZE - THIS->buffered;
#else
  return 0x7fffffff;
#endif
}

int32 NPP_Write(NPP instance,
		NPStream *stream,
		int32 offset,
		int32 len,
		void *buf)
{
#ifdef STREAMER
  int32 origlen=len;
  int i;
  char *buffer=buf;
  D("Write(len=%d, offset=%d)\n",len,offset);
  if (!instance) return 0;

  if(THIS->fd==-1)
  {
    THIS->buffered=0;
    return len;
  }
		
  if(!trywrite(instance))
    return -1;

  D("Write: THIS->buffered=%d\n",THIS->buffered);
  if(!THIS->buffered && !THIS->buffering)
  {
    D("Attempting direct write\n");
    do {
      i=write(THIS->fd, buffer, len);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	D("Nothing written\n");
	break;

      default:
	D("Errno: %d\n",errno);
	return -1;
      }
    }else{
      D("Wrote %d bytes directly\n",i);
      buffer+=i;
      len-=i;
    }
  }

  while(len>0 && THIS->buffered<BUFSIZE)
  {
    int end=THIS->offset + THIS->buffered;
    end%=BUFSIZE;
    i=MIN(len, BUFSIZE-end);

    i=MIN(i, BUFSIZE-THIS->buffered);

    memcpy(THIS->buffer+end, buffer, i);
    len-=i;
    buffer+=i;
    THIS->buffered+=i;
  }

  D("Write returns %d\n",origlen-len);
  return origlen-len;
#else
  return len;
#endif
}

NPError NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
  D("Destroystream\n");
#ifdef STREAMER
  if(THIS->flags & H_STREAM)
  {
    THIS->buffering=0;

    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    if(trywrite(instance))
    {
      if(THIS->buffered)
      {
	/* We fork again, somebody has to send the rest of our
	 * stream to our children.
	 */
	int pid=my_fork(instance,THIS->fd,-1);
	if(pid==-1) return NPERR_GENERIC_ERROR;
	if(!pid)
	{
	  fcntl(THIS->fd, F_SETFL, 0);
	  while(THIS->buffered && trywrite(instance));
	  D("Buffer-cleanup done\n");
	  _exit(0);
	}
      }
    }
    
    close(THIS->fd);
    THIS->fd=-1;
  }
  D("Destroystream done\n");
#endif
  return NPERR_NO_ERROR;
}

void NPP_StreamAsFile(NPP instance,
		      NPStream *stream,
		      const char* fname)
{
  int pid;
  D("Streamasfile\n");
  if(!fname) return;
  if (instance == NULL) return;

  NPN_Status (instance, "Running helper ...");

  if(!strcmp(THIS->command, "internal:url") && fname)
  {
    int fd;
    long len;
    struct stat buf;
    char *url;

    if(stat(fname, &buf))
    {
      if(stream->end)
      {
	len=stream->end;
      }else{
	NPN_Status(instance,"Plugger: Failed to determine length of file\n");
	return;
      }
    }else{
      len=buf.st_size;
    }
  

    url=NPN_MemAlloc(len+1);

    D("INTERNAL URL\n");
    fd=open(fname,O_RDONLY);
    if(fd<0)
    {
      NPN_Status(instance,"Plugger: Hey, where did the file go?\n");
    } else {
      if(read(fd, url, len) == len)
      {
	url[len]=0;
	FEND(url,'\n')[0]=0;
	D("URL: %s\n",url);
	NPN_GetURL(instance, url, 0);
      }
      close(fd);
    }
    NPN_MemFree(url);
  }else{
    int bar[2];
    bar[0]=-1;
    bar[1]=-1;
    socketpair(AF_UNIX, SOCK_STREAM, 0, bar);
    
    D("......going to fork......\n");
    THIS->pid2=my_fork(instance,THIS->waitfd,bar[1]);

    if(THIS->pid2==-1) return;
    
    if(!THIS->pid2)
    {
      D("CHILD RUNNING run() [7]\n");
#ifdef STREAMER
      if(THIS->flags & H_STREAM)
      {
	char foo[1];
	D("Waiting for streaming child to exit.\n");
	while(read(THIS->waitfd, foo, 1) < 0 && errno==EINTR);
	if(THIS->repeats < MAXINT)
	  THIS->repeats--;
      }
#endif
      D("CHILD RUNNING run() [9]\n");
      THIS->waitfd=bar[1];

      if(!find_command(instance, 0))
      {
	if(!find_command(instance, 1))
	{
	  if(inpath("xmessage"))
	  {
	    D("Using xmessage!!!\n");
	    /* Hard-coded default */
	    THIS->command="xmessage -buttons '' \"Plugger: No approperiate application for type $mimetype found!\"";
	    THIS->flags = H_REPEATCOUNT | H_FILL | H_SWALLOW | H_IGNORE_ERRORS;
	    THIS->winname="Xmessage";
	  }else{
	    exit(EX_UNAVAILABLE);
	  }
	}
      }

      D("CHILD RUNNING run() [2]\n");
      run(instance, fname);
      exit(69);
    }else{
      THIS->waitfd=bar[0];
      close(bar[1]);
    }
  }
}
  
NPError NPP_Initialize(void)
{
  D("init\n");
  return NPERR_NO_ERROR;
}
jref NPP_GetJavaClass()
{
  D("Getjavaclass\n");
  return NULL;
}
void NPP_Shutdown(void)
{
  D("Shutdown\n");
}

NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
  D("SETWINDOW\n");

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  if (!window)
    return NPERR_NO_ERROR;
  
  if (!window->window)
    return (NPERR_NO_ERROR);

  if (!window->ws_info)
    return (NPERR_NO_ERROR);

  THIS->display= ((NPSetWindowCallbackStruct *)window->ws_info)->display;
  THIS->displayname=XDisplayName( DisplayString(THIS->display));
  THIS->windata = *window;
  D("Displayname=%s Window=%x %d %d %d %d\n",
    THIS->displayname,
    WINDOW,
    window->x,window->y,
    window->width, window->height);

  XResizeWindow(THIS->display,
		WINDOW,
		window->width,
		window->height);

  XSync (THIS->display, FALSE);

  if(THIS->waitfd!=-1)
  {
    D("Writing WIN to fd %d\n",THIS->waitfd);
    my_write(THIS->waitfd, (char *)window, sizeof(*window));
  }
  if(THIS->pid1 != -1)
  {
    D("Sending SIGWINCH to pid1 (%d)\n",THIS->pid1);
    kill(THIS->pid1, SIGWINCH);
  }
  if(THIS->pid2 != -1)
  {
    D("Sending SIGWINCH to pid2 (%d)\n",THIS->pid2);
    kill(THIS->pid2, SIGWINCH);
  }

  usleep(4000);

  return NPERR_NO_ERROR;
}

void NPP_Print(NPP instance, NPPrint* printInfo)
{
  D("PRINT\n");
  if(printInfo == NULL)
    return;
  
  if (instance != NULL)
  {
    if (printInfo->mode == NP_FULL) {
      void* platformPrint =
	printInfo->print.fullPrint.platformPrint;
      NPBool printOne =
	printInfo->print.fullPrint.printOne;
      
      /* Do the default*/
      printInfo->print.fullPrint.pluginPrinted = FALSE;
    }
    else
    {	/* If not fullscreen, we must be embedded */
      NPWindow* printWindow =
	&(printInfo->print.embedPrint.window);
      void* platformPrint =
	printInfo->print.embedPrint.platformPrint;
    }
  }
}


