/* files.c
 *
 * (C) Copyright 2001 Air Force Office of Special Investigations
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "foremost.h"

/* Audit File
   The audit file is opened before we do the digging. 
   This file will be closed either at the end of the program,
   or by a signal_handler if the user sends a SIGINT or SIGTERM...
*/

int openAuditFile(struct foremostState* state){

  time_t now = time(NULL);
  char* timestring = ctime(&now);
  char fn[MAX_STRING_LENGTH];
  
  snprintf(fn,MAX_STRING_LENGTH,"%s/audit.txt",
	   state->outputdirectory);

  if(!(state->auditFile = fopen(fn,"w"))){
    fprintf(stderr,"Couldn't open %s -- %s\n",fn,strerror(errno));
    return FOREMOST_ERROR_FILE_OPEN;
  }

  fprintf(state->auditFile,
	  "\nForemost version %s audit file\nStarted at %swith commandline: %s\n\n",
	  FOREMOST_VERSION, timestring, state->invocation);

  fprintf(state->auditFile,
	  "Using output directory: %s\n\n",
	  state->outputdirectory);
  
  if (state->skip){
    fprintf(state->auditFile,"Skipped the first %Ld bytes of %s...\n",
	    state->skip, state->imagefile);
  }
  
  if (state->modeQuick){
    fprintf(state->auditFile,
	    "Quick mode enabled. (Searches the beginning of each sector only)\n");
  }

  fprintf(state->auditFile,
	  "File            Found at Byte  Interior   Length     Extracted From\n");
  return FOREMOST_OK;
}



int closeFile(FILE* f){

  time_t now = time(NULL);
  char* timestring = ctime(&now);

  fprintf (f, "\n\nCompleted at %s", timestring);

  if(fclose(f)){
    return FOREMOST_ERROR_FILE_CLOSE;
  }

  return FOREMOST_OK;
}



/* writeToDisk(char*                 suffix of file to be written,
               struct foremostState* program state
               struct CharBucket*    text to be written)
	       unsigned long long    offset in the image file
	                               from we should start writing
   
   Writes text, a file of format suffix, to the disk, in order.
   Returns FOREMOST_OK on success.                             */

int writeToDisk(char* suffix, 
		struct foremostState* state, 
		struct CharBucket* text,
		unsigned long long offset){

  char  fn[MAX_STRING_LENGTH], sectorHeader = ' ';
  FILE* f;
  long byteswritten = 0;

  snprintf(fn,MAX_STRING_LENGTH,"%s/%08d.%s",
	   state->outputdirectory,state->fileswritten,suffix);

  if (offset % FOREMOST_BLOCK_SIZE != 0) {
    sectorHeader = 'X';
  }

  /* Originally we wrote the full path to the audit file, but that leads
     to problems when the path is long. Because the path is printed at 
     the top of the audit file, we don't have to repeat it every time. (JK) */

  fprintf(state->auditFile,
	  "%08d.%s    %13Ld     %c      %7ld     %s\n",
	  state->fileswritten,
	  suffix,offset,sectorHeader,text->length,state->imagefile);

  if(!(f = fopen(fn,"w"))){
    fprintf (stderr,           "Error opening file: %s -- %s\n", 
	     fn, strerror(errno));
    fprintf (state->auditFile, "Error opening file: %s -- %s\n", 
	     fn, strerror(errno));
    return FOREMOST_ERROR_FILE_OPEN;
  }
  
  if ((byteswritten = fwrite(text->str,
			     sizeof(char),
			     text->length,f)) != text->length) {

    fprintf(stderr,"Error writing to file: %s -- %s\n",
	    fn, strerror(ferror(f)));
    fprintf(state->auditFile,"Error writing to file: %s -- %s\n",
	    fn, strerror(ferror(f)));
    return FOREMOST_ERROR_FILE_WRITE;
  }

  if(fclose(f)){
    fprintf(stderr,           "Error closing file: %s -- %s\n\n",
	    fn,strerror(ferror(f)));
    fprintf(state->auditFile, "Error closing file: %s -- %s\n\n",
	    fn,strerror(ferror(f)));
    return (FOREMOST_ERROR_FILE_WRITE);
  }
    
  if (state->modeVerbose) {
    fprintf (stdout,"Wrote file %s -- Success\n", fn);
  }

  /* We only say that we wrote the file if we were successful. This 
     statement was originally immediately after the snprintf for the
     filename. Because we use the variable fileswritten elsewhere in
     this function I've moved it down here.   (JK) */
  state->fileswritten++;

  return FOREMOST_OK;
}



/* Return the size, in bytes of an open file stream. On error, return -1 */
unsigned long long measureOpenFile(FILE *f){
  
  int descriptor = 0;
  unsigned long long numsectors, total, original = ftello(f);
  
  descriptor = fileno(f);
  
  if ((fseeko(f,0,SEEK_END)))
    return -1;
  total = ftello(f);
  if ((fseeko(f,original,SEEK_SET)))
    return -1;

  if (total == 0) { 

    /* Maybe it's a device that is incapable of seeking? */

    if (ioctl(descriptor, BLKGETSIZE, &numsectors)){
#ifdef DEBUG
      perror("BLKGETSIZE failed");
#endif
    } else {
      total = numsectors*512; /* This is an assumption... Maybe we should find 
				 a way to get the sectors size from device. */
    }
  }
  return (total - original);
}










