/* foremost.h
 *
 * (C) Copyright 2001 Air Force Office of Special Investigations 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef FOREMOST_H
#define FOREMOST_H
#define FOREMOST_VERSION "0.62"

#define _USE_LARGEFILE              1
#define _USE_FILEOFFSET64           1
#define _USE_LARGEFILE64            1
#define _LARGEFILE_SOURCE           1
#define _LARGEFILE64_SOURCE         1
#define _FILE_OFFSET_BITS           64

#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <sys/mount.h>
#include <time.h>
#include <error.h>
#include <errno.h>
#include <signal.h>


/* SIZE_OF_BUFFER represents how much data we read from an image file
   at a time. We use 10 MB = 10 * 1024 * 1024 bytes = 10485760 bytes
   This value must be divisible by FOREMOST_BLOCK_SIZE for quick mode to 
   work. If it's not, the program will abort with the error message 
   that follows.     */

#define SIZE_OF_BUFFER            10485760

#define FOREMOST_SIZEOFBUFFER_PANIC_STRING \
"PANIC: There is a serious problem with this copy of foremost.\n"\
"       Please obtain a clean version of this program and try again.\n"\
"       (For programmers, the constant SIZE_OF_BUFFER must be divisible\n"\
"       by FOREMOST_BLOCK_SIZE for quick mode to work correctly.\n"


#define FOREMOST_BLOCK_SIZE           512
#define MAX_STRING_LENGTH            4096
#define MAX_NEEDLES                   254
#define NUM_SEARCH_SPEC_ELEMENTS        5
#define MAX_SUFFIX_LENGTH               8
#define MAX_FILE_TYPES                100

#define FOREMOST_OK                     0
#define FOREMOST_ERROR_NO_SEARCH_SPEC   1

/* When we can't open an image file */
#define FOREMOST_ERROR_FILE_OPEN        2

/* When we can't read an image file */
#define FOREMOST_ERROR_FILE_READ        3

/* When we can't write to the output directory */
#define FOREMOST_ERROR_FILE_WRITE       4

/*  When we can't close a file */
#define FOREMOST_ERROR_FILE_CLOSE       5



/* These are used when displaying how much of the file we've snarfed
   (Never let it be said we didn't plan for the future!)              */
#define UNITS_BYTES                     0
#define UNITS_KILOB                     1
#define UNITS_MEGAB                     2
#define UNITS_GIGAB                     3
#define UNITS_TERAB                     4
#define UNITS_PETAB                     5
#define UNITS_EXAB                      6


/* signal_caught can't go in the state because the signal handler 
   can't take arguments and can only modify global variables...      */
int signal_caught; 

/* Wildcard is a global variable because it's used by very simple
   functions that don't need the whole state passed to them */
char wildcard;

extern char *__progname;
extern int  errno;

#define FOREMOST_DEFAULT_WILDCARD       '?'
#define FOREMOST_DEFAULT_CONFIG_FILE    "foremost.conf"

#define FOREMOST_BANNER_STRING \
"foremost version %s\n"\
"Written by Kris Kendall and Jesse Kornblum.\n\n", FOREMOST_VERSION

#define FOREMOST_COPYRIGHT_STRING \
"Copyright (C) 2002 Air Force Office of Special Investigations\n"\
"This is free software; see the source for copying conditions. There is NO\n"\
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE\n"

#define FOREMOST_USAGE_STRING \
"Digs through an image file to find files within using header information."\
"\n\nUsage: %s [-h|V] [-qv] [-s num] [-i <file>] [-o <outputdir>] \\\n"\
"           [-c <config file>] <imgfile> [<imgfile>] ...\n\n"\
"-h  Print this help message and exit\n"\
"-V  Print copyright information and exit\n"\
"-v  Verbose mode. Highly recommended\n"\
"-q  Quick mode. Only searches the beginning of each sector. While\n"\
"      this is faster, you may miss some files. See man page for details.\n"\
"-i  Read names of files to dig from a file\n"\
"-o  Set output directory for dug files\n"\
"-c  Set configuration file to use. See man page for format.\n"\
"-s  Skip n bytes in the input file before digging\n", __progname


/* The fields for SearchSpecLine, CharBucket and foremostState 
   are in a specific order for the constructors in foremost.c. 
   Don't change them, or you'll break foremost.c! */

struct SearchSpecLine 
{
  char* suffix;
  int casesensitive;
  int length;
  char* begin;
  int beginlength;
  char* end;
  int endlength;
};

struct CharBucket
{
   long length;
   char *str;
};

struct foremostState {
  char* imagefile; 
  char* conffile; 
  char* outputdirectory;
  struct SearchSpecLine* SearchSpec;
  int fileswritten;
  int modeVerbose;
  FILE *auditFile;
  char* invocation;
  unsigned long long skip;
  int modeQuick;
};




struct NeedlesinHaystack_ret
{
  int needlenum;
  char* foundat;
};


/* These are functions defined in helpers.c */

void handleError(struct foremostState *s, int error);

int memwildcardcmp(const void* s1, const void* s2, 
		   size_t n, int caseSensitive);

int writeToDisk(char* suffix, 
		struct foremostState* state, 
		struct CharBucket* text,
		unsigned long long offset);

struct CharBucket* extractString(struct CharBucket* ,
				 unsigned long long,
				 FILE* ,struct SearchSpecLine);


int findLongestNeedle  (struct SearchSpecLine* SearchSpec);
int findShortestNeedle (struct SearchSpecLine* SearchSpec);
int findLongestLength  (struct SearchSpecLine* SearchSpec);

struct NeedlesinHaystack_ret* needlesinhaystack 
   (struct SearchSpecLine* needlearray, 
    char* haystack, 
    size_t haystack_len,
    int quick);

/* Return the size, in bytes of an open file stream and return the file
   position to the beginning. On error, return -1 */
unsigned long long measureOpenFile(FILE *f);
						
int openAuditFile(struct foremostState* state);
int closeFile(FILE* f);

int translate(char *str);
char *skipWhiteSpace(char *str);

/* Declare the functions we use from strings.h instead of implicitly 
   declaring them */

char *strsignal(int sig);

#endif   /* ifndef FOREMOST_H */






























