/* helpers.c
 *
 * (C) Copyright 2001 Air Force Office of Special Investigations 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "foremost.h"


int charactersMatch(char a, char b, int caseSensitive){
  if (a == wildcard) return 1;
  if (a == b) return 1;
  if (caseSensitive) return 0;
  if (a < 'A' || a > 'z' || b < 'A' || b > 'z') return 0;

  /*  This line is equivalent to (abs(a-b)) == 'a' - 'A' */
  if (abs(a-b) == 32) return 1;
  return 0;
}


/*
 memwildcardcmp is exactly like memcmp, except a wilcard 
 (usually '?' but can be redefined in the configuration file)  
 in s1 will match any single character in s2.
 We added it to support wildcards in the search specification.
 Be *very* careful about the order of s1 and s2... We want for a 
 wildcard in s1 to match anything, but not the other way around.
*/

int memwildcardcmp(const void *s1, const void *s2, 
		   size_t n,int caseSensitive){
  if (n!=0){
    register const unsigned char *p1 = s1, *p2 = s2;
    do{
      if(!charactersMatch(*p1++, *p2++, caseSensitive))
	return (*--p1 - *--p2);
    } while (--n !=0);
  }
  return(0);
}

char* needleinhaystack(char* needle, size_t needle_len, 
		       char* haystack, size_t haystack_len,
		       int caseSensitive){
  
  register char* begin;
  register char* last_possible = (char *) haystack + haystack_len - needle_len;
  
  if (needle_len == 0)
    return haystack;
  
  for (begin = (char*) haystack;
       begin <=last_possible;
       ++begin)
    if((begin[0] == needle[0] || needle[0] == wildcard || !caseSensitive) && 
       !memwildcardcmp(needle, begin, needle_len,caseSensitive))
      return begin;
  return NULL;
}

struct NeedlesinHaystack_ret* needlesinhaystack(struct SearchSpecLine* needlearray, 
						char* haystack, 
						size_t haystack_len,
						int quick){
  
  static struct NeedlesinHaystack_ret ret = {-1,NULL};
  
  register char* begin;
  register char* last_possible = (char *) haystack + haystack_len;
  int i=0,j=0,jumpforwardby=1;
  
  /* If we are doing a quick scan we only search sector boundaries */
  if (quick){
    jumpforwardby = 512;
  }
  
  ret.needlenum = -1;
  ret.foundat = NULL;
  
  for (begin = (char*) haystack;
       begin <= last_possible - jumpforwardby;
       begin += jumpforwardby){
    for(i=0;needlearray[i].suffix != NULL;i++){
      /* skip over any wildcard's at the beginning of the needle...  */
      j=0;
      while(needlearray[i].begin[j] == wildcard){
	j++;
      }
      if((begin[j] == needlearray[i].begin[j] || !needlearray[i].casesensitive) && 
	 last_possible-begin >= needlearray[i].beginlength && 
	 !memwildcardcmp(needlearray[i].begin+j,
			 begin+j, 
			 needlearray[i].beginlength-j,
			 needlearray[i].casesensitive)){
	ret.needlenum = i;
	ret.foundat = begin;
	return(&ret);
      }
    }
  }
  return &ret;
}



struct CharBucket* extractString (struct CharBucket* extractbuf,
				  unsigned long long  fileoffset,
				  FILE* infile,
				  struct SearchSpecLine needle) {

  unsigned long long currentfileposition = 0;
  char* foundat;
  unsigned long long bytesread = 0;

  /* Right now we just re-suck in then buffer from the file...
     It might be nice to only do this if we need to. */
  
  currentfileposition = ftello(infile);
  fseeko(infile,fileoffset,SEEK_SET);
  bytesread = fread(extractbuf->str,1,needle.length,infile);
  fseeko(infile,currentfileposition,SEEK_SET);
  
  extractbuf->length = (bytesread<needle.length)?bytesread:needle.length;

  /* Now the buffer has the region we want to search for the end marker in.
     If we don't have an endmarker we just want to return...
     else look for the endmarker.....  */

  if (needle.endlength > 0){
    
    if((foundat = (char*) needleinhaystack(needle.end,
					   needle.endlength,
					   extractbuf->str,
					   bytesread,
					   needle.casesensitive))){
      extractbuf->length = (foundat+needle.endlength - extractbuf->str);
      
    }
  }
  return(extractbuf);
}

int removeWhitespace(char* str){

  int i=0, numremoved = 0, length = strlen(str);
  char* curr=str;

  while(curr-str < length){

    /* Originally this line just checked for spaces and tabs.
       We'll be extra safe and check against all kinds of whitespace (JK) */
    if (isspace(*curr)) {
      
      for(i=0 ; i < length-(curr-str) ; i++) {
	*(curr+i) = *(curr+i+1);
      }
      length--;
      numremoved++;
    }
    curr++;
  }
  
  return numremoved;
}


/* These three functions could be combined into one to save some time.
   But because the time saved would be minimal (two loops through the
   SeachSpecLine array), it's not worth the added complication)   */

int findLongestNeedle(struct SearchSpecLine* SearchSpec){
  int longest = 0;
  int i=0;
  for(i=0; SearchSpec[i].suffix != NULL; i++)
    if(SearchSpec[i].beginlength > longest)
      longest = SearchSpec[i].beginlength;
  return(longest);  
}     

int findShortestNeedle(struct SearchSpecLine* SearchSpec){
  int i, shortest = 0;
  shortest = SearchSpec[0].beginlength;
  for(i=0; SearchSpec[i].suffix != NULL; i++)
    if(SearchSpec[i].beginlength < shortest)
      shortest = SearchSpec[i].beginlength;
  return(shortest);    
}     

int findLongestLength(struct SearchSpecLine* SearchSpec){
  int i, longest = 0;
  for(i=0; SearchSpec[i].suffix != NULL; i++)
    if(SearchSpec[i].length > longest)
      longest = SearchSpec[i].length;
  return(longest);    
}     


/* translate() decodes strings with escape sequences in them and
   returns the total length of the string it translated (we 
   can't use strlen because we would like to have null's in 
   our "strings") and has the side effect of truncating the string */
   
int translate(char *str) {
  char* temp = malloc(3*sizeof(char));
  char *here=str;
  size_t len=strlen(str);
  int num;
  int numlen;
  int finalsize=0;
  finalsize = strlen(str);
  while (NULL!=(here=strchr(here,'\\')))
    {
      finalsize--;
      numlen=1;
      switch (here[1])
	{
	case '\\':
	  here++;
	  break;
	  
	case 'r':
	  *here = '\r';
	  here++;
	  break;
	  
	case 'n':
	  *here = '\n';
	  here++;
	  break;
	  
	case 't':
	  *here = '\t';
	  here++;
	  break;
	  
	case 'v':
	  *here = '\v';
	  here++;
	  break;
	  
	case 'a':
	  *here = '\a';
	  here++;
	  break;
	  
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	  finalsize -= 2;
	  strncpy(temp,here+1,4);
	  numlen = sscanf(temp,"%o",&num);
	  if (numlen==1){
	    *here = (char)num;
	    numlen=3;
	    here++;
	  }
	  break;
	  
	case 'x':
	  finalsize -= 2;
	  strncpy(temp,here+2,3);
	  numlen = sscanf(temp,"%x",&num);
	  if (numlen==1){
	    *here = (char) num;
	    numlen=3;
	    here++;
	  }
	  break;
	}
      num = here - str + numlen;
      memmove(here,here+numlen,len-num+2);
    }
  free(temp);
  return finalsize;
}


char* skipWhiteSpace(char* str){
  while (isspace(str[0])) 
    str++;
  return str;
}



void handleError(struct foremostState *state, int error) {

  switch(error) {


  case FOREMOST_ERROR_FILE_OPEN:

    /* This is not a fatal error so we can return */
    fprintf (stderr, "Foremost was unable to open the image file: %s\nSkipping...\n\n",
	     state->imagefile);
    
    fprintf (state->auditFile, 
	     "Foremost was unable to open the image file: %s\nSkipping...\n\n",
	     state->imagefile);
    break;

  case FOREMOST_ERROR_FILE_READ:

    /* This is not a fatal error so we can return */
    fprintf (stderr, "Foremost was unable to read the image file\n: %s\nSkipping...\n\n",
	     state->imagefile);
    
    fprintf (state->auditFile, 
	     "Foremost was unable to read the image file\n: %s\nSkipping...\n\n",
	     state->imagefile);
    break;


  case FOREMOST_ERROR_FILE_WRITE:

    /* We can't write out files, which is very bad.
       If we can't party, we'll just go home. */



    fprintf (stderr, 
	     "Foremost was unable to write output files and will abort.\n");
    fprintf (state->auditFile, 
	     "Foremost was unable to write output files and will abort.\n");    
    closeFile(state->auditFile);
    exit (-1);

  default:

    fprintf (stderr, "Foremost has encountered an error it doesn't know how to handle.\nError code: %d\n", error);
    fprintf (state->auditFile, "Foremost has encountered an error it doesn't know how to handle.\nError code: %d\n", error);

    closeFile(state->auditFile);
    exit (-1);
  }
}



