/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- CVS Information ---
 *  $Id: AlternateLocation.java,v 1.12 2004/03/27 15:23:15 gregork Exp $
 */
package phex.common;

import java.net.*;
import java.util.*;


import phex.host.*;
import phex.utils.*;
import phex.security.*;


public class AlternateLocation
{
    /**
     * pushNeeded | serverBusy | I'm firewalled | useHasUploaded | Rating
     *         NO |    UNKNOWN |                |              y |      5
     *
     * hasUploaded == QHD_UNKNOWN_FLAG -> Raiting + 1
     */
    public static final Short DEFAULT_HOST_RATING = new Short( (short)6 );
    private HostAddress hostAddress;
    private URN urn;

    public AlternateLocation( HostAddress hostAddress, URN urn )
    {
        this.hostAddress = hostAddress;
        this.urn = urn;
    }
    
    /**
     * @return the host address of the AltLoc.
     */
    public HostAddress getHostAddress()
    {
        return hostAddress;
    }
    
    /**
     * Returns the urn of the AlternateLocation.
     * @return the urn of the AlternateLocation.
     */
    public URN getURN()
    {
        return urn;
    }

    /**
     * Returns the alternate location in http format for header values.
     * @return
     */
    public String getHTTPString()
    {
        StringBuffer buffer = new StringBuffer( hostAddress.getHostName() );
        int port = hostAddress.getPort();
        if ( port != 6346 )
        {
            buffer.append( ':' );
            buffer.append( port );
        }
        return buffer.toString();
    }

    /**
     * Indicates whether some other object is "equal to" this one.
     *
     * Note: The class AlternateLocationCollection relies on a equal implementation
     * that uses hostAddress and urn only!
     *
     * @param obj the reference object with which to compare.
     * @return <code>true</code> if this object is the same as the obj
     *         argument; <code>false</code> otherwise.
     */
    public boolean equals(Object obj)
    {
        if( !(obj instanceof AlternateLocation) )
        {
            return false;
        }

        if( obj == this )
        {
            return true;
        }
        AlternateLocation altLoc = (AlternateLocation)obj;
        return hostAddress.equals( altLoc.hostAddress ) &&
            urn.equals( altLoc.urn );
    }

    /**
     * Hash code to equal on hash based collections.
     * Note: The class AlternateLocationCollection relies on a hashCode implementation
     * that uses hostAddress and urn only!
     * @return a hash code value for this AlternateLocation.
     */
    public int hashCode()
    {
        int h = 7;
        h = ( (31 *h) + ( (hostAddress != null) ? hostAddress.hashCode() : 0 ) );
        h = ( (31 *h) + ( (urn != null) ? urn.hashCode() : 0 ) );
        return h;
    }
    
    /**
     * Parse possible alternate location values. Possible values are:
     * 10.0.0.82
     * 10.0.0.65:8765
     * @param line
     * @return the parsed AlternateLocation or null if parsing failes.
     */
    public static AlternateLocation parseCompactIpAltLoc( String line, URN urn )
    {
        byte[] ip = HostAddress.parseIP( line );
        int port = HostAddress.parsePort( line );
        // Limewire is not setting default port...
        if ( port == -1 )
        {
            port = 6346;
        }
        else if ( !HostAddress.isPortInRange( port ) )
        {
            Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                "Invalid alt-location URL (Port out of range): " + line );
            return null;
        }

        HostAddress hostAddress = new HostAddress( ip, port );
        
        try
        {
            byte access = PhexSecurityManager.getInstance().controlHostIPAccess(
                hostAddress.getHostIP() );
            switch ( access )
            {
                case PhexSecurityManager.ACCESS_DENIED:
                case PhexSecurityManager.ACCESS_STRONGLY_DENIED:
                    Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Alt-Location host denied: " + line );
                return null;
            }
        
            if ( !hostAddress.isValidIP() )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Invalid IP used): " + line );
                return null;
            }
            if ( hostAddress.isPrivateIP() )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Private IP used): " + line );
                return null;
            }
                
            AlternateLocation loc = new AlternateLocation( hostAddress, urn );
            return loc;
        }
        catch ( UnknownHostException exp )
        {
            Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                "Invalid alt-location URL (UnknownHost: " + exp.getMessage()
                + " ): " + line );
            return null;
        }
    }


    /**
     * Parse possible alternate location values. Possible values are:
     *
     * http://www.clip2.com/GnutellaProtocol04.pdf (not supported)
     * http://10.0.0.10:6346/get/2468/GnutellaProtocol04.pdf (not supported)
     * http://10.0.0.25:6346/uri-res/N2R?urn:sha1:PLSTHIPQGSSZTS5FJUPAKUZWUGYQYPFB
     *      2002-04-30T08:30Z
     * 
     * But we currently only support uri-res URLs and IPs since there are too
     * many fakes on the net.
     * 
     * Parses and validates the URL more strictly. We currently only like HTTP URLs,
     * (this might be a bit too strict but all right currently), with a valid port
     * and only none private addresses.
     *
     * @param line
     * @return the parsed AlternateLocation or null if parsing failes.
     */
    public static AlternateLocation parseUriResAltLoc( String line )
    {
        StringTokenizer tokenizer = new StringTokenizer( line, " \t\n\r\f\"" );

        String urlStr = null;
        if ( tokenizer.hasMoreTokens() )
        {
            urlStr = tokenizer.nextToken();
        }
        if ( tokenizer.hasMoreTokens() )
        {
            // skip date time string
            tokenizer.nextToken();
            //parseTimestamp( dateTimeStr );
        }
        try
        {
            URL url = new URL( urlStr );
            // verify for HTTP URL
            String protocol = url.getProtocol();
            if ( !"http".equals( protocol ) )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Not a http URL): " + urlStr );
                return null;
            }
            
            String host = url.getHost();
            int port = url.getPort();
            if ( port == -1 )
            {
                // Due to the wide confusion on how to handle Alt-Locs without 
                // port Phex will ignore all legacy Alt-Locs without a defined
                // port. Vendors seem not to use same default port (80 or 6346).
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid legacy alt-loc without specified port." );
                return null;
                
                // create new url with changed http port
                //port = 80;
            }
            else if ( !HostAddress.isPortInRange( port ) )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Port out of range): " + urlStr );
                return null;
            }
    
            HostAddress hostAddress = new HostAddress( host, port );
            byte access = PhexSecurityManager.getInstance().controlHostIPAccess(
                hostAddress.getHostIP() );
            switch ( access )
            {
                case PhexSecurityManager.ACCESS_DENIED:
                case PhexSecurityManager.ACCESS_STRONGLY_DENIED:
                    Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Alt-Location host denied: " + urlStr );
                return null;
            }
    
            if ( !hostAddress.isValidIP() )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Invalid IP used): " + urlStr );
                return null;
            }
            if ( hostAddress.isPrivateIP() )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Invalid alt-location URL (Private IP used): " + urlStr );
                return null;
            }
    
            URN urn = URN.parseURNFromUriRes( url.getFile() );
            if ( urn == null )
            {
                Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                    "Alt-location path without URN: " + line );
                return null;
            }
            
            AlternateLocation loc = new AlternateLocation( hostAddress, urn );
            return loc;
        }
        catch ( UnknownHostException exp )
        {
            Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                "Invalid alt-location URL (UnknownHost: " + exp.getMessage()
                + " ): " + urlStr );
            return null;
        }
        catch ( MalformedURLException exp )
        {
            Logger.logMessage( Logger.FINE, Logger.DOWNLOAD,
                "Invalid alt-location URL (Malformed: " + exp.getMessage()
                + " ): " + urlStr );
            return null;
        }

    }
}