/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.common;

import phex.xml.*;

public class MediaType
{
    public static final String MEDIA_TYPE_ANY = "AnyMediaType";
    public static final String MEDIA_TYPE_AUDIO = "AudioMediaType";
    public static final String MEDIA_TYPE_VIDEO = "VideoMediaType";
    public static final String MEDIA_TYPE_PROGRAM = "ProgramMediaType";
    public static final String MEDIA_TYPE_IMAGES = "ImagesMediaType";
    public static final String MEDIA_TYPE_DOCUMENTS = "DocumentsMediaType";

    private static MediaType[] allMediaTypes;

    static
    {
        allMediaTypes = new MediaType[ 6 ];
        // the any media type must be on index 0 for method getMediaTypeAny()
        allMediaTypes[ 0 ] = new MediaType( MEDIA_TYPE_ANY, null );
        allMediaTypes[ 1 ] = new MediaType( MEDIA_TYPE_AUDIO,
            new String[] { "aif", "aifc", "aiff", "ape", "apl", "au", "iso", "lqt",
                "mac", "med", "mid", "midi", "mod", "mp3", "mpa", "mpga", "mp1",
                "ogg", "ra", "ram", "rm", "rmi", "rmj", "snd", "vqf", 
                "wav", "wma" } );
                
        allMediaTypes[ 2 ] = new MediaType( MEDIA_TYPE_VIDEO,
            new String[] { "asf", "avi", "dcr", "div", "divx", "dv", "dvd",
                "dvx", "flc", "fli", "flx", "jve", "m2p", "m2v", "m1v", "mng",
                "mov", "mp2", "mp2v", "mp4", "mpe", "mpeg", "mpg", "mpv", "mpv2",
                "nsv", "ram", "rm", "rv", "smi", "smil", "swf", "qt", "vcd",
                "vob", "vrml", "wml", "wmv" } );
 
        allMediaTypes[ 3 ] = new MediaType( MEDIA_TYPE_PROGRAM,
            new String[] { "7z", "ace", "arj", "awk", "bin", "bz2", "cab", "csh",
                "deb", "dmg", "exe", "gz", "gzip", "hqx", "iso", "jar", "lzh", "lha",
                "mdb", "msi", "msp", "pl", "rar", "rpm", "sh", "shar", "sit",
                "tar", "tgz", "taz", "z", "zip", "zoo" } );
                
        allMediaTypes[ 4 ] = new MediaType( MEDIA_TYPE_IMAGES,
            new String[] { "ani", "bmp", "cpt", "cur", "dcx", "dib", "drw",
                "emf", "fax", "gif", "icl", "ico", "iff", "ilbm", "img", "jif",
                "jiff", "jpe", "jpeg", "jpg", "lbm", "mac", "mic", "pbm", "pcd",
                "pct", "pcx", "pic", "png", "pnm", "ppm", "psd", "ras", "rgb", "rle",
                "sgi", "sxd", "svg", "tga", "tif", "tiff", "wmf", "wpg", "xbm", "xcf", "xpm", "xwd" } );
                
        allMediaTypes[ 5 ] = new MediaType( MEDIA_TYPE_DOCUMENTS,
            new String[] { "ans", "asc", "chm", "csv", "dif", "diz", "doc", "eml",
                "eps", "epsf", "hlp", "html", "htm", "info", "latex", "man", "mcw",
                "mht", "mhtml", "pdf", "ppt", "ps", "rtf", "sxw", "sxc", "tex", "texi", "txt", "wk1", 
                "wps", "wri", "xhtml", "xls", "xml" } );
    }

    private String name;
    private String[] fileTypes;
    private String fileTypesView;

    private MediaType( String aName, String[] fileTypeArray )
    {
        name = aName;
        fileTypes = fileTypeArray;
    }

    public String getName()
    {
        return name;
    }

    public String getFileTypesUIText()
    {
        if ( fileTypesView == null )
        {
            if ( fileTypes == null )
            {
                return "";
            }
            StringBuffer buffer = new StringBuffer( fileTypes.length * 5 );
            buffer.append( "<html>" );
            int charCount = 0;
            for ( int i = 0; i < fileTypes.length; i++ )
            {
                buffer.append( fileTypes[i] );
                charCount += fileTypes[i].length();
                if ( i < fileTypes.length - 1 )
                {
                    buffer.append( ", " );
                    charCount ++;
                }
                if ( charCount > 50 )
                {
                    charCount = 0;
                    buffer.append ( "<br>" );
                }
            }
            fileTypesView = buffer.toString();
        }
        return fileTypesView;
    }

    /**
     * Verifys that the extension is a extension of this media type.
     * Returnes true if the given extension is a extension of this media type.
     */
    public boolean isExtensionOf( String extension )
    {
        // this is for any media type
        if ( fileTypes == null )
        {
            return true;
        }
        extension = extension.toLowerCase();
        for ( int i = 0; i < fileTypes.length; i++ )
        {
            if ( fileTypes[i].equals( extension ) )
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Verifys that the extension of the given filename is a extension of this
     * media type.
     * Returnes true if the given filename extension is a extension of this media type.
     */
    public boolean isFilenameOf( String filename )
    {
        // this is for any media type
        if ( fileTypes == null )
        {
            return true;
        }

        int index = filename.lastIndexOf(".");
        // if no '.' or index is last char of the file name return false
        if (index == -1 || index == filename.length() )
        {
            return false;
        }
        String extension = filename.substring( index + 1, filename.length() );
        return isExtensionOf( extension );
    }

    public static MediaType[] getAllMediaTypes()
    {
        return allMediaTypes;
    }

    public static MediaType getMediaTypeAny()
    {
        // first element in all array...
        return allMediaTypes[0];
    }

    public static XJBMediaType convertToXJBMediaType( MediaType type )
    {
        if ( type.name.equals( MEDIA_TYPE_ANY ) )
        {
            return XJBMediaType.ANY;
        }
        else if ( type.name.equals( MEDIA_TYPE_AUDIO ) )
        {
            return XJBMediaType.AUDIO;
        }
        else if ( type.name.equals( MEDIA_TYPE_VIDEO ) )
        {
            return XJBMediaType.VIDEO;
        }
        else if ( type.name.equals( MEDIA_TYPE_PROGRAM ) )
        {
            return XJBMediaType.PROGRAM;
        }
        else
        {
            throw new IllegalArgumentException( "Unknown media type: " +
                type.name );
        }
    }

    public static MediaType convertFromXJBMediaType( XJBMediaType xjbType )
    {
        if ( xjbType == XJBMediaType.ANY )
        {
            return allMediaTypes[0];
        }
        else if ( xjbType == XJBMediaType.AUDIO )
        {
            return allMediaTypes[1];
        }
        else if ( xjbType == XJBMediaType.VIDEO )
        {
            return allMediaTypes[2];
        }
        else if ( xjbType == XJBMediaType.PROGRAM )
        {
            return allMediaTypes[3];
        }
        else
        {
            throw new IllegalArgumentException( "Unknown media type: " +
                xjbType.toString() );
        }
    }
}
