/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.download;

import java.net.Socket;
import java.util.ArrayList;
import java.util.Iterator;

import phex.msg.GUID;
import phex.utils.Logger;

public class PushHandler
{
    /**
     * This is a stand alone global class responsible for push handling.
     */
    private static PushHandler singleton = new PushHandler();

    /**
     * A list is used instead of a map since this will never contain many
     * entrys. And it's hard to create a usefull key since the file index might
     * change.
     */
    private ArrayList pushSleeperList;

    private PushHandler()
    {
        pushSleeperList = new ArrayList(5);
    }

    public static void handleIncommingGIV(Socket aGivenSocket, GUID givenGUID,
        String givenFileName)
    {
        singleton.internalHandleIncommingGIV(aGivenSocket, givenGUID,
            givenFileName);
    }

    public static Socket requestSocketViaPush(
        IDownloadCandidate downloadCandidate)
    {
        if ( downloadCandidate.getGUID() == null ) { return null; }
        return singleton.internalRequestSocketViaPush(downloadCandidate
            .getGUID(), downloadCandidate.getFileIndex(), downloadCandidate
            .getFileName());
    }

    /**
     *
     * @param aClientGUID
     * @param aFileIndex
     * @param aFileName
     * @return Returns null if push request failes.
     */
    public static Socket requestSocketViaPush(GUID aClientGUID,
        long aFileIndex, String aFileName)
    {
        return singleton.internalRequestSocketViaPush(aClientGUID, aFileIndex,
            aFileName);
    }

    public static void unregisterPushRequestSleeper(PushRequestSleeper sleeper)
    {
        singleton.internalUnregisterPushRequestSleeper(sleeper);
    }

    private void internalHandleIncommingGIV(Socket aGivenSocket,
        GUID givenGUID, String givenFileName)
    {
        Logger.logMessage(Logger.FINER, Logger.DOWNLOAD,
            "Handle incomming GIV response: " + " - " + givenFileName);
        synchronized (pushSleeperList)
        {
            Iterator iterator = pushSleeperList.iterator();
            while (iterator.hasNext())
            {
                boolean succ = ((PushRequestSleeper) iterator.next())
                    .acceptGIVConnection(aGivenSocket, givenGUID);
                if ( succ )
                {
                    Logger.logMessage(Logger.FINER, Logger.DOWNLOAD,
                        "Accepted GIV response: " + " - " + givenFileName);
                    return;
                }
            }
        }
        Logger.logMessage(Logger.FINER, Logger.DOWNLOAD,
            "No Push request for GIV found: " + givenFileName);
    }

    private Socket internalRequestSocketViaPush(GUID aClientGUID,
        long aFileIndex, String aFileName)
    {
        PushRequestSleeper pushSleeper = new PushRequestSleeper(aClientGUID,
            aFileIndex);
        synchronized (pushSleeperList)
        {
            pushSleeperList.add(pushSleeper);
        }
        Socket socket = pushSleeper.requestSocketViaPush();
        return socket;
    }

    private void internalUnregisterPushRequestSleeper(PushRequestSleeper sleeper)
    {
        synchronized (pushSleeperList)
        {
            pushSleeperList.remove(sleeper);
        }
    }
}