/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.common;

import java.awt.*;

import javax.swing.*;

import phex.download.swarming.*;
import phex.event.*;
import phex.host.*;
import phex.statistic.*;
import phex.upload.UploadManager;
import phex.utils.Localizer;


public class StatusBar extends JPanel
{
    private NetworkHostsContainer hostsContainer;
    private SwarmingManager swarmingMgr;
    private UploadManager uploadMgr;

    private JLabel connectionLabel;
    private JLabel downloadLabel;
    private JLabel uploadLabel;
    private Icon connectedIcon;
    private Icon disconnectedIcon;

    public StatusBar()
    {
        super( new GridBagLayout() );
        setBorder( BorderFactory.createLoweredBevelBorder() );

        connectionLabel = new JLabel();
            GridBagConstraints constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.SOUTHWEST;
            constraints.insets = new Insets( 0, 2, 0, 2 );
            constraints.weightx = 0;
            constraints.weighty = 1;
        add( connectionLabel, constraints );

        downloadLabel = new JLabel();
        downloadLabel.setIcon(GUIRegistry.getInstance().getIconFactory()
            .getIcon("DownloadSmall"));
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.SOUTHWEST;
            constraints.insets = new Insets( 0, 2, 0, 2 );
            constraints.weightx = 0;
            constraints.weighty = 1;
        add( downloadLabel, constraints );

        uploadLabel = new JLabel();
        uploadLabel.setIcon(GUIRegistry.getInstance().getIconFactory()
            .getIcon("UploadSmall"));
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.SOUTHWEST;
            constraints.insets = new Insets( 0, 2, 0, 2 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        add( uploadLabel, constraints );

        IconFactory factory = GUIRegistry.getInstance().getIconFactory();
        connectedIcon = factory.getIcon( "Network" );
        disconnectedIcon = factory.getIcon( "Disconnect" );

        HostManager hostMgr = HostManager.getInstance();
        hostsContainer = hostMgr.getNetworkHostsContainer();
        swarmingMgr = SwarmingManager.getInstance();
        uploadMgr = UploadManager.getInstance();

        updateStatusBar();

        StatusChangedListener statusListener = new StatusChangedListener();

        hostsContainer.addNetworkHostsChangeListener( statusListener );
        uploadMgr.addUploadFilesChangeListener( statusListener );
        swarmingMgr.addDownloadFilesChangeListener( statusListener );
    }

    private void updateStatusBar()
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                int hostCount = hostsContainer.getTotalConnectionCount();
                if ( hostCount > 0 )
                {
                    connectionLabel.setIcon( connectedIcon );
                }
                else
                {
                    connectionLabel.setIcon( disconnectedIcon );
                }

                Object[] args = new Object[]
                {
                    new Integer( hostCount )
                };
                String text = Localizer.getFormatedString( "StatusBarConnections", args );
                connectionLabel.setText( text );
                connectionLabel.setToolTipText( Localizer.getString( "TTTStatusBarConnections" ) );

                setPreferredSize( new Dimension( connectionLabel.getPreferredSize().width,
                    connectionLabel.getPreferredSize().height + 6 ) );

                args = new Object[]
                {
                    new Integer( swarmingMgr.getDownloadFileCount(
                        SWDownloadConstants.STATUS_FILE_DOWNLOADING ) ),
                    new Integer( swarmingMgr.getDownloadFileCount() ),
                    new Integer( swarmingMgr.getDownloadFileCount(
                        SWDownloadConstants.STATUS_FILE_COMPLETED ) )
                };
                text = Localizer.getFormatedString( "StatusBarDownloads", args );
                downloadLabel.setText( text );
                downloadLabel.setToolTipText( Localizer.getString( "TTTStatusBarDownloads" ) );

                setPreferredSize( new Dimension( downloadLabel.getPreferredSize().width,
                    downloadLabel.getPreferredSize().height + 6 ) );

                StatisticsManager statsMgr = StatisticsManager.getInstance();
                StatisticProvider uploadCountProvider = statsMgr.getStatisticProvider(
                    StatisticProviderConstants.SESSION_UPLOAD_COUNT_PROVIDER );
                args = new Object[]
                {
                    new Integer( uploadMgr.getUploadingCount() ),
                    new Integer( uploadMgr.getUploadQueueSize() ),
                    uploadCountProvider.getValue()
                };
                text = Localizer.getFormatedString( "StatusBarUploads", args );
                uploadLabel.setText( text );
                uploadLabel.setToolTipText( Localizer.getString( "TTTStatusBarUploads" ) );

                setPreferredSize( new Dimension( uploadLabel.getPreferredSize().width,
                    uploadLabel.getPreferredSize().height + 6 ) );
            }
        });
    }

    private class StatusChangedListener implements
        DownloadFilesChangeListener, UploadFilesChangeListener,
        NetworkHostsChangeListener
    {
        ////////////////////// End IDataChangedListener ///////////////

        ////////////////////// Start NetworkHostsChangeListener ///////////////
        public void networkHostChanged( int position )
        {
            updateStatusBar();
        }

        public void networkHostAdded( int position )
        {
            updateStatusBar();
        }

        public void networkHostRemoved( int position )
        {
            updateStatusBar();
        }
        ////////////////////// End NetworkHostsChangeListener ///////////////

        ////////////////////// Start UploadFilesChangeListener ///////////////
        public void uploadFileChanged( final int position )
        {
            updateStatusBar();
        }

        public void uploadFileAdded( final int position )
        {
            updateStatusBar();
        }

        /**
         * Called if the upload queue was changed.
         */
        public void uploadQueueChanged()
        {
            updateStatusBar();
        }

        public void uploadFileRemoved( final int position )
        {
            updateStatusBar();
        }
        ////////////////////// End UploadFilesChangeListener ///////////////

        ////////////////////// Start DownloadFilesChangeListener ///////////////
        /**
         * Called if a download file changed.
         */
        public void downloadFileChanged( int position )
        {
            updateStatusBar();
        }

        /**
         * Called if a download file was added.
         */
        public void downloadFileAdded( int position )
        {
            updateStatusBar();
        }

        /**
         * Called if a download file was removed.
         */
        public void downloadFileRemoved( int position )
        {
            updateStatusBar();
        }
        ////////////////////// End DownloadFilesChangeListener ///////////////
    }
}