/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- CVS Information ---
 *  $Id: SharedFilesTableModel.java,v 1.15 2004/08/04 10:13:57 gregork Exp $
 */
package phex.gui.models;

import java.util.Comparator;

import javax.swing.event.*;

import phex.event.*;
import phex.gui.common.*;
import phex.gui.common.table.*;
import phex.gui.common.table.FWSortedTableModel;
import phex.share.*;
import phex.utils.*;
import phex.gui.renderer.*;

public class SharedFilesTableModel extends FWSortableTableModel
{
    public static final int FILE_COLUMN_ID = 1001;
    public static final int DIRECTORY_COLUMN_ID = 1002;
    public static final int SIZE_COLUMN_ID = 1003;
    public static final int SEARCH_COUNT_COLUMN_ID = 1004;
    public static final int UPLOAD_COUNT_COLUMN_ID = 1005;
    public static final int SHA1_COLUMN_ID = 1006;
	public static final int THEX_COLUMN_ID = 1007;

    public static final int FILE_MODEL_INDEX = 0;
    public static final int DIRECTORY_MODEL_INDEX = 1;
    public static final int SIZE_MODEL_INDEX = 2;
    public static final int SEARCH_COUNT_MODEL_INDEX = 3;
    public static final int UPLOAD_COUNT_MODEL_INDEX = 4;
    public static final int SHA1_MODEL_INDEX = 5;
	public static final int THEX_MODEL_INDEX = 6;

    private static String[] tableColumns;
    private static Class[] tableClasses;

    static
    {
        tableColumns = new String[]
        {
            Localizer.getString( "File" ),
            Localizer.getString( "Directory" ),
            Localizer.getString( "Size" ),
            Localizer.getString( "SearchCount" ),
            Localizer.getString( "UploadCount" ),
            Localizer.getString( "SHA1" )
            // TODOTHEX
			//Localizer.getString( "THEX" )
        };

        tableClasses = new Class[]
        {
            String.class,
            String.class,
            FileSizeCellRenderer.class,
            Integer.class,
            Integer.class,
            String.class
            //TODOTHEX
            //String.class
        };
    }

    private FileAdministration sharedFileAdmin;

    public SharedFilesTableModel()
    {
        super( tableColumns, tableClasses );
        sharedFileAdmin = ShareManager.getInstance().getFileAdministration();
        sharedFileAdmin.addSharedFilesChangeListener( new SharedFilesListener() );
    }

    public int getRowCount()
    {
        return sharedFileAdmin.getFileCount();
    }

    public Object getValueAt(int row, int col)
    {
        ShareFile shareFile = sharedFileAdmin.getFileAt( row );
        if ( shareFile == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }
        switch ( col )
        {
            case FILE_MODEL_INDEX:
                return shareFile.getFileName();

            case DIRECTORY_MODEL_INDEX:
                return shareFile.getFile().getParent();

            case SIZE_MODEL_INDEX:
                return shareFile.getFileSizeObject();

            case SEARCH_COUNT_MODEL_INDEX:
                return new Integer( shareFile.getSearchCount() );

            case UPLOAD_COUNT_MODEL_INDEX:
                return new Integer( shareFile.getUploadCount() );
            case SHA1_MODEL_INDEX:
                return shareFile.getSHA1();
            // TODOTHEX
			//case THEX_MODEL_INDEX:
			//	return shareFile.getRootTigerTree();
        }

        return "";
    }
    
    /**
     * Returns the most comparator that is used for sorting of the cell values
     * in the column. This is used by the FWSortedTableModel to perform the
     * sorting. If not overwritten the method returns null causing the
     * FWSortedTableModel to use a NaturalComparator. It expects all Objects that
     * are returned from getComparableValueAt() to implement the Comparable interface.
     *
     */
    public Comparator getColumnComparator( int column )
    {
        switch ( column )
        {
            case SIZE_MODEL_INDEX:
                return FWSortedTableModel.COMPARABLE_COMAPRATOR;
        }
        return null;
    }

    /**
     * Maps the unique column id to the model index. This needs to be done to
     * be able identify columns and there index after changes in Phex releases.
     * The unique column id is not allowed to ever change over Phex releases. It
     * is used when serializing column information. The column id is containd in
     * the identifier field of the TableColumn.
     */
    public int mapColumnIDToModelIndex( int columnId )
    {
        switch( columnId )
        {
            case FILE_COLUMN_ID:
                return FILE_MODEL_INDEX;
            case DIRECTORY_COLUMN_ID:
                return DIRECTORY_MODEL_INDEX;
            case SIZE_COLUMN_ID:
                return SIZE_MODEL_INDEX;
            case SEARCH_COUNT_COLUMN_ID:
                return SEARCH_COUNT_MODEL_INDEX;
            case UPLOAD_COUNT_COLUMN_ID:
                return UPLOAD_COUNT_MODEL_INDEX;
            case SHA1_COLUMN_ID:
                return SHA1_MODEL_INDEX;
            // TODOTHEX    
			//case THEX_COLUMN_ID:
			//	return THEX_MODEL_INDEX;
            default:
                return -1;
        }
    }

    /**
     * Indicates if a column is hideable.
     */
    public boolean isColumnHideable( int columnID )
    {
        if ( columnID == FILE_COLUMN_ID )
        {
            return false;
        }
        return true;
    }

    public static int[] getColumnIdArray()
    {
        int[] columnIds = new int[]
        {
            FILE_COLUMN_ID,
            DIRECTORY_COLUMN_ID,
            SIZE_COLUMN_ID,
            SEARCH_COUNT_COLUMN_ID,
            UPLOAD_COUNT_COLUMN_ID,
            SHA1_COLUMN_ID
            // TODOTHEX
			//THEX_COLUMN_ID
        };
        return columnIds;
    }

    private class SharedFilesListener implements SharedFilesChangeListener
    {
        private LazyEventQueue lazyEventQueue;

        public SharedFilesListener()
        {
            lazyEventQueue = GUIRegistry.getInstance().getLazyEventQueue();
        }

        /**
         * Called if a shared file changed.
         */
        public void sharedFileChanged( int position )
        {
            fireTableCellUpdated( position, position );
        }

        /**
         * Called if a shared file was added.
         */
        public void sharedFileAdded( int position )
        {
            lazyEventQueue.addTableModelEvent(
                new TableModelEvent( SharedFilesTableModel.this,
                    position, position, TableModelEvent.ALL_COLUMNS,
                    TableModelEvent.INSERT ) );
        }

        /**
         * Called if a shared file was removed.
         */
        public void sharedFileRemoved( int position )
        {
            lazyEventQueue.addTableModelEvent(
                new TableModelEvent( SharedFilesTableModel.this,
                    position, position, TableModelEvent.ALL_COLUMNS,
                    TableModelEvent.DELETE ) );
        }

        /**
         * Called if all shared files changed.
         */
        public void allSharedFilesChanged( )
        {
            fireTableDataChanged();
        }
    }
}