/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.models;

import java.util.Comparator;

import javax.swing.event.TableModelEvent;

import phex.event.UploadFilesChangeListener;
import phex.gui.common.*;
import phex.gui.common.table.*;
import phex.gui.comparator.*;
import phex.gui.renderer.*;
import phex.upload.*;
import phex.utils.Localizer;

public class UploadFilesTableModel extends FWSortableTableModel
{
    /**
     * The unique column id is not allowed to ever change over Phex releases. It
     * is used when serializing column information. The column id is containd in
     * the identifier field of the TableColumn.
     */
    public static final int HOST_COLUMN_ID = 1001;
    public static final int VENDOR_COLUMN_ID = 1002;
    public static final int FILE_COLUMN_ID = 1003;
    public static final int PROGRESS_COLUMN_ID = 1004;
    public static final int SIZE_COLUMN_ID = 1005;
    public static final int RATE_COLUMN_ID = 1006;
    public static final int STATUS_COLUMN_ID = 1007;

    public static final int HOST_MODEL_INDEX = 0;
    public static final int VENDOR_MODEL_INDEX = 1;
    public static final int FILE_MODEL_INDEX = 2;
    public static final int PROGRESS_MODEL_INDEX = 3;
    public static final int SIZE_MODEL_INDEX = 4;
    public static final int RATE_MODEL_INDEX = 5;
    public static final int STATUS_MODEL_INDEX = 6;

    private static String[] tableColumns;
    private static Class[] tableClasses;

    static
    {
        tableColumns = new String[]
        {
            Localizer.getString( "Host" ),
            Localizer.getString( "Vendor" ),
            Localizer.getString( "File" ),
            Localizer.getString( "PercentSign" ),
            Localizer.getString( "Size" ),
            Localizer.getString( "Rate" ),
            Localizer.getString( "Status" )
        };

        tableClasses = new Class[]
        {
            HostAddressCellRenderer.class,
            String.class,
            String.class,
            ProgressCellRenderer.class,
            TransferSizeCellRenderer.class,
            TransferRateCellRenderer.class,
            String.class
        };
    }

    private UploadManager uploadMgr;

    public UploadFilesTableModel()
    {
        super( tableColumns, tableClasses );
        uploadMgr = UploadManager.getInstance();
        uploadMgr.addUploadFilesChangeListener( new UploadFilesListener() );
    }

    public int getRowCount()
    {
        return uploadMgr.getUploadListSize();
    }

    public Object getValueAt( int row, int col )
    {
        UploadState uploadState = uploadMgr.getUploadStateAt( row );
        if ( uploadState == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }

        switch ( col )
        {
            case HOST_MODEL_INDEX:
                return uploadState.getHostAddress();

            case VENDOR_MODEL_INDEX:
                String vendor = uploadState.getVendor();
                if ( vendor == null )
                {
                    return "";
                }
                else
                {
                    return vendor;
                }

            case FILE_MODEL_INDEX:
                return uploadState.getFileName();

            case PROGRESS_MODEL_INDEX:
                return uploadState.getProgress();

            case SIZE_MODEL_INDEX:
            case RATE_MODEL_INDEX:
                return uploadState;

            case STATUS_MODEL_INDEX:
                return UploadStatusInfo.getUploadStatusString( uploadState.getStatus() );
        }
        return "";
    }

    /**
     * Returns the most comparator that is used for sorting of the cell values
     * in the column. This is used by the FWSortedTableModel to perform the
     * sorting. If not overwritten the method returns null causing the
     * FWSortedTableModel to use a NaturalComparator. It expects all Objects that
     * are returned from getComparableValueAt() to implement the Comparable interface.
     *
     */
    public Comparator getColumnComparator( int column )
    {
        switch( column )
        {
            case HOST_MODEL_INDEX:
                return new HostAddressComparator();
            case SIZE_MODEL_INDEX:
                return new TransferSizeComparator();
            case RATE_MODEL_INDEX:
                return new TransferRateComparator();
            // for all other columns use default comparator
            default:
                return null;
        }
    }

    /**
     * Maps the unique column id to the model index. This needs to be done to
     * be able identify columns and there index after changes in Phex releases.
     * The unique column id is not allowed to ever change over Phex releases. It
     * is used when serializing column information. The column id is containd in
     * the identifier field of the TableColumn.
     */
    public int mapColumnIDToModelIndex( int columnId )
    {
        switch( columnId )
        {
            case HOST_COLUMN_ID:
                return HOST_MODEL_INDEX;
            case VENDOR_COLUMN_ID:
                return VENDOR_MODEL_INDEX;
            case FILE_COLUMN_ID:
                return FILE_MODEL_INDEX;
            case PROGRESS_COLUMN_ID:
                return PROGRESS_MODEL_INDEX;
            case SIZE_COLUMN_ID:
                return SIZE_MODEL_INDEX;
            case RATE_COLUMN_ID:
                return RATE_MODEL_INDEX;
            case STATUS_COLUMN_ID:
                return STATUS_MODEL_INDEX;
            default:
                return -1;
        }
    }

    /**
     * Indicates if a column is hideable.
     */
    public boolean isColumnHideable( int columnID )
    {
        if ( columnID == FILE_COLUMN_ID )
        {
            return false;
        }
        return true;
    }

    private class UploadFilesListener
        implements UploadFilesChangeListener
    {
        private LazyEventQueue lazyEventQueue;

        public UploadFilesListener()
        {
            lazyEventQueue = GUIRegistry.getInstance().getLazyEventQueue();
        }

        public void uploadFileChanged( final int position )
        {
            lazyEventQueue.addTableModelEvent(
                new TableModelEvent( UploadFilesTableModel.this, position, position,
                    TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE ) );
        }

        public void uploadFileAdded( final int position )
        {
            fireTableChanged(
                new TableModelEvent(UploadFilesTableModel.this, position, position,
                    TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT ) );
        }

        /**
         * Called if the upload queue was changed.
         */
        public void uploadQueueChanged()
        {
        }

        public void uploadFileRemoved( final int position )
        {
            fireTableChanged(
                new TableModelEvent(UploadFilesTableModel.this, position, position,
                    TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE ) );
        }
    }
}