/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.msg;

import java.io.IOException;

import phex.msg.vendor.VendorMsg;
import phex.utils.*;


public class MessageProcessor
{
    public static Message parseMessage( GnutellaInputStream inStream )
        throws IOException, InvalidMessageException
    {
        MsgHeader header = parseMessageHeader( inStream, new byte[ MsgHeader.DATA_LENGTH ] );
        if ( header == null )
        {
            throw new IOException("Connection closed by remote host");
        }
        return parseMessage( header, inStream );
    }

    public static Message parseMessage( MsgHeader header, GnutellaInputStream inStream )
        throws IOException, InvalidMessageException
    {
        byte[] body = readMessageBody( inStream, header.getDataLen() );

        Message message = createMessageFromBody( header, body );

        return message;
    }

    public static Message createMessageFromBody( MsgHeader header, byte[] body )
        throws InvalidMessageException
    {
        switch( header.getPayload() )
        {
            case MsgHeader.PING_PAYLOAD:
                return new MsgPing( header, body );
            case MsgHeader.PONG_PAYLOAD:
                return new MsgPong( header, body );
            case MsgHeader.PUSH_PAYLOAD:
                return new MsgPushRequest( header, body );
            case MsgHeader.QUERY_HIT_PAYLOAD:
                return new MsgQueryResponse( header, body );
            case MsgHeader.QUERY_PAYLOAD:
                return new MsgQuery( header, body );
            case MsgHeader.ROUTE_TABLE_UPDATE_PAYLOAD:
                return RouteTableUpdateMsg.parseMessage( header, body );
            case MsgHeader.VENDOR_MESSAGE_PAYLOAD:
            case MsgHeader.STANDARD_VENDOR_MESSAGE_PAYLOAD:
                return VendorMsg.parseMessage( header, body );
        }
        return null;
    }


    public static byte[] readMessageBody( GnutellaInputStream inStream, int dataLength )
        throws IOException
    {
        byte[] body = new byte[ dataLength ];

        int dataRead = 0;
        int len;
        int readSize;
        while ( dataRead < dataLength )
        {
            readSize = Math.min( dataLength - dataRead, 1024 );
            len = inStream.read( body, dataRead, readSize );
            if ( len == -1 )
            {
                throw new IOException( "Connection closed by remote host" );
            }
            dataRead += len;
        }

        return body;
    }

    public static MsgHeader parseMessageHeader( GnutellaInputStream inStream,
        byte[] buffer )
        throws IOException
    {
        int lenRead = 0;
        int len;
        while ( lenRead < MsgHeader.DATA_LENGTH )
        {
            len = inStream.read( buffer, lenRead, MsgHeader.DATA_LENGTH - lenRead);
            if ( len == -1 )
            {
                return null;
            }
            lenRead += len;
        }

        byte[] guidArr = new byte[ GUID.DATA_LENGTH ];
        System.arraycopy( buffer, 0, guidArr, 0, GUID.DATA_LENGTH );

        byte payload = buffer[ 16 ];

        byte ttl = buffer[ 17 ];

        byte hops = buffer[ 18 ];

        int dataLength = IOUtil.deserializeIntLE( buffer, 19 );

        MsgHeader header = new MsgHeader( new GUID( guidArr ), payload, ttl, hops,
            dataLength );

        return header;
    }
}