/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *                2001 Peter Hunnisett (hunnise@yahoo.com)
 *                2000 William W. Wong (williamw@jps.net)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.utils;

import java.text.*;
import java.util.*;

import phex.common.Cfg;

/**
* A collection of custom utilities to modify or specially display Strings.
**/

public final class StrUtil
{
    public static final String FILE_DELIMITERS = " -._+/*()[]\\";

    /**
     * Represents 1 Kilo Byte ( 1024 ).
     */
    public final static long ONE_KB = 1024L;

    /**
     * Represents 1 Mega Byte ( 1024^2 ).
     */
    public final static long ONE_MB = ONE_KB * 1024L;

    /**
     * Represents 1 Giga Byte ( 1024^3 ).
     */
    public final static long ONE_GB = ONE_MB * 1024L;

    /**
     * Represents 1 Tera Byte ( 1024^4 ).
     */
    public final static long ONE_TB = ONE_GB * 1024L;

    public static DecimalFormat byteFormat = new DecimalFormat( "0.00" );
    public static NumberFormat FILE_LENGTH_FORMAT = NumberFormat.getInstance();

    /**
     * Print only the most significant portion of the time. This is
     * the two most significant units of time. Form will be something
     * like "3h 26m" indicating 3 hours 26 minutes and some insignificant
     * number of seconds.
     */
    public static String formatSignificantElapsedTime( long seconds )
    {
        final long days = seconds / 86400;
        if ( days > 0 ) // Display days and hours
        {
            Object[] args = new Object[]
            {
                new Long( days ),
                new Integer( (int)((seconds / 3600) % 24) ) // hours
            };
            return Localizer.getFormatedString( "TimeFormatDH", args );
        }

        final int hours = (int)((seconds / 3600) % 24);
        if( hours > 0 ) // Display hours and minutes
        {
            Object[] args = new Object[]
            {
                new Integer( hours ),
                new Integer( (int)((seconds / 60) % 60) ) // minutes
            };
            return Localizer.getFormatedString( "TimeFormatHM", args );
        }

        final int minutes =  (int)((seconds / 60) % 60);
        if( minutes > 0 ) // Display minutes and seconds
        {
            Object[] args = new Object[]
            {
                new Integer( minutes ),
                new Integer( (int)(seconds % 60) ) // seconds
            };
            return Localizer.getFormatedString( "TimeFormatMS", args );
        }

        final int secs = (int)(seconds % 60);
        Object[] args = new Object[]
        {
            new Integer( secs ),
        };
        return Localizer.getFormatedString( "TimeFormatS", args );
    }
    
    /**
     * Converts the given seconds to a time format with following format:
     * days:hours:minutes:seconds. When days &lt; 0 the format will be 
     * hours:minutes:seconds. When hours &lt; 0 the format will be minutes:seconds.
     * Values &lt; 10 will be padded with a 0. 
     */
    public static String convertSecondsToTime( int seconds )
    {
        StringBuffer buffer = new StringBuffer();
        int days = seconds / 86400;
        int hours = (seconds / 3600) % 24;
        int minutes = (seconds / 60) % 60;
        int secs = seconds % 60;
        
        if ( days > 0 ) // Display days and hours
        {
            buffer.append( Integer.toString( days ) );
            buffer.append( ":" );
            if ( hours < 10 )
            {
                buffer.append( "0" );
            }
        }
        if ( days > 0 || hours > 0 )
        {
            buffer.append( Integer.toString( hours ) );
            buffer.append( ":" );
            if ( minutes < 10 )
            {
                buffer.append( "0" );
            }
        }
        
        buffer.append( Integer.toString( minutes ) );
        buffer.append( ":" );
        if ( secs < 10 )
        {
            buffer.append( "0" );
        }
        buffer.append( Integer.toString( secs ) );
        return buffer.toString();
    }

    /**
     * Formats the the size as a most significant number of bytes.
     */
    public static String formatSizeBytes( double size )
    {
        StringBuffer buf = new StringBuffer( 16 );
        String text;
        double divider;
        if (size < ONE_KB)
        {
            text = "bytes";
            divider = 1.0;
        }
        else if (size < ONE_MB)
        {
            text = "KB";
            divider = ONE_KB;
        }
        else if (size < ONE_GB)
        {
            text = "MB";
            divider = ONE_MB;
        }
        else if (size < ONE_TB)
        {
            text = "GB";
            divider = ONE_GB;
        }
        else
        {
            text = "TB";
            divider = ONE_TB;
        }
        double d = ((double)size) / divider;
        byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(' ').append( text );
        return buf.toString();
    }

    public static String formatSizeBytes( Number number )
    {
        return formatSizeBytes( number.doubleValue() );
    }


    /**
    * Converts points and dashes to spacesand removes possible
    * file extensions. This should be used to create search-terms
    * from filenames.
    **/
    public static String createNaturalSearchTerm( String searchTerm )
    {
        int extensionIdx = searchTerm.lastIndexOf(".");
        if ( extensionIdx >= 0 )
        {
            // cut of file extension
            searchTerm = searchTerm.substring( 0, extensionIdx );
        }

        StringTokenizer tokenizer = new StringTokenizer( searchTerm, "-._()[]" );
        int tokenCount = tokenizer.countTokens();
        if ( tokenCount == 0 )
        {// no tokens -> the term must be already natural.
            return searchTerm;
        }

        String token;
        StringBuffer searchTermBuffer = new StringBuffer( searchTerm.length() );
        while( tokenizer.hasMoreTokens() )
        {
            token = tokenizer.nextToken().trim();
            if ( token.length() >= Cfg.MIN_SEARCH_TERM_LENGTH )
            {
                searchTermBuffer.append( token );
                searchTermBuffer.append( ' ' );
            }
        }
        return searchTermBuffer.toString();
    }
    
    /**
     * <p>Joins the elements of the provided array into a single String
     * containing the provided list of elements.</p>
     *
     * <p>No delimiter is added before or after the list.
     * A <code>null</code> separator is the same as an empty String ("").
     * Null objects or empty strings within the array are represented by
     * empty strings.</p>
     *
     * <pre>
     * StringUtils.join(null, *)                = null
     * StringUtils.join([], *)                  = ""
     * StringUtils.join([null], *)              = ""
     * StringUtils.join(["a", "b", "c"], "--")  = "a--b--c"
     * StringUtils.join(["a", "b", "c"], null)  = "abc"
     * StringUtils.join(["a", "b", "c"], "")    = "abc"
     * StringUtils.join([null, "", "a"], ',')   = ",,a"
     * </pre>
     *
     * @param array  the array of values to join together, may be null
     * @param separator  the separator character to use, null treated as ""
     * @return the joined String, <code>null</code> if null array input
     * 
     * Taken from org.apache.commons.lang.StringUtils
     */
    public static String join(Object[] array, String separator)
    {
        if ( array == null ) { return null; }
        if ( separator == null )
        {
            separator = "";
        }
        int arraySize = array.length;

        // ArraySize ==  0: Len = 0
        // ArraySize > 0:   Len = NofStrings *(len(firstString) + len(separator))
        //           (Assuming that all Strings are roughly equally long)
        int bufSize = ((arraySize == 0) ? 0
            : arraySize
                * ((array[0] == null ? 16 : array[0].toString().length()) + separator.length()));

        StringBuffer buf = new StringBuffer( bufSize );

        for (int i = 0; i < arraySize; i++)
        {
            if ( i > 0 )
            {
                buf.append( separator );
            }
            if ( array[i] != null )
            {
                buf.append( array[i] );
            }
        }
        return buf.toString();
    }
}