/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2004 Gregor Koukkoullis ( phex <at> kouk <dot> de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.xml;

import java.lang.reflect.*;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.io.*;

import javax.xml.bind.*;

import phex.utils.*;

/**
 * Helper class to create, generate and parse a xml Document from various
 * resources.
 * @author Gregor Koukkoullis
 */
public class XMLBuilder
{
    /**
     * Trys to load the XJBPhex marshalled object from the given file.
     * If the file doesn't exist, null is returned.
     * Method takes care of file locking.
     * @param aFile the file to load the XJBPhex object from.
     * @return the XJBPhex object or null if file doesn't exist.
     */
    public static XJBPhex loadXJBPhexFromFile( File aFile )
        throws JAXBException
    {
        if ( !aFile.exists() )
        {
            return null;
        }

        XJBPhex phex;
        FileInputStream fis = null;
        // File locking is causing problem on read in Mac OS X
        //FileLock lock = null;
        try
        {
            fis = new FileInputStream( aFile );
            // we only lock the first 10 bytes here and on load. This is a try to work around
            // various locking bugs on different plattforms.
//          File locking is causing problem on read in Mac OS X
            //lock = lockFile(fis.getChannel(), 10, true);

            // create a JAXBContext
            //JAXBContext jc = JAXBContext.newInstance( "phex.xml" );
            JAXBContext jc = getJAXBContextFix();

            // create an Unmarshaller
            Unmarshaller unmarshaller = jc.createUnmarshaller();

            phex = (XJBPhex)unmarshaller.unmarshal( fis );
            return phex;
        }
        catch ( FileNotFoundException exp )
        {
            // TODO bring a GUI message that file cant be created
            Logger.logError( exp );
            return null;
        }
        catch ( IOException exp )
        {
            throw new JAXBException( exp );
        }
        finally
        {
            IOUtil.closeQuietly( fis );
//          File locking is causing problem on read in Mac OS X
//            if ( lock != null )
//            {
//                try
//                {
//                    lock.release();
//                }
//                catch (IOException e)
//                {
//                    Logger.logError( e );
//                }
//            }
        }
    }
    
    /**
     * Trys to read the XJBPhex marshalled object from the given stream.
     * @param inStream the stream to read the XJBPhex object from.
     * @return the XJBPhex object.
     */
    public static XJBPhex readXJBPhexFromStream( InputStream inStream )
        throws JAXBException
    {
        XJBPhex phex;
        
        // create a JAXBContext
        //JAXBContext jc = JAXBContext.newInstance( "phex.xml" );
        JAXBContext jc = getJAXBContextFix();

        // create an Unmarshaller
        Unmarshaller unmarshaller = jc.createUnmarshaller();
        
        phex = (XJBPhex)unmarshaller.unmarshal( inStream );
        return phex;
    }
    
    public static byte[] serializeToBytes( XJBPhex xjbPhex )
        throws JAXBException
    {
        ByteArrayOutputStream bos = null;
    
        // create a JAXBContext
        //JAXBContext jc = JAXBContext.newInstance( "phex.xml" );
        JAXBContext jc = getJAXBContextFix();

        // marshal to file
        Marshaller m = jc.createMarshaller();
        m.setProperty( "jaxb.formatted.output", Boolean.TRUE );

        bos = new ByteArrayOutputStream( );
        m.marshal( xjbPhex, bos );
        return bos.toByteArray();
    }

    /**
     * Method takes care of file locking.
     * @param aFile
     * @param xjbPhex
     * @throws JAXBException
     */
    public static void saveToFile( File aFile, XJBPhex xjbPhex )
        throws JAXBException
    {
        BufferedOutputStream bufferedOutStream = null;
        FileOutputStream fos = null;
        FileLock lock = null;
        try
        {
            fos = new FileOutputStream( aFile );
            // we only lock the first 10 bytes here and on load. This is a try to work around
            // various locking bugs on different plattforms.
            lock = lockFile(fos.getChannel(), 10, false);
            
            // create a JAXBContext
            //JAXBContext jc = JAXBContext.newInstance( "phex.xml" );
            JAXBContext jc = getJAXBContextFix();

            // marshal to file
            Marshaller m = jc.createMarshaller();
            m.setProperty( "jaxb.formatted.output", Boolean.TRUE );
            bufferedOutStream = new BufferedOutputStream( fos );
            m.marshal( xjbPhex, bufferedOutStream );
        }
        catch ( FileNotFoundException exp )
        {
            throw new JAXBException( exp );
        }
        catch ( IOException exp )
        {
            throw new JAXBException( exp );
        }
        finally
        {
            IOUtil.closeQuietly( bufferedOutStream );
            IOUtil.closeQuietly( fos );
            if ( lock != null )
            {
                try
                {
                    lock.release();
                }
                catch (IOException e)
                {
                    Logger.logError( e );
                }
            }
        }
    }

    /**
     * @param fos
     * @throws IOException
     */
    private static FileLock lockFile(FileChannel channel, long size, boolean shared) throws IOException
    {
        FileLock lock = null;
        int lockTryCount = 0;
        while (lock==null)
        {
            lock = channel.tryLock( 0, size, shared );
            if ( lock == null )
            {
                lockTryCount ++;
                if ( lockTryCount > 10 )
                {
                    throw new IOException( "Locking file failed.");
                }
                try
                {
                    Thread.sleep( 1000 );
                }
                catch (InterruptedException e)
                {
                    Thread.interrupted();
                }
            }
        }
        return lock;
    }

    /**
     * Fixes a bug in JAXB 1.0 beta see
     * http://developer.java.sun.com/developer/bugParade/bugs/4769996.html
     * @return
     */
    private static JAXBContext getJAXBContextFix()
        throws JAXBException
    {
        /* class loader to be used */
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

        try {
            Class spiClass;
            if (classLoader == null) {
                spiClass = Class.forName("com.sun.xml.bind.ContextFactory");
            } else {
                spiClass = classLoader.loadClass("com.sun.xml.bind.ContextFactory");
            }

            /*
             * javax.xml.bind.context.factory points to a class which has a
             * static method called 'createContext' that takes a single string
             * argument and returns a javax.xml.JAXBContext.
             *
             * add exception handling
             *
             */
            Class paramTypes[] = {
                java.lang.String.class,
                ClassLoader.class
            };
            Method m = spiClass.getMethod( "createContext", paramTypes );

            Object invocationParams[] = {
                "phex.xml",
                classLoader
            };

            // In the RI, this is equivalent to:
            // com.sun.xml.bind.ContextFactory.createContext( contextPath )
            return (JAXBContext)m.invoke( null, invocationParams );
        }
        catch (ClassNotFoundException x)
        {
            throw new JAXBException(x);
        }
        catch ( InvocationTargetException exp )
        {
            Logger.logError( Logger.GLOBAL, "Wrapped InvocationTargetException:" );
            Logger.logError( exp.getTargetException() );
            throw new JAXBException( exp );
        }
        catch (Exception x)
        {
            throw new JAXBException( x );
        }
    }




    /*private static final DocumentBuilder documentBuilder;

    static
    {
        try
        {
            documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        }
        catch ( ParserConfigurationException exp )
        {
            exp.printStackTrace();
            throw new RuntimeException( exp.getMessage() );
        }
    }*/

    /*public static XmlDocument createNewDocument()
    {
        return (XmlDocument) documentBuilder.newDocument();
    }

    public static Document loadFromSystemResource(String name)
    {
        InputStream stream = ClassLoader.getSystemResourceAsStream(name);
        if (stream == null)
        {
            return null;
        }
        BufferedInputStream bufferedStream = new BufferedInputStream(stream);
        return loadFromStream(bufferedStream);
    }

    public static Document loadFromResource( String resourceName )
    {
        InputStream stream =
            XMLBuilder.class.getResourceAsStream( resourceName );
        if ( stream != null )
        {
            BufferedInputStream bStream = new BufferedInputStream( stream );
            return loadFromStream( bStream );
        }
        else
        {
            System.out.println( "Can't find resource: " + resourceName + ".");
            return null;
        }
    }

    public static Document loadFromFile( File aFile )
    {
        BufferedInputStream bufferedStream = null;
        try
        {
            FileInputStream stream = new FileInputStream( aFile );
            if (stream == null)
            {
                return null;
            }
            bufferedStream = new BufferedInputStream(stream);
            // synchronize device... before reading...
            stream.getFD().sync();
            Document doc = loadFromStream(bufferedStream);

            return doc;
        }
        // TO DO should throw a exception instead... more clean!
        catch ( IOException exp )
        {
            Logger.logError( exp );
            return null;
        }
        finally
        {
            if ( bufferedStream != null )
            {
                try
                {
                    bufferedStream.close();
                }
                catch ( IOException exp )
                {
                }
            }
        }
    }*/

    /*public static void saveToFile( File aFile, XmlDocument doc )
        throws XMLException
    {
        BufferedWriter writer = null;
        try
        {
            // write backup file
            FileOutputStream fos = new FileOutputStream( aFile );
            writer = new BufferedWriter( new OutputStreamWriter( fos, "UTF8" ) );
            // synchronize device... before writing...
            fos.getFD().sync();
            doc.write( writer, "UTF-8" );
        }
        catch ( IOException exp )
        {
            Logger.logError( exp );
            throw new XMLException( exp.getMessage() );
        }
        finally
        {
            if ( writer != null )
            {
                try
                {
                    writer.close();
                }
                catch ( IOException exp )
                {
                }
            }
        }
    }

    public static Document loadFromStream(InputStream stream)
    {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        try
        {
            DocumentBuilder builder = factory.newDocumentBuilder();
            return builder.parse(stream);
        }
        catch (SAXParseException exp)
        {
            Logger.logError( exp );
        }
        catch (Exception exp)
        {
            Logger.logError( exp );
        }
        return null;
    }*/
}
