/*
 *  SRM  --  GRand Unified Bootloader
 *  Copyright (C) 1996  Erich Boleyn  <erich@uruk.org>
 *  Copyright (C) 2000, 2001  Free Software Foundation, Inc.
 *  Copyright (C) 2001  BERTRAND Jol
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "shared.h"

int view_x0 = 0, view_y0 = 0, view_x1 = 80, view_y1 = 25;
srm_jmp_buf restart_env;
#ifdef VGA16
int vga_disabled = 0, viewport_set = 0;
int sx0, sy0, sx1, sy1;
#endif

#ifdef PRESET_MENU_STRING

static const char *preset_menu = PRESET_MENU_STRING;
static int preset_menu_offset;

static int
open_preset_menu (void)
{
  preset_menu_offset = 0;
  return 1;
}

static int
read_from_preset_menu (char *buf, int maxlen)
{
  int len = srm_strlen (preset_menu + preset_menu_offset);

  if (len > maxlen)
    len = maxlen;

  srm_memmove (buf, preset_menu + preset_menu_offset, len);
  preset_menu_offset += len;

  return len;
}

static void
close_preset_menu (void)
{
  /* Do nothing.  */
}

#else /* ! PRESET_MENU_STRING */

#define open_preset_menu()	0
#define read_from_preset_menu(buf, maxlen)	0
#define close_preset_menu()

#endif /* ! PRESET_MENU_STRING */


static char *
get_entry (char *list, int num, int nested)
{
  int i;

  for (i = 0; i < num; i++)
    {
      do
	{
	  while (*(list++));
	}
      while (nested && *(list++));
    }

  return list;
}


static void
print_entries (int y, int size, int first, char *menu_entries)
{
  int i;
  int disp_up = DISP_UP;
  int disp_down = DISP_DOWN;

#ifdef SUPPORT_SERIAL
  if (terminal & TERMINAL_SERIAL)
    {
      disp_up = ACS_UARROW;
      disp_down = ACS_DARROW;
    }
#endif /* SUPPORT_SERIAL */
  
  gotoxy (view_x1 - 3, y + 1);

  if (first)
    putchar (disp_up);
  else
    putchar (' ');

  menu_entries = get_entry (menu_entries, first, 0);

  for (i = 1; i <= size; i++)
    {
      int j = 0;

      gotoxy (view_x0 + 3, y + i);

      while (*menu_entries)
	{
	  if (j < (view_x1 - view_x0) - 9)
	    {
	      putchar (*menu_entries);
	      j++;
	    }

	  menu_entries++;
	}

      if (*(menu_entries - 1))
	menu_entries++;

      for (; j < (view_x1 - view_x0) - 9; j++)
	putchar (' ');
    }

  gotoxy (view_x0 + 3, y + size);

  if (*menu_entries)
    putchar (disp_down);
  else
    putchar (' ');
}


static void
print_entries_raw (int size, int first, char *menu_entries)
{
  int i;

#define LINE_LENGTH 67

  for (i = 0; i < LINE_LENGTH; i++)
    srm_putchar ('-');
  srm_putchar ('\n');

  for (i = first; i < size; i++)
    {
      /* srm's printf can't %02d so ... */
      if (i < 10)
	srm_putchar (' ');
      srm_printf ("%d: %s\n", i, get_entry (menu_entries, i, 0));
    }

  for (i = 0; i < LINE_LENGTH; i++)
    srm_putchar ('-');
  srm_putchar ('\n');

#undef LINE_LENGTH
}


static void
print_border (int y, int size)
{
  int i;
  int disp_ul = DISP_UL;
  int disp_ur = DISP_UR;
  int disp_ll = DISP_LL;
  int disp_lr = DISP_LR;
  int disp_horiz = DISP_HORIZ;
  int disp_vert = DISP_VERT;

#ifdef SUPPORT_SERIAL
  if (terminal & TERMINAL_SERIAL)
    {
      disp_ul = ACS_ULCORNER;
      disp_ur = ACS_URCORNER;
      disp_ll = ACS_LLCORNER;
      disp_lr = ACS_LRCORNER;
      disp_horiz = ACS_HLINE;
      disp_vert = ACS_VLINE;
    }
#endif /* SUPPORT_SERIAL */
  
#ifndef SRM_UTIL
  /* Color the menu. The menu is 75 * 14 characters.  */
# ifdef SUPPORT_SERIAL
  if ((terminal & TERMINAL_CONSOLE)
#  ifdef SUPPORT_HERCULES
      || (terminal & TERMINAL_HERCULES)
#  endif
      )
# endif
    {
      for (i = 0; i < 14; i++)
	{
	  int j;
	  for (j = view_x0; j < (view_x1 - view_x0) - 5; j++)
	    {
	      gotoxy (j + 1, i + y);
	      set_attrib (normal_color);
	    }
	}
    }
#endif

  gotoxy (view_x0 + 1, y);

  putchar (disp_ul);
  for (i = view_x0; i < view_x1 - 7; i++)
    putchar (disp_horiz);
  putchar (disp_ur);

  i = 1;

  while (1)
    {
      gotoxy (view_x0 + 1, y + i);

      if (i > size)
	break;

      putchar (disp_vert);
      gotoxy (view_x1 - 5, y + i);
      putchar (disp_vert);

      i++;
    }

  putchar (disp_ll);
  for (i = view_x0; i < view_x1 - 7; i++)
    putchar (disp_horiz);
  putchar (disp_lr);
}

static void
set_line (int y, int entryno, int attr, char *menu_entries)
{
  int x;

#ifdef SUPPORT_SERIAL
  if (terminal & TERMINAL_SERIAL)
    {
      menu_entries = get_entry (menu_entries, entryno, 0);
      gotoxy (view_x0 + 2, y);
      putchar (' ');
      for (x = view_x0 + 3; x < view_x1 - 5; x++)
	{
	  if (*menu_entries && x < view_x1 - 9)
	    putchar (*menu_entries++);
	  else
	    putchar (' ');
	}
    }
  else
#endif /* SUPPORT_SERIAL */
    {
      for (x = view_x0 + 2; x < view_x1 - 5; x++)
	{
	  gotoxy (x, y);
	  set_attrib (attr);
	}
    }

  gotoxy (view_x1 - 6, y);
}

/* Set the attribute of the line Y to normal state.  */
static void
set_line_normal (int y, int entryno, char *menu_entries)
{
#ifdef SRM_UTIL
  set_line (y, entryno, A_NORMAL, menu_entries);
#else
  set_line (y, entryno, normal_color, menu_entries);
#endif
}

/* Set the attribute of the line Y to highlight state.  */
static void
set_line_highlight (int y, int entryno, char *menu_entries)
{
#ifdef SUPPORT_SERIAL
  if (terminal & TERMINAL_SERIAL)
    srm_printf ("\e[7m");
#endif /* SUPPORT_SERIAL */
  
#ifdef SRM_UTIL
  set_line (y, entryno, A_REVERSE, menu_entries);
#else
  set_line (y, entryno, highlight_color, menu_entries);
#endif
  
#ifdef SUPPORT_SERIAL
  if (terminal & TERMINAL_SERIAL)
    srm_printf ("\e[0m");
#endif /* SUPPORT_SERIAL */
}

static void
run_menu (char *menu_entries, char *config_entries, int num_entries,
	  char *heap, int entryno)
{
  int c, time1, time2 = -1, first_entry = 0;
  char *cur_entry = 0;
  int disp_up = DISP_UP;
  int disp_down = DISP_DOWN;
  int ask_pass = 0;

  /*
   *  Main loop for menu UI.
   */

restart:
  /* Dumb terminal always use all entries for display 
     invariant for TERMINAL_DUMB: first_entry == 0  */
  if (! (terminal & TERMINAL_DUMB))
    {
      errnum = ERR_NONE;
      while (entryno > 9)
	{
	  first_entry++;
	  entryno--;
	}
    }

  /* If the timeout was expired or wasn't set, force to show the menu
     interface. */
  if (srm_timeout < 0)
    show_menu = 1;
  
  /* If SHOW_MENU is false, don't display the menu until ESC is pressed.  */
  if (! show_menu)
    {
      /* Get current time.  */
      while ((time1 = getrtsecs ()) == 0xFF)
	;

      while (1)
	{
	  /* Check if ESC is pressed.  */
	  if (checkkey () != -1 && ASCII_CHAR (getkey ()) == '\e')
	    {
	      srm_timeout = -1;
	      show_menu = 1;
	      break;
	    }

	  /* If SRM_TIMEOUT is expired, boot the default entry.  */
	  if (srm_timeout >=0
	      && (time1 = getrtsecs ()) != time2
	      && time1 != 0xFF)
	    {
	      if (srm_timeout <= 0)
		{
		  srm_timeout = -1;
		  goto boot_entry;
		}
	      
	      time2 = time1;
	      srm_timeout--;
	      
	      /* Print a message.  */
	      srm_printf ("\rPress `ESC' to enter the menu... %d   ",
			   srm_timeout);
	    }
	}
    }

  /* Only display the menu if the user wants to see it. */
  if (show_menu)
    {
      /* Disable the auto fill mode.  */
      auto_fill = 0;
      
      init_page ();
#ifndef SRM_UTIL
# ifdef SUPPORT_SERIAL
      if (terminal & TERMINAL_CONSOLE)
# endif /* SUPPORT_SERIAL */
	nocursor ();
#endif /* ! SRM_UTIL */

      if (! (terminal & TERMINAL_DUMB))      
	  print_border (3 + view_y0, 12);

#ifdef SRM_UTIL
      /* In the srm shell, always use ACS_*.  */
      disp_up = ACS_UARROW;
      disp_down = ACS_DARROW;
#else /* ! SRM_UTIL */
# ifdef SUPPORT_SERIAL
      if ((terminal & TERMINAL_CONSOLE)
#  ifdef SUPPORT_HERCULES
	  || (terminal & TERMINAL_HERCULES)
#  endif /* SUPPORT_HERCULES */
	  )
	{
	  disp_up = DISP_UP;
	  disp_down = DISP_DOWN;
	}
      else
	{
	  disp_up = ACS_UARROW;
	  disp_down = ACS_DARROW;
	}
# endif /* SUPPORT_SERIAL */
#endif /* ! SRM_UTIL */
      
      if (terminal & TERMINAL_DUMB)
	  print_entries_raw (num_entries, first_entry, menu_entries);

      srm_printf ("\n\
      Use the %c and %c keys to select which entry is highlighted.\n",
		   disp_up, disp_down);

      if (! auth && password && ask_pass) 
	{
	/* Do password check here! */
	char entered[32];
	char *pptr = password;

	/* we've asked, so don't ask again */
	ask_pass = 0;

	if (terminal & TERMINAL_DUMB)      
	  srm_printf ("\n\nThe selected entry is %d ", entryno);
	else 
	  {
	    print_entries (view_y0 + 3, 12, first_entry, menu_entries);
	    /* highlight initial line */
	    set_line_highlight (view_y0 + 4 + entryno, first_entry + entryno, 
				menu_entries);
	  }
	
	/* Wipe out the previously entered password */
	memset (entered, 0, sizeof (entered));
	gotoxy (view_x0 + 1, view_y0 + 16);
	get_cmdline (" Password: ", entered, 31, '*', 0);
	
	while (! isspace (*pptr) && *pptr)
	  pptr++;
	
	/* Make sure that PASSWORD is NUL-terminated.  */
	*pptr++ = 0;
	
	if (! check_password (entered, password, password_type))
	  {
	    char *new_file = config_file;

	    while (isspace (*pptr))
	      pptr++;
	    
	    /* If *PPTR is NUL, then allow the user to use
	       privileged instructions, otherwise, load
	       another configuration file.  */
	       if (*pptr != 0)
		 {
		   while ((*(new_file++) = *(pptr++)) != 0)
		     ;
		   
		   /* Make sure that the user will not have
		      authority in the next configuration.  */
		   auth = 0;
		   return;
		 }
	       else
		 {
		   /* Now the user is superhuman.  */
		   auth = 1;
		   goto restart;
		 }
	    }
	else
	  {
	    printf (" Failed!\n      Press any key to continue...");
	    getkey ();
	    goto restart;
	  }
        }  
      else if (! auth && password)
	{
	  printf ("\
      Press enter to boot the selected OS or \'p\' to enter a\n\
      password to unlock the next set of features.");
	}
      else
	{
	  if (config_entries)
	    printf ("\
      Press enter to boot the selected OS, \'e\' to edit the\n\
      commands before booting, or \'c\' for a command-line.");
	  else
	    printf ("\
      Press \'b\' to boot, \'e\' to edit the selected command in the\n\
      boot sequence, \'c\' for a command-line, \'o\' to open a new line\n\
      after (\'O\' for before) the selected line, \'d\' to remove the\n\
      selected line, or escape to go back to the main menu.");
	}

      if (terminal & TERMINAL_DUMB)      
	srm_printf ("\n\nThe selected entry is %d ", entryno);
      else
      {
	  print_entries (view_y0 + 3, 12, first_entry, menu_entries);
	  
	  /* highlight initial line */
	  set_line_highlight (view_y0 + 4 + entryno, first_entry + entryno, 
			      menu_entries);
      }
    }

  /* XX using RT clock now, need to initialize value */
  while ((time1 = getrtsecs()) == 0xFF);

  while (1)
    {
      /* Initialize to NULL just in case...  */
      cur_entry = NULL;

      if (srm_timeout >= 0 && (time1 = getrtsecs()) != time2 && time1 != 0xFF)
	{
	  if (srm_timeout <= 0)
	    {
	      srm_timeout = -1;
	      break;
	    }

	  /* else not booting yet! */
	  time2  = time1;

	  if (terminal & TERMINAL_DUMB)
		  if (action == 2)
		  {
			  srm_printf ("\r    The system will be halted automatically in %d seconds.    ", srm_timeout);
		  }
		  else
		  {	  
			  srm_printf ("\r    Entry %d will be booted automatically in %d seconds.   ", 
				   entryno, srm_timeout);
		  }
	  else
	  {
	      gotoxy (view_x0 + 3, view_y1 - 3);
		  if (action == 2)
		  {
			  printf ("The system will be halted automatically in %d seconds.    ", srm_timeout);
		  }
		  else
		  {
			  printf ("The highlighted entry will be booted automatically in %d seconds.    ", srm_timeout);
		  }
	      gotoxy (view_x0 - 6, view_y0 + 4 + entryno);
	  }
	  
	  srm_timeout--;
	}

      /* Check for a keypress, however if TIMEOUT has been expired
	 (SRM_TIMEOUT == -1) relax in GETKEY even if no key has been
	 pressed.  
	 This avoids polling (relevant in the srm-shell and later on
	 in srm if interrupt driven I/O is done).  */
      if ((checkkey () != -1) || (srm_timeout == -1))
	{
	  /* Key was pressed, show which entry is selected before GETKEY,
	     since we're comming in here also on SRM_TIMEOUT == -1 and
	     hang in GETKEY */
	  if (terminal & TERMINAL_DUMB)
	    srm_printf ("\r    Highlighted entry is %d: ", entryno);

		c = translate_keycode (getkey ());

	  /* We told them above (at least in SUPPORT_SERIAL) to use
	     '^' or 'v' so accept these keys.  */
	  if (c == 16 || c == '^')
	    {
	      if (terminal & TERMINAL_DUMB)
		{
		  if (entryno > 0)
		    entryno--;
		}
	      else
		{
		  if (entryno > 0)
		    {
		      set_line_normal (view_y0 + 4 + entryno, first_entry + entryno,
				       menu_entries);
		      entryno--;
		      set_line_highlight (view_y0 + 4 + entryno, first_entry + entryno,
					  menu_entries);
		    }
		  else if (first_entry > 0)
		    {
		      first_entry--;
		      print_entries (view_y0 + 3, 12, first_entry, menu_entries);
		      set_line_highlight (view_y0 + 4, first_entry + entryno, 
					  menu_entries);
		    }
		}
	    }
	  if ((c == 14 || c == 'v') && first_entry + entryno + 1 < num_entries)
	    {
	      if (terminal & TERMINAL_DUMB)
		entryno++;
	      else
		if (entryno < 11)
		  {
		    set_line_normal (view_y0 + 4 + entryno, first_entry + entryno,
				     menu_entries);
		    entryno++;
		    set_line_highlight (view_y0 + 4 + entryno, first_entry + entryno,
					menu_entries);
		  }
		else if (num_entries > 12 + first_entry)
		  {
		    first_entry++;
		    print_entries (view_y0 + 3, 12, first_entry, menu_entries);
		    set_line_highlight (view_y0 + 15, first_entry + entryno, menu_entries);
		  }
	    }

	  if (config_entries)
	    {
	      if ((c == '\n') || (c == '\r'))
		break;
	    }
	  else
	    {
	      if ((c == 'd') || (c == 'o') || (c == 'O'))
		{
		  if (! (terminal & TERMINAL_DUMB))
		    set_line_normal (view_y0 + 4 + entryno, first_entry + entryno,
				     menu_entries);

		  /* insert after is almost exactly like insert before */
		  if (c == 'o')
		    {
		      /* But `o' differs from `O', since it may causes
			 the menu screen to scroll up.  */
		      if (entryno < 11 || (terminal & TERMINAL_DUMB))
			entryno++;
		      else
			first_entry++;
		      
		      c = 'O';
		    }

		  cur_entry = get_entry (menu_entries,
					 first_entry + entryno,
					 0);

		  if (c == 'O')
		    {
		      memmove (cur_entry + 2, cur_entry,
			       ((int) heap) - ((int) cur_entry));

		      cur_entry[0] = ' ';
		      cur_entry[1] = 0;

		      heap += 2;

		      num_entries++;
		    }
		  else if (num_entries > 0)
		    {
		      char *ptr = get_entry(menu_entries,
					    first_entry + entryno + 1,
					    0);

		      memmove (cur_entry, ptr, ((int) heap) - ((int) ptr));
		      heap -= (((int) ptr) - ((int) cur_entry));

		      num_entries--;

		      if (entryno >= num_entries)
			entryno--;
		      if (first_entry && num_entries < 12 + first_entry)
			first_entry--;
		    }

		  if (terminal & TERMINAL_DUMB)
		    {
		      srm_printf ("\n\n");
		      print_entries_raw (num_entries, first_entry,
					 menu_entries);
		      srm_printf ("\n");
		    }
		  else
		    {
		      print_entries (view_y0 + 3, 12, first_entry, menu_entries);
		      set_line_highlight (view_y0 + 4 + entryno, first_entry + entryno,
					  menu_entries);
		    }
		}

	      cur_entry = menu_entries;
	      if (c == 27)
		return;
	      if (c == 'b')
		break;
	    }

	  if (! auth && password)
	    {
	      if (c == 'p')
		{
		  ask_pass = 1;
		  goto restart;
		}
	    }
	  else
	    {
	      if (c == 'e')
		{
		  int new_num_entries = 0, i = 0;
		  char *new_heap;

		  if (config_entries)
		    {
		      new_heap = heap;
		      cur_entry = get_entry (config_entries,
					     first_entry + entryno,
					     1);
		    }
		  else
		    {
		      /* safe area! */
		      new_heap = heap + NEW_HEAPSIZE + 1;
		      cur_entry = get_entry (menu_entries,
					     first_entry + entryno,
					     0);
		    }

		  do
		    {
		      while ((*(new_heap++) = cur_entry[i++]) != 0);
		      new_num_entries++;
		    }
		  while (config_entries && cur_entry[i]);

		  /* this only needs to be done if config_entries is non-NULL,
		     but it doesn't hurt to do it always */
		  *(new_heap++) = 0;

		  if (config_entries)
		    run_menu (heap, NULL, new_num_entries, new_heap, 0);
		  else
		    {
		      cls ();
		      print_cmdline_message (0);

		      new_heap = heap + NEW_HEAPSIZE + 1;

		      saved_drive = boot_drive;
		      saved_partition = install_partition;
		      current_drive = 0xFF;

		      if (! get_cmdline (PACKAGE " edit> ", new_heap,
					 NEW_HEAPSIZE + 1, 0, 1))
			{
			  int j = 0;

			  /* get length of new command */
			  while (new_heap[j++])
			    ;

			  if (j < 2)
			    {
			      j = 2;
			      new_heap[0] = ' ';
			      new_heap[1] = 0;
			    }

			  /* align rest of commands properly */
			  memmove (cur_entry + j, cur_entry + i,
				   ((int) heap) - (((int) cur_entry) + i));

			  /* copy command to correct area */
			  memmove (cur_entry, new_heap, j);

			  heap += (j - i);
			}
		    }

		  goto restart;
		}
	      if (c == 'c')
		{
		  enter_cmdline (heap, 0);
		  goto restart;
		}
#ifdef SRM_UTIL
	      if (c == 'q')
		{
		  /* The same as ``quit''.  */
		  stop ();
		}
#endif
	    }
	  if (srm_timeout >= 0)
	    {
	      srm_timeout = -1;
	      fallback_entry = -1;
	      goto restart;
	    }
	}
    }

  /* Attempt to boot an entry.  */
  
 boot_entry:
  /* Enable the auto fill mode.  */
  auto_fill = 1;
  
  if (action == 2) /* HALT */
  {
	  int no_apm;

	  srm_halt(0);
	  /* Never reach here. */
	  return 1;
  }

  while (1)
    {
      cls ();
#ifdef VGA16
      if (display->End)
	(*display->End)();
      display = &display_entries[0];
#endif

      if (config_entries)
	printf ("  Booting \'%s\'\n\n",
		get_entry (menu_entries, first_entry + entryno, 0));
      else
	printf ("  Booting command-list\n\n");

      if (! cur_entry)
	cur_entry = get_entry (config_entries, first_entry + entryno, 1);

      /* Set CURRENT_ENTRYNO for the command "savedefault".  */
      current_entryno = first_entry + entryno;
      
      if (run_script (cur_entry, heap))
	{
	  if (fallback_entry < 0)
	    break;
	  else
	    {
	      cur_entry = NULL;
	      first_entry = 0;
	      entryno = fallback_entry;
	      fallback_entry = -1;
	    }
	}
      else
	break;
    }

#ifdef VGA16
      if (display_idx >= 0)
	display = &display_entries[display_idx];
      else if (!vga_disabled)
	display = &display_entries[1];

      if (!display->Begin || !(*display->Begin)())
	display = &display_entries[0];
#endif

  show_menu = 1;
  goto restart;
}


static int
get_line_from_config (char *cmdline, int maxlen, int read_from_file)
{
  int pos = 0, literal = 0, comment = 0;
  char c;  /* since we're loading it a byte at a time! */
  
  while (1)
    {
      if (read_from_file)
	{
	  if (! srm_read (&c, 1))
	    break;
	}
      else
	{
	  if (! read_from_preset_menu (&c, 1))
	    break;
	}
      
      /* translate characters first! */
      if (c == '\\' && ! literal)
	{
	  literal = 1;
	  continue;
	}
      if (c == '\r')
	continue;
      if ((c == '\t') || (literal && (c == '\n')))
	c = ' ';

      literal = 0;

      if (comment)
	{
	  if (c == '\n')
	    comment = 0;
	}
      else if (! pos)
	{
	  if (c == '#')
	    comment = 1;
	  else if ((c != ' ') && (c != '\n'))
	    cmdline[pos++] = c;
	}
      else
	{
	  if (c == '\n')
	    break;

	  if (pos < maxlen)
	    cmdline[pos++] = c;
	}
    }

  cmdline[pos] = 0;

  return pos;
}

#ifdef VGA16
struct display_entry display_entries[MAX_DISPLAYS + 1] = {
    {"text", 0, 0, srm_cls, srm_gotoxy, srm_putchar, srm_set_attrib,
     srm_getxy, srm_nocursor},
    {"vga16",
     vga16_begin, vga16_end, vga16_cls, vga16_gotoxy, vga16_putchar,
     vga16_set_attrib, vga16_getxy, vga16_nocursor},
    {0, 0, 0, 0, 0, 0, 0, 0, 0}
};
struct display_entry *display;
int display_idx = -1;

/* default vga palette */
char vga16pal[16][3] = {
    { 0,  0,  0},
    { 0,  0, 42},
    { 0, 42,  0},
    { 0, 42, 42},
    {42,  0,  0},
    {42,  0, 42},
    {42, 21,  0},
    {42, 42, 42},
    {21, 21, 21},
    {21, 21, 63},
    {21, 63, 21},
    {21, 63, 63},
    {63, 21, 21},
    {63, 21, 63},
    {63, 63, 21},
    {63, 63, 63},
};

int fontx, fonty;
unsigned char *font8x16;
int saved_videomode, no_scroll = 0, no_cursor = 0, shade = 1, vga_inited = 0;
unsigned short text[80 * 30];
int foreground = (63 << 16) | (63 << 8) | (63), background = 0, border = 0;
int set_image;
char image[64];
#define VSHADOW VSHADOW1
unsigned char VSHADOW1[38400];
unsigned char VSHADOW2[38400];
unsigned char VSHADOW4[38400];
unsigned char VSHADOW8[38400];

static inline void
outb(unsigned short port, unsigned char val)
{
    __asm __volatile ("outb %0,%1"::"a" (val), "d" (port));
}

static void
ModeReg(int value)
{
    outb(0x3ce, 5);
    outb(0x3cf, value);
}

static void
MapMask(int value)
{
    outb(0x3c4, 2);
    outb(0x3c5, value);
}

/* set/reset register */
static void
SetRes(int value)
{
    outb(0x3ce, 0);
    outb(0x3cf, value);
}

/* enable set/reset register */
static void
ESetRes(int value)
{
    outb(0x3ce, 1);
    outb(0x3cf, value);
}

static void
ReadMap(int value)
{
    outb(0x3ce, 4);
    outb(0x3cf, value);
}

/* bit mask register */
static void
BitMask(int value)
{
    outb(0x3ce, 8);
    outb(0x3cf, value);
}

void
srm_memcpy(void *dest, const void *src, int len)
{
    int i;
    register char *d = (char*)dest, *s = (char*)src;

    for (i = 0; i < len; i++)
	d[i] = s[i];
}

int
hex(int v)
{
    if (v >= 'A' && v <= 'F')
	return (v - 'A' + 10);
    if (v >= 'a' && v <= 'f')
	return (v - 'a' + 10);
    return (v - '0');
}

static void
SetXY(int col, int row)
{
    if (col >= view_x0 && col < view_x1) {
	fontx = col;
	cursorX = col << 3;
    }
    if (row >= view_y0 && row < view_y1) {
	fonty = row;
	cursorY = row << 4;
    }
}

void
cursor(int set)
{
    unsigned char *pat, *mem, *ptr, chr[16 << 2];
    int i, ch, invert, offset;

    if (set && (no_cursor || no_scroll))
	return;

    offset = cursorY * 80 + fontx;
    ch = text[fonty * 80 + fontx] & 0xff;
    invert = (text[fonty * 80 + fontx] & 0xff00) != 0;
    pat = font8x16 + (ch << 4);

    mem = (unsigned char*)VIDEO + offset;

    if (!set) {
	for (i = 0; i < 16; i++) {
	    unsigned char mask = pat[i];

	    if (!invert) {
		chr[i	  ] = ((unsigned char*)VSHADOW1)[offset];
		chr[16 + i] = ((unsigned char*)VSHADOW2)[offset];
		chr[32 + i] = ((unsigned char*)VSHADOW4)[offset];
		chr[48 + i] = ((unsigned char*)VSHADOW8)[offset];

		if (shade) {
		    if (ch == DISP_VERT || ch == DISP_LL ||
			ch == DISP_UR || ch == DISP_LR) {
			unsigned char pmask = ~(pat[i] >> 1);

			chr[i     ] &= pmask;
			chr[16 + i] &= pmask;
			chr[32 + i] &= pmask;
			chr[48 + i] &= pmask;
		    }
		    if (i > 0 && ch != DISP_VERT) {
			unsigned char pmask = ~(pat[i - 1] >> 1);

			chr[i	  ] &= pmask;
			chr[16 + i] &= pmask;
			chr[32 + i] &= pmask;
			chr[48 + i] &= pmask;
			if (ch == DISP_HORIZ || ch == DISP_UR || ch == DISP_LR) {
			    pmask = ~pat[i - 1];

			    chr[i     ] &= pmask;
			    chr[16 + i] &= pmask;
			    chr[32 + i] &= pmask;
			    chr[48 + i] &= pmask;
			}
		    }
		}
		chr[i     ] |= mask;
		chr[16 + i] |= mask;
		chr[32 + i] |= mask;
		chr[48 + i] |= mask;

		offset += 80;
	    }
	    else {
		chr[i	  ] = mask;
		chr[16 + i] = mask;
		chr[32 + i] = mask;
		chr[48 + i] = mask;
	    }
	}
    }
    else {
	MapMask(15);
	ptr = mem;
	for (i = 0; i < 16; i++, ptr += 80) {
	    cursorBuf[i] = pat[i];
	    *ptr = ~pat[i];
	}
	return;
    }

    offset = 0;
    for (i = 1; i < 16; i <<= 1, offset += 16) {
	int j;

	MapMask(i);
	ptr = mem;
	for (j = 0; j < 16; j++, ptr += 80)
	    *ptr = chr[j + offset];
    }

    MapMask(15);
}

int
read_image(void)
{
    char buf[32], pal[16];
    unsigned char c, base, mask, *s1, *s2, *s4, *s8;
    unsigned i, len, idx, colors, x, y, width, height;

    if (!srm_open(image))
	return (0);

    /* read header */
    if (!srm_read((char*)&buf, 10) || srm_memcmp(buf, "/* XPM */\n", 10)) {
	srm_close();
	return (0);
    }

    /* parse info */
    while (srm_read(&c, 1)) {
	if (c == '"')
	    break;
    }

    while (srm_read(&c, 1) && (c == ' ' || c == '\t'))
	;

    i = 0;
    width = c - '0';
    while (srm_read(&c, 1)) {
	if (c >= '0' && c <= '9')
	    width = width * 10 + c - '0';
	else
	    break;
    }
    while (srm_read(&c, 1) && (c == ' ' || c == '\t'))
	;

    height = c - '0';
    while (srm_read(&c, 1)) {
	if (c >= '0' && c <= '9')
	    height = height * 10 + c - '0';
	else
	    break;
    }
    while (srm_read(&c, 1) && (c == ' ' || c == '\t'))
	;

    colors = c - '0';
    while (srm_read(&c, 1)) {
	if (c >= '0' && c <= '9')
	    colors = colors * 10 + c - '0';
	else
	    break;
    }

    base = 0;
    while (srm_read(&c, 1) && c != '"')
	;

    /* palette */
    for (i = 0, idx = 1; i < colors; i++) {
	len = 0;

	while (srm_read(&c, 1) && c != '"')
	    ;
	srm_read(&c, 1);	/* char */
	base = c;
	srm_read(buf, 4);	/* \t c # */

	while (srm_read(&c, 1) && c != '"') {
	    if (len < sizeof(buf))
		buf[len++] = c;
	}

	if (len == 6 && idx < 15) {
	    int r = ((hex(buf[0]) << 4) | hex(buf[1])) >> 2;
	    int g = ((hex(buf[2]) << 4) | hex(buf[3])) >> 2;
	    int b = ((hex(buf[4]) << 4) | hex(buf[5])) >> 2;

	    pal[idx] = base;
	    set_palette(idx, r, g, b);
	    ++idx;
	}
    }

    x = y = len = 0;

    s1 = (unsigned char*)VSHADOW1;
    s2 = (unsigned char*)VSHADOW2;
    s4 = (unsigned char*)VSHADOW4;
    s8 = (unsigned char*)VSHADOW8;

    for (i = 0; i < 38400; i++)
	s1[i] = s2[i] = s4[i] = s8[i] = 0;

    /* parse xpm data */
    while (y < height) {
	while (1) {
	    if (!srm_read(&c, 1)) {
		srm_close();
		return (0);
	    }
	    if (c == '"')
		break;
	}

	while (srm_read(&c, 1) && c != '"') {
	    for (i = 1; i < 15; i++)
		if (pal[i] == c) {
		    c = i;
		    break;
		}

	    mask = 0x80 >> (x & 7);
	    if (c & 1)
		s1[len + (x >> 3)] |= mask;
	    if (c & 2)
		s2[len + (x >> 3)] |= mask;
	    if (c & 4)
		s4[len + (x >> 3)] |= mask;
	    if (c & 8)
		s8[len + (x >> 3)] |= mask;

	    if (++x >= 640) {
		x = 0;

		if (y < 480)
		    len += 80;
		++y;
	    }
	}
    }

    srm_close();

    set_palette(0, (background >> 16), (background >> 8) & 63, background & 63);
    set_palette(15, (foreground >> 16), (foreground >> 8) & 63, foreground & 63);

    set_palette(0x11, (border >> 16), (border >> 8) & 63, border & 63);

    return (1);
}

int
vga16_begin(void)
{
    if (vga_inited)
	return (1);

    if (!*image)
	srm_strcpy(image, "/boot/srm/splash.xpm");

    saved_videomode = set_videomode(0x12);
    if (!read_image()) {
	set_videomode(saved_videomode);
	return (0);
    }

    font8x16 = (unsigned char*)get_font();

    cursorWidth = 8;
    cursorHeight = 16;

    set_int1c_handler();

    view_x0 = sx0;
    view_y0 = sy0;
    view_x1 = sx1;
    view_y1 = sy1;

	normal_color = A_NORMAL;
	highlight_color = A_REVERSE;

    return (vga_inited = 1);
}

void
vga16_end(void)
{
    if (vga_inited) {
	unset_int1c_handler();
	set_videomode(saved_videomode);
	vga_inited = 0;
	no_cursor = 0;
    }

    sx0 = view_x0;
    sy0 = view_y0;
    sx1 = view_x1;
    sy1 = view_y1;
    view_x0 = 0;
    view_x1 = 80;
    view_y0 = 0;
    view_y1 = 25;
}

void
vga16_cls(void)
{
    int i;
    unsigned char *mem, *s1, *s2, *s4, *s8;

    SetXY(view_x0, view_y0);

    mem = (unsigned char*)VIDEO;
    s1 = (unsigned char*)VSHADOW1;
    s2 = (unsigned char*)VSHADOW2;
    s4 = (unsigned char*)VSHADOW4;
    s8 = (unsigned char*)VSHADOW8;

    for (i = 0; i < 80 * 30; i++)
	text[i] = ' ';

    BitMask(0xff);

    /* plano 1 */
    MapMask(1);
    srm_memcpy(mem, s1, 38400);

    /* plano 2 */
    MapMask(2);
    srm_memcpy(mem, s2, 38400);

    /* plano 3 */
    MapMask(4);
    srm_memcpy(mem, s4, 38400);

    /* plano 4 */
    MapMask(8);
    srm_memcpy(mem, s8, 38400);

    MapMask(15);

    if (no_cursor) {
	no_cursor = 0;
	set_int1c_handler();
    }
}

void
vga16_gotoxy(int x, int y)
{
    cursor(0);

    SetXY(x, y);

    cursor(1);
}

static void
scroll(void)
{
    int i, j;

    if (no_scroll)
	return;

    no_scroll = 1;

    for (j = view_y0 + 1; j < view_y1; j++) {
	gotoxy(view_x0, j - 1);
	for (i = view_x0; i < view_x1; i++)
	    putchar(text[j * 80 + i]);
    }

    gotoxy(view_x0, view_y1 - 1);
    for (i = view_x0; i < view_x1; i++)
	putchar(' ');

    SetXY(view_x0, view_y1 - 1);

    no_scroll = 0;
}

void
vga16_putchar(int ch)
{
    ch &= 0xff;

    cursor(0);

    if (ch == '\n') {
	SetXY(view_x0, fonty);
	if (fonty + 1 < view_y1)
	    SetXY(view_x0, fonty + 1);
	else
	    scroll();
	cursor(1);
	return;
    }
    else if (ch == '\r') {
	SetXY(view_x0, fonty);
	cursor(1);
	return;
    }

    text[fonty * 80 + fontx] = ch;

    cursor(0);

    if ((fontx + 1) >= view_x1) {
	SetXY(view_x0, fonty);
	if (fonty + 1 < view_y1)
	    SetXY(view_x0, fonty + 1);
	else
	    scroll();
    }
    else
	SetXY(fontx + 1, fonty);

    cursor(1);
}

int
vga16_getxy()
{
    return ((fontx << 8) | fonty);
}

void
vga16_nocursor()
{
    if (!no_cursor) {
	no_cursor = 1;
	unset_int1c_handler();
	cursor(0);
    }
}

void
vga16_set_attrib(int attrib)
{
    text[fonty * 80 + fontx] &= 0x00ff;
    if (attrib & 0xf0)
	text[fonty * 80 + fontx] |= 0x100;
    cursor(0);
}
#endif

/* This is the starting function in C.  */
void
cmain (void)
{
  int config_len, menu_len, num_entries;
  char *config_entries, *menu_entries;
  char *kill_buf = (char *) KILL_BUF;

#ifdef VGA16
  /* Make sure it points to a valid entry */
  display = &display_entries[0];

  if (ASCII_CHAR(checkkey()) == 0x1b)
    vga_disabled = 1;
#endif

  /* Initialize the environment for restarting Stage 2.  */
  srm_setjmp (restart_env);
  
  /* Initialize the kill buffer.  */
  *kill_buf = 0;

  /* Never return.  */
  for (;;)
    {
      int is_opened = 0;
      int is_preset = 0;
      
      auto_fill = 1;
      config_len = 0;
      menu_len = 0;
      num_entries = 0;
      config_entries = (char *) mbi.drives_addr + mbi.drives_length;
      menu_entries = (char *) MENU_BUF;
      init_config ();

      /* Here load the configuration file.  */

#ifdef SRM_UTIL
      if (use_config_file)
#endif /* SRM_UTIL */
	{
	  is_opened = srm_open (config_file);
	  errnum = ERR_NONE;
	  if (! is_opened)
	    is_opened = is_preset = open_preset_menu ();
	}
      
      if (is_opened)
	{
	  /* STATE 0:  Before any title command.
	     STATE 1:  In a title command.
	     STATE >1: In a entry after a title command.  */
	  int state = 0, prev_config_len = 0, prev_menu_len = 0;
	  char *cmdline;

	  cmdline = (char *) CMDLINE_BUF;
	  while (get_line_from_config (cmdline, NEW_HEAPSIZE, ! is_preset))
	    {
	      struct builtin *builtin;

	      /* Get the pointer to the builtin structure.  */
	      builtin = find_command (cmdline);
	      errnum = 0;
	      if (! builtin)
		/* Unknown command. Just skip now.  */
		continue;

	      if (builtin->flags & BUILTIN_TITLE)
		{
		  char *ptr;

		  /* the command "title" is specially treated.  */
		  if (state > 1)
		    {
		      /* The next title is found.  */
		      num_entries++;
		      config_entries[config_len++] = 0;
		      prev_menu_len = menu_len;
		      prev_config_len = config_len;
		    }
		  else
		    {
		      /* The first title is found.  */
		      menu_len = prev_menu_len;
		      config_len = prev_config_len;
		    }

		  /* Reset the state.  */
		  state = 1;

		  /* Copy title into menu area.  */
		  ptr = skip_to (1, cmdline);
		  while ((menu_entries[menu_len++] = *(ptr++)) != 0)
		    ;
		}
	      else if (! state)
		{
		  /* Run a command found is possible.  */
		  if (builtin->flags & BUILTIN_MENU)
		    {
		      char *arg = skip_to (1, cmdline);
		      (builtin->func) (arg, BUILTIN_MENU);
		      errnum = 0;
		    }
		  else
		    /* Ignored.  */
		    continue;
		}
	      else
		{
		  char *ptr = cmdline;

		  state++;
		  /* Copy config file data to config area.  */
		  while ((config_entries[config_len++] = *ptr++) != 0)
		    ;
		}
	    }

	  if (state > 1)
	    {
	      /* Finish the last entry.  */
	      num_entries++;
	      config_entries[config_len++] = 0;
	    }
	  else
	    {
	      menu_len = prev_menu_len;
	      config_len = prev_config_len;
	    }

	  menu_entries[menu_len++] = 0;
	  config_entries[config_len++] = 0;
	  srm_memmove (config_entries + config_len, menu_entries, menu_len);
	  menu_entries = config_entries + config_len;

	  /* Check if the default entry is present. Otherwise reset
	     it to fallback if fallback is valid, or to DEFAULT_ENTRY 
	     if not.  */
	  if (default_entry >= num_entries)
	    {
	      if (fallback_entry < 0 || fallback_entry >= num_entries)
		default_entry = 0;
	      else
		default_entry = fallback_entry;
	    }

	  if (is_preset)
	    close_preset_menu ();
	  else
	    srm_close ();
	}

#ifdef VGA16
      if (display_idx >= 0)
	display = &display_entries[display_idx];
      else if (!vga_disabled) {
	display = &display_entries[1];
      }
      if (!viewport_set)
	view_y1 = 30;
      sx0 = view_x0;
      sy0 = view_y0;
      sx1 = view_x1;
      sy1 = view_y1;

      if (!display->Begin || !(*display->Begin)())
	display = &display_entries[0];

      if (!vga_inited) {
	sx0 = sy0 = view_x0 = view_y0 = 0;
	sx1 = view_x1 = 80;
	sy1 = view_y1 = 25;	  
      }
#endif

      if (! num_entries)
	{
	  /* If no acceptable config file, goto command-line, starting
	     heap from where the config entries would have been stored
	     if there were any.  */
	  enter_cmdline (config_entries, 1);
	}
      else
	{
	  /* Run menu interface.  */
	  run_menu (menu_entries, config_entries, num_entries,
		    menu_entries + menu_len, default_entry);
	}
    }
}
