/* Time-stamp: <2005-08-09 19:17:46 poser> */

/*
 * Copyright (C) 1993-2003 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static char rcsid[] = "$Id: misc.c,v 1.10 2003/11/20 02:51:49 poser Exp poser $";

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifdef INTLIZE
#include <locale.h>
#include <libintl.h>
#else
#define gettext(x) (x)
#endif
#include <wchar.h>
#include <errno.h>
#include "unicode.h"
#include "exitcode.h"
#include "retcodes.h"
#include "ex_codes.h"
#include "retcodes.h"
#include "defs.h"

void
DumpExclusions(unsigned char *exct,long exctsize,FILE *fp)
{
  long i;
  unsigned char flags;
	
  fprintf(fp,gettext("Characters excluded:\n"));
  if(exctsize > 1) printf("\n");
  for(i = 0L;i < exctsize; i++){
    if(exct[i]){
      flags = exct[i];
      fprintf(fp,"\t0x%04X",i);
      if(flags & EX_INITIAL) fprintf(fp,gettext("\tinitial"));
      if(flags & EX_MEDIAL) fprintf(fp,gettext("\tmedial"));
      if(flags & EX_FINAL) fprintf(fp,gettext("\tfinal"));
      fprintf(fp,"\n");
    }
  }
}


/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table.
 */

int
Exclude(KEYTYPE *str, int len, wchar_t *exct)
{
  int i;
  int cnt;

  cnt = 0;
  if(!(exct[str[0]] & EX_INITIAL)){
    str[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++){
    if(exct[str[i]] & EX_MEDIAL) continue;
    else str[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)){
    str[cnt++] = str[len-1];
  }
  str[cnt] = ((KEYTYPE) 0); /* Null terminate */
  return(cnt);
}

void
KeyCopy(KEYTYPE *t,KEYTYPE *s,int len)
{
  register int i;
  for(i = 0;i < len;i++) *t++ = *s++;
}

void
RevKeyCopy(KEYTYPE *t,KEYTYPE *s,int len)
{
  register int i;
	
  s += (len-2);
	
  for(i = 0;i < len-1;i++) *t++ = *s--;
  *t= (KEYTYPE) 0;
}

void
CopyCommandLine(FILE *fp,int ac, char **av)
{
  int i;
	
  for(i=0; i < ac-1; i++){
    fprintf(fp,"%s ",av[i]);
  }
  fprintf(fp,"%s\n",av[ac-1]);
}

/* 
 * This produces a value that is suitable for ordering dates and times.
 * It is NOT accurate for computing differences between dates and times
 * because the conversion of dates to days is not exact. For simplicity's
 * sake it treats years as containing 366 days and months as all containing
 * 31 days.
 */
#define DTITYPE unsigned long
#define SECONDSINDAY (60*60*24)
int
GetISO8601Key(wchar_t *field, double *key)
{
  unsigned int year = 0;
  unsigned int month =0;
  unsigned int day = 0;
  unsigned int hour = 0;
  unsigned int minute = 0;
  unsigned int second = 0;
  UTF8 *cfield;

  extern unsigned int u8len (UTF8 *);

  DTITYPE Days;
  DTITYPE DaySeconds;
  int f1, f2;

  f1=swscanf(field,L"%4u-%2u-%2uT%2u:%2u.%2u", &year,&month,&day,&hour,&minute,&second);
  if(f1 != 6) {
    f2=swscanf(field,L"%4u-%2u-%2uT%2u:%2u", &year,&month,&day,&hour,&minute,&second);
    if(f2 != 5) return(ERROR);
  }

  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  if(hour < 0 || minute < 0 || second < 0) return (ERROR);
  if(hour > 24) return(ERROR);
  if(minute > 60 || second > 60) return (ERROR);

  Days = (DTITYPE) ( (366 * (DTITYPE) year) + (31 * month) + day);
  DaySeconds = (DTITYPE) (((((DTITYPE) hour * 60) + (DTITYPE) minute) * 60) + (DTITYPE) second);
  *key = (double) ( ( (double)Days * (double)SECONDSINDAY) + (double)DaySeconds);
  return(SUCCESS);
}

/* Given a date string, write the equivalent double into key. */

int
GetDateKey(wchar_t *field, double *key, struct ymdinfo *ymd)
{
  long int num[3];
  double month;
  double day;
  wchar_t *sepptr1;
  wchar_t *sepptr2;

  extern int errno;

  sepptr1 = wcschr(field,ymd->sep1);
  sepptr2 = wcsrchr(field,ymd->sep2);
  if( (sepptr1 == NULL) || (sepptr2 == NULL) ) {
    return(ERROR);
  }
  *sepptr1 = L'\0';
  *sepptr2 = L'\0';
  num[0] = wcstol(field,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  num[1] = wcstol(sepptr1+1,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  num[2] = wcstol(sepptr2+1,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  month = (double) num[ymd->m];
  day = (double) num[ymd->d];
  fprintf(stderr,"year = %d month = %d day = %d\n",num[ymd->y],num[ymd->m],num[ymd->d]);
  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  *key = (double) ((366 * (double) num[ymd->y]) + (31 * month) + day);
  return(SUCCESS);
}


/* Given a time string, write the equivalent double into key. */

int
GetTimeKey(wchar_t *field, double *key)
{
  int hours;
  int minutes;
  int seconds;
  UTF8 *cfield;

  extern UTF8 * ws2u8(wchar_t *);
  extern unsigned int u8len (UTF8 *);

  cfield = ws2u8(field);
  seconds = 0;
  if(field[u8len(cfield)-3] == '.'){
    if(sscanf((char *)cfield,"%d:%d.%d",&hours,&minutes,&seconds) != 3) {
      free(cfield);
      return(ERROR);
    }
  }
  else{
    if(sscanf((char *)cfield,"%d:%d",&hours,&minutes) != 2) {
      free(cfield);
      return(ERROR);
    }
  }
  free(cfield);
  if(hours < 0 || minutes < 0 || seconds < 0) return (ERROR);
  if(hours > 24) return(ERROR);
  if(minutes > 60 || seconds > 60) return (ERROR);
	
  *key = (double) (((((double) hours * 60) + (double) minutes) * 60) + (double) seconds);
  return(SUCCESS);
}

void
IdentifySelf(FILE *fp)   
{
  extern char progname[];
  extern char version[];
  extern char* compdate;
	
  fprintf(fp,"%s %s\n",progname,version);
  fprintf(fp,gettext("Compiled %s\n"),compdate);
}

FILE *
OpenFile(char *file,char *mode,char *pgname)
{
  FILE *fp;
  extern FILE *fopen();
   
  if((fp = fopen(file,mode)) != NULL) return fp;
  else{
    switch(*mode){
    case 'r': 
      fprintf(stderr,gettext("%s: can't open file %s for reading.\n"),pgname,file);
      break;
    case 'w':
      fprintf(stderr,gettext("%s: can't open file %s for writing.\n"),pgname,file);
      break;
    default:
      fprintf(stderr,gettext("%s: can't open file %s for appending.\n"),pgname,file);
    }
    exit(OPENERROR);
  }
  /* NOTREACHED */
}


/*
 * Rewrite a string, replacing backslash escapes.
 *
 */

#define DEFAULT        0
#define READBACKSLASH  1
#define READONEDIGIT   2
#define READTWODIGITS  3

int
EvalEscapes(wchar_t *s)
{
  int state;
  wchar_t c;
  wchar_t odigit1;
  wchar_t odigit2;
  wchar_t sum;
  wchar_t *SStart;
  wchar_t *TStart;
  wchar_t *t;

  if(s == NULL) return(SUCCESS);

  /* Allocate storage for de-escaped copy of string */
	
  t = (wchar_t *) malloc(sizeof(wchar_t) * (wcslen(s)+1));
  if(t == NULL) return(ERROR);

  SStart = s;
  TStart = t;

  /* Work through string */
	
  state = 0;
  while((c = *s++) != L'\0'){
    switch(c){
    case '\\':
      if(state == READBACKSLASH){
	*t++ = L'\\'; /* Literal backslash */
	state = DEFAULT;
      }
      else if(state  == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
	state = DEFAULT;
      }
      else if(state  == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
	state = DEFAULT;
      }
      else{				/* state == DEFAULT */
	state = READBACKSLASH;
      }
      break;
    case 'n':
      if(state == READBACKSLASH){
	*t++ = L'\n';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case ' ':
      if(state == READBACKSLASH){
	*t++ = L' ';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case 't':
      if(state == READBACKSLASH){
	*t++ = L'\t';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case '0':
    case '1':
    case '2':
    case '3':
      if(state == READBACKSLASH){
	odigit1 = c;
	sum = ( (c - L'0') * 64);
	state = READONEDIGIT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    case '4':
    case '5':
    case '6':
    case '7':
      /* These are not possible leading digits */
      if(state == READBACKSLASH){
	*t++ = L'\\';
	*t++ = c;
	state = DEFAULT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    default:
      if(state == READBACKSLASH){
	*t++ = L'\\';
      }
      else if(state == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
      }
      else if(state == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
      }
      *t++ = c;
      state = DEFAULT;
      break;
    }
  }
  
  /* Null terminate copy and overwrite original */
  
  *t = L'\0';
  wcscpy(SStart,TStart);
  free((void *) TStart);
  return(SUCCESS);
}

/*
 * Given a number represented as a string, insert delimiters to break it up for
 * readability. Normally, the delimiter will be a comma which will be inserted every
 * three digits. However, the delimiter and groupsize are arguments, permitting use
 * in other locales.
 * 
 * The string is assumed to consist of digits, possibly preceded by spaces,
 * and possibly containing a decimal point, i.e.: [:space:]*[:digit:]*\.[:digit:]*
 * 
 * The delimited string is returned in freshly allocated memory, or NULL storage
 * cannot be allocated. 
 */

char * DelimitNumber(char *number, char delim, int groupsize)
{
  int newsize;			/* Characters needed for delimited string */
  int length;			/* Length of input string */
  int commas;			/* Number of delimiters to insert */
  int commacnt;			/* Number of delimiters inserted so far */
  int digits;			/* Number of digits preceding decimal point */
  int cnt;			/* Number of digits processed */
  char *new;			/* String in which to write delimited number */
  char *point;			/* Location of decimal point */
  char *pos;			/* Current position in original number */
  char *npos;			/* Current position in new number */
  char *NonSpaceNumber;		/* Location of first non-space character in number*/

  length = strlen(number);
  point = strrchr(number,'.');
  if(point == NULL) point = number+(length-1);
  else point = point -1;	/* Point now points at last digit preceding decimal */
  NonSpaceNumber = strrchr(number,040);
  if(NonSpaceNumber == NULL) NonSpaceNumber = number;
  else NonSpaceNumber+=1;
  digits = 1+point - NonSpaceNumber;
  commas = (digits-1)/groupsize;

  newsize = length + commas +1;	/* Add one for final null */
  new = (char *) malloc(newsize * sizeof(char));
  if(new){
    /* First copy right hand part of number, up to and including decimal point */
    pos = number+length+1;
    npos = new+newsize;
    for(; pos > point;) *npos-- = *pos--;

    /* Now copy lefthand part, inserting delimiters as we go */
    for(cnt=commacnt=0; pos >= NonSpaceNumber;){
      *npos-- = *pos--;
      cnt++;
      if( (cnt % groupsize == 0) && (commacnt++ < commas )) *npos-- = delim;
    }
    while(pos >= number) *npos-- = *pos--; /* Copy leading spaces, if any */
  }
  return new;
}

/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table. This version differs from the main Exclude
 * function in operating on chars and in not overwriting its input.
 */

int
ExcludeChar(wchar_t *str, wchar_t **out, int len, wchar_t *exct)
{
  int i;
  int cnt;
  static wchar_t *OutStr;

  cnt = 0;
  if(OutStr != NULL) free((void *) OutStr);
  OutStr = (wchar_t *) malloc ( (size_t) (len+1) * sizeof(wchar_t));
  if (OutStr == NULL) {
    fprintf(stderr,gettext("ExcludeChar: out of memory\n"));
    exit(OUTOFMEMORY);
  }
  if(!(exct[str[0]] & EX_INITIAL) ){
    OutStr[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++) {
    if(exct[str[i]] & EX_MEDIAL) continue;
    else OutStr[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)) {
    OutStr[cnt++] = str[len-1];
  }
  OutStr[cnt] = ((wchar_t) 0); /* Null terminate */
  *out = OutStr;
  return(cnt);
}
