/* 2006-03-14 SMS.
   VMS-specific SCSI code for MTOOLS.

   Note that the privileges DIAGNOSE and PHY_IO or LOG_IO are required
   to use the IO$_DIAGNOSE function, which means that these features
   will be available only to an appropriately privileged user, unless
   the executable is linked /NOTRACEBACK and INSTALLed with the
   appropriate privileges.

   Old documentation suggests that these features require use of the
   generic SCSI driver (GK), but in reasonably modern VMS versions, the
   standard SCSI disk driver (DK) seems to work.  (DQ, too???)
*/

#include <errno.h>
#include <stdio.h>
#include <string.h>

/* Work around /NAMES = AS_IS problems on VAX. */

#ifdef __VAX
# ifndef sys$assign
#  define sys$assign SYS$ASSIGN
# endif /* ndef sys$assign */
# ifndef sys$dassgn
#  define sys$dassgn SYS$DASSGN
# endif /* ndef sys$dassgn */
# ifndef sys$qiow
#  define sys$qiow SYS$QIOW
# endif /* ndef sys$qiow */
#endif /* def VAX */

#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>
#include <starlet.h>
#include <stsdef.h>

/* Use <iosbdef.h> if available.  Otherwise declare IOSB here. */

#if !defined( __VAX) && (__CRTL_VER >= 70000000)
#include <iosbdef.h>
#else /* !defined( __VAX) && __CRTL_VER >= 70000000 */
#pragma __member_alignment __save
typedef struct _iosb {
#pragma __nomember_alignment
        unsigned short int iosb$w_status;       /* Final I/O status   */
        unsigned int iosb$l_bcnt;               /* 32-bit byte count    */
        unsigned int iosb$w_dev_depend_high;    /* 16-bit dev dependent */
    } IOSB;
#pragma __member_alignment __restore
#endif /* !defined( __VAX) && (__CRTL_VER >= 70000000) */

#include "scsi.h"


/* 2006-03-14 SMS.
 *
 * scsi_max_length()
 *
 *    Unchanged from Mtools code.
 */

int scsi_max_length( void)
{
#ifdef OS_linux
        return 8;
#else
        return 255;
#endif
}


/* 2006-03-14 SMS.
 *
 * scsi_close()
 *
 *    Deassign the channel to a SCSI device.
 */

int scsi_close( int fd)
{
    unsigned short dev_chan;
    int sts;

    /* Extract channel from fd.  (High 16 bits available for flags.) */
    dev_chan = (fd& 0xffff);

    sts = sys$dassgn( dev_chan);
    if ((sts& STS$M_SEVERITY) != STS$K_SUCCESS)
    {
        /* Not success.  Set errno (and friend).  Return failure code. */
        errno = EVMSERR;
        vaxc$errno = sts;
        sts = -1;
    }
    else
    {
        sts = 0;
    }

    return sts;                         /* 0 = success, -1 = failure. */
}


/* 2006-03-14 SMS.
 *
 * scsi_open()
 *
 *    Assign the channel to a SCSI device.
 *
 *    Note that the return value (int, fd) is larger than the actual VMS
 *    channel (unsigned short, dev_chan), so that the upper 16 bits sre
 *    used to hold the mode flags, which are currently ignored.
 *    (The "extra_data" construct is currently unused.)
 */

int scsi_open( const char *name, int flag, int mode, void **extra_data)
{
    unsigned short dev_chan;
    int fd;
    int sts;

    struct dsc$descriptor_s dev_name_dscr =
     { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, NULL };

    dev_name_dscr.dsc$a_pointer = (char *) name;
    dev_name_dscr.dsc$w_length = strlen( name);

    sts = sys$assign( &dev_name_dscr, &dev_chan, 0, 0);
    if ((sts& STS$M_SEVERITY) != STS$K_SUCCESS)
    {
        fd = -1;
        errno = EVMSERR;
        vaxc$errno = sts;
    }
    else
    {
        fd = dev_chan| ((mode& 0x7fff)<< 16);
    }

    return fd;                          /* >=0 = success, -1 = failure. */
}


/* 2006-03-14 SMS.
 *
 * scsi_cmd()
 *
 *    Send a command to a SCSI device.
 *
 *    Note that open mode flags are hidden in the upper 16 bits of the
 *    (int) fd argument, but these are currently ignored.  A check that
 *    the proper mode is permitted for the Read/Write "mode" argument
 *    here could be added.
 */

int scsi_cmd( int fd,                   /* Channel. */
              unsigned char *cdb,       /* SCSI command buffer. */
              int clen,                 /* SCSI command length (6?) */
              scsi_io_mode_t mode,      /* Read/Write. */
	      void *data,               /* Data buffer. */
              size_t len,               /* Data buffer length. */
              void *extra_data)         /* Extra data (unused). */
{
    unsigned short dev_chan;
    int sts;
    struct _iosb iosb;

    /* VMS generic SCSI command descriptor. */
    struct
    {
        int opcode;             /* Op code.  (1 = pass through.) */
        int flags;              /* Flags.  See SCSI_FLAG_* macros, below. */
        void *cmd_addr;         /* Command buffer. */
        int cmd_len;            /* Command buffer length. */
        void *data_addr;        /* Data buffer. */
        int data_len;           /* Data buffer length. */
        int pad_len;            /* Pad. */
        int ph_ch_timeout;      /* Phase change timeout.  (0 = use default.) */
        int disc_timeout;       /* Disconnect timeout.  (0 = use default.) */
        int reserved[ 6];       /* Reserved. */
    } scsi_dscr;

/* SCSI command flag masks. */
#define SCSI_FLAG_M_DIR_IN 0x0001       /* Direction = in. */
#define SCSI_FLAG_M_DIS 0x0002          /* Allow disconnect. */
#define SCSI_FLAG_M_SYN 0x0004          /* Use synchronous mode, if ok. */
#define SCSI_FLAG_M_DPR 0x0008          /* Disable port retry. */

    /* Fill in the SCSI command descriptor. */
    scsi_dscr.opcode = 1;                       /* Pass through. */
    scsi_dscr.flags = SCSI_FLAG_M_DIS|          /* Allow disconnect. */
                      SCSI_FLAG_M_SYN;          /* Sync mode, if ok. */
    if (mode == SCSI_IO_READ)
    {
        scsi_dscr.flags |= SCSI_FLAG_M_DIR_IN;  /* Direction.  (0 = out.) */
    }
    scsi_dscr.cmd_addr = cdb;                   /* Command buffer. */
    scsi_dscr.cmd_len = clen;                   /* Command buffer length. */
    scsi_dscr.data_addr = data;                 /* Data buffer. */
    scsi_dscr.data_len = len;                   /* Data buffer length. */
    scsi_dscr.pad_len = 0;                      /* Pad. */
    scsi_dscr.ph_ch_timeout = 0;                /* Phase change timeout. */
    scsi_dscr.disc_timeout = 0;                 /* Disconnect timeout. */
    scsi_dscr.reserved[ 0] = 0;                 /* Reserved. */
    scsi_dscr.reserved[ 1] = 0;
    scsi_dscr.reserved[ 2] = 0;
    scsi_dscr.reserved[ 3] = 0;
    scsi_dscr.reserved[ 4] = 0;
    scsi_dscr.reserved[ 5] = 0;

    /* Extract channel from fd.  (High 16 bits available for flags.) */
    dev_chan = (fd& 0xffff);

    /* Send the SCSI command.  */
    sts = sys$qiow( 0,                          /* Event flag. */
                    dev_chan,                   /* Channel. */
                    IO$_DIAGNOSE,               /* Function code. */
                    &iosb,                      /* I/O status block. */
                    0,                          /* AST address. */
                    0,                          /* AST parameter. */
                    &scsi_dscr,                 /* P1 = SCSI descr addr. */
                    sizeof( scsi_dscr),         /* P2 = size of P1. */
                    0, 0, 0, 0);                /* P3-P6 not used. */

    /* In case anyone should ever care, the SCSI status should be in the
       high 8 bits of iosb.iosb$w_dev_depend_high.  (But what could go
       wrong?)
    */

    if ((sts& STS$M_SEVERITY) == STS$K_SUCCESS)
    {
        /* If initial success, get the final status from the IOSB. */
        sts = iosb.iosb$w_status;
    }

    if ((sts& STS$M_SEVERITY) != STS$K_SUCCESS)
    {
        /* Not success.  Set errno (and friend).  Return failure code. */
        errno = EVMSERR;
        vaxc$errno = sts;
        sts = -1;
    }
    else
    {
        sts = 0;
    }

    return sts;                         /* 0 = success, -1 = failure. */
}

