/*
 *    VMS supplement for "wget".
 *
 *======================================================================
 *
 *       decc_init()
 *
 *    On non-VAX systems, uses LIB$INITIALIZE to set a collection of C
 *    RTL features without using the DECC$* logical name method.
 *
 *----------------------------------------------------------------------
 *
 *       set_ods5_dest()
 *
 *    Sets a global flag ("ods5_dest") according to the file system type
 *    of the destination device.
 *
 *----------------------------------------------------------------------
 *
 *       ods2_conform()
 *
 *   Simplifies a fancy URL-derived file name into an ODS2-compatible
 *   file name.
 *    
 *----------------------------------------------------------------------
 *
 *       utime()
 *
 *    VMS C RTL before V7.3 lacks utime().  In V7.3, utime() sets only
 *    the modified (revised) date, not the created date of a file.
 *
 *    UNIX "ls -l" reports the modified time.  VMS "DIRECTORY /DATE"
 *    reports the creation time.  Reconciling these in FTP DIR reports
 *    is non-trivial.
 *
 *    UNIX utime() sets revision and access times.  VMS does not always
 *    maintain access times, so this utime() replacement sets the
 *    creation and revision times to the specified revision (or
 *    creation?) time.  Any access time is ignored.
 *
 *----------------------------------------------------------------------
 *
 *       getpwuid()
 *
 *    VMS C RTL before V7.0 lacks getpwuid().
 *
 *----------------------------------------------------------------------
 *
 */

#include "vms.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unixlib.h>

#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <rms.h>
#include <stsdef.h>
#include <lib$routines.h>
#include <starlet.h>

/* Use <iosbdef.h> if available.  Otherwise declare IOSB here. */

#if !defined( __VAX) && (__CRTL_VER >= 70000000)
#include <iosbdef.h>
#else /* __CRTL_VER >= 70000000 */
typedef struct _iosb {
        unsigned short int iosb$w_status; /* Final I/O status   */
        unsigned short int iosb$w_bcnt; /* 16-bit byte count    */
        unsigned int iosb$l_dev_depend; /* 32-bit dev dependent */
    } IOSB;
#endif /* !defined( __VAX) && (__CRTL_VER >= 70000000) */

/* Ugly work-around for bad type in VAX <atrdef.h>. */

#ifdef __VAX
#define UWA (unsigned int)
#else /* def __VAX */
#define UWA
#endif /* def __VAX */

#include "config.h"
#include "wget.h"

/*--------------------------------------------------------------------*/

/* Global storage. */

/*    Flag for an ODS5 destination directory. */

int ods5_dest = -1;

/*    Flag to sense if decc_init() was called. */

int decc_init_done = -1;

/*--------------------------------------------------------------------*/

#if !defined( __VAX) && (__CRTL_VER >= 70301000)

/* decc_init()

      Uses LIB$INITIALIZE to set a collection of C RTL features without
      requiring the user to define the corresponding logical names.
*/

/* Structure to hold a DECC$* feature name and its desired value. */

typedef struct
   {
   char *name;
   int value;
   } decc_feat_t;

/* Array of DECC$* feature names and their desired values. */

decc_feat_t decc_feat_array[] = {
   /* Preserve command-line case with SET PROCESS/PARSE_STYLE=EXTENDED */
 { "DECC$ARGV_PARSE_STYLE", 1 },
   /* Preserve case for file names on ODS5 disks. */
 { "DECC$EFS_CASE_PRESERVE", 1 },
   /* Enable multiple dots (and most characters) in ODS5 file names,
      while preserving VMS-ness of ";version". */
 { "DECC$EFS_CHARSET", 1 },
   /* List terminator. */
 { (char *)NULL, 0 } };

/* LIB$INITIALIZE initialization function. */

static void decc_init( void)
{
int feat_index;
int feat_value;
int feat_value_max;
int feat_value_min;
int i;
int sts;

/* Set the global flag to indicate that LIB$INITIALIZE worked. */

decc_init_done = 1;

/* Loop through all items in the decc_feat_array[]. */

for (i = 0; decc_feat_array[ i].name != NULL; i++)
   {
   /* Get the feature index. */
   feat_index = decc$feature_get_index( decc_feat_array[ i].name);
   if (feat_index >= 0)
      {
      /* Valid item.  Collect its properties. */
      feat_value = decc$feature_get_value( feat_index, 1);
      feat_value_min = decc$feature_get_value( feat_index, 2);
      feat_value_max = decc$feature_get_value( feat_index, 3);

      if ((decc_feat_array[ i].value >= feat_value_min) &&
       (decc_feat_array[ i].value <= feat_value_max))
         {
         /* Valid value.  Set it if necessary. */
         if (feat_value != decc_feat_array[ i].value)
            {
            sts = decc$feature_set_value( feat_index,
             1,
             decc_feat_array[ i].value);
            }
         }
      else
         {
         /* Invalid DECC feature value. */
         printf( " INVALID DECC FEATURE VALUE, %d: %d <= %s <= %d.\n",
          feat_value,
          feat_value_min, decc_feat_array[ i].name, feat_value_max);
         }
      }
   else
      {
      /* Invalid DECC feature name. */
      printf( " UNKNOWN DECC FEATURE: %s.\n", decc_feat_array[ i].name);
      }
   }
}

/* Get "decc_init()" into a valid, loaded LIB$INITIALIZE PSECT. */

#pragma nostandard

/* Establish the LIB$INITIALIZE PSECT, with proper alignment and
   attributes.
*/
globaldef { "LIB$INITIALIZ" } readonly _align (LONGWORD)
   int spare[ 8] = { 0 };
globaldef { "LIB$INITIALIZE" } readonly _align (LONGWORD)
   void (*x_decc_init)() = decc_init;

/* Fake reference to ensure loading the LIB$INITIALIZE PSECT. */

#pragma extern_model save
int lib$initialize(void);
#pragma extern_model strict_refdef
int dmy_lib$initialize = (int) lib$initialize;
#pragma extern_model restore

#pragma standard

#endif /* !defined( __VAX) && (__CRTL_VER >= 70301000) */

/*--------------------------------------------------------------------*/

/* set_ods5_dest()

      Sets global "ods5_dest" according to the file system type of the
      argument: 0 for ODS2, 1 for ODS5.  (No change if other/unknown or
      failure.)

      Return value:  Status from sys$getdvi().
*/

int set_ods5_dest( char *path)
{
#ifdef DVI$C_ACP_F11V5

/* Should know about ODS5 file system.  Do actual check.
   (This should be non-VAX with __CRTL_VER >= 70200000.)
*/

struct dsc$descriptor_s dev_descr =
 { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, 0 };

int acp_code;
int sts;

/* Load path argument into device descriptor.
   Default to current default device.
*/
if (path == NULL)
   {
   dev_descr.dsc$a_pointer = "SYS$DISK";
   }
else
   {
   dev_descr.dsc$a_pointer = path;
   }
dev_descr.dsc$w_length = strlen( dev_descr.dsc$a_pointer);

/* Get filesystem type code.
   (Text results for this item code have been unreliable.)
*/
sts = lib$getdvi( &((int) DVI$_ACPTYPE),
                  0,
                  &dev_descr,
                  &acp_code,
                  0,
                  0);

if ((sts & STS$M_SUCCESS) == STS$K_SUCCESS)
   {
   if (acp_code == DVI$C_ACP_F11V2)
      {
      ods5_dest = 0;
      }
   else if (acp_code == DVI$C_ACP_F11V5)
      {
      ods5_dest = 1;
      }
   }

return sts;

#else /* def DVI$C_ACP_F11V5 */

/* Too old for ODS5 file system.  Do nothing. */

return STS$K_SUCCESS;

#endif /* def DVI$C_ACP_F11V5 */
}

/*--------------------------------------------------------------------*/

/* ods2_conform()

   Replace ODS2-troublesome characters in the argument, overwriting the
   original string.  Replace "~" with "-", "@" with "$", and invalid
   dots with "_".  (Invalid dots are any in a directory name, and all
   but the last in a file name.)

   Return value: path.  (Allows a change to leave the original string
   unchanged, and to return a freshly allocated string.
*/

char *ods2_conform( char *path)
{
char *p, *prd, *prs;

/* Locate the last slash. */
prs = rindex( path, '/');
if (prs == NULL)
   {
   prs = path;
   }

/* Locate the last dot after the last slash. */
prd = rindex( prs, '.');
if (prd == NULL)
   {
   prd = prs;
   }

/* Replace with "_" all dots except the last dot after the last slash. */
for (p = path ; p != prd ; p++)
   {
   if (*p == '.')
      {
      *p = '_';
      }
   }

/* The following probably should use a look-up table. */

/* Replace with "-" all "~". */
while ((p = index( path, '~')) != NULL)
   {
   *p = '-';
   }

/* Replace with "@" all "$". */
while ((p = index( path, '@')) != NULL)
   {
   *p = '$';
   }

return path;
}

/*--------------------------------------------------------------------*/

/* Wget-private utime() code. */

/* Use long name (NAML) structure only where available.
   (This should be non-VAX with __CRTL_VER >= 70200000.)
*/

#ifdef NAML$C_BID

/* Use long name (NAML) structure. */

#define FAB$L_NAMX fab$l_naml
#define NAMX NAML
#define NAMX$C_MAXRSS NAML$C_MAXRSS
#define NAMX$B_DEV naml$l_long_dev_size
#define NAMX$L_DEV naml$l_long_dev
#define NAMX$L_ESA naml$l_long_expand
#define NAMX$B_ESL naml$l_long_expand_size
#define NAMX$B_ESS naml$l_long_expand_alloc
#define NAMX$W_FID naml$w_fid
#define NAMX$L_RSA naml$l_long_result
#define NAMX$B_RSL naml$l_long_result_size
#define NAMX$B_RSS naml$l_long_result_alloc
#define CC$RMS_NAMX cc$rms_naml

#else /* def NAML$C_BID */

/* Use short name (NAM) structure. */

#define FAB$L_NAMX fab$l_nam
#define NAMX NAM
#define NAMX$C_MAXRSS NAM$C_MAXRSS
#define NAMX$B_DEV nam$b_dev
#define NAMX$L_DEV nam$l_dev
#define NAMX$L_ESA nam$l_esa
#define NAMX$B_ESL nam$b_esl
#define NAMX$B_ESS nam$b_ess
#define NAMX$W_FID nam$w_fid
#define NAMX$L_RSA nam$l_rsa
#define NAMX$B_RSL nam$b_rsl
#define NAMX$B_RSS nam$b_rss
#define CC$RMS_NAMX cc$rms_nam

#endif /* def NAML$C_BID */

/*--------------------------------------------------------------------*/

/* Wget-private utime() code. */

/* Action routine for decc$to_vms(), in utime(). */

char vms_path[ NAMX$C_MAXRSS+ 1];

int set_vms_name( char *name, int type)
{
   strncpy( vms_path, name, NAMX$C_MAXRSS);
   vms_path[ NAMX$C_MAXRSS] = '\0';
   return 1;
}

/*--------------------------------------------------------------------*/

/* utime() replacement. */

int utime( const char *path, const struct utimbuf *times)
{
time_t utc_unsigned;

int chan, i;
int sts, sts2;

unsigned short int vms_num_vec_time[ 7];
unsigned int vms_abs_time[ 2];
struct tm *tms;
struct _iosb iosb_q;

/* QIOW item list used to set creation and revision dates. */

struct atrdef ut_atr[ 3] = {
 {sizeof( vms_abs_time), ATR$C_CREDATE, UWA vms_abs_time},
 {sizeof( vms_abs_time), ATR$C_REVDATE, UWA vms_abs_time},
 {0,0,0}};

/* Various RMS structures used for file access. */

struct FAB ut_fab = cc$rms_fab;
struct RAB ut_rab = cc$rms_rab;
struct NAMX ut_namx = CC$RMS_NAMX;
static struct fibdef ut_fib;

/* Device and file name buffers and their descriptors. */

static char dev_namx[ NAMX$C_MAXRSS+ 1];
char esa_namx[ NAMX$C_MAXRSS+ 1];
char rsa_namx[ NAMX$C_MAXRSS+ 1];

struct dsc$descriptor dev_dsc =
 {0, DSC$K_DTYPE_T, DSC$K_CLASS_S, dev_namx};

struct dsc$descriptor fib_dsc =
 {sizeof( ut_fib), DSC$K_DTYPE_T, DSC$K_CLASS_S, (char *) &ut_fib};

/* "wget" provides a UNIX-like path name.  With "-O", a user may provide
   a VMS-like path name.  If a slash is found in the name, assume that
   it's UNIX-like, and convert it to VMS form.  Otherwise, use it as-is.
*/

if (strchr( path, '/') != NULL)
   {
   sts = decc$to_vms( path, set_vms_name, 0, 0);
   path = vms_path;
   }

/* Install the VMS file specification into the FAB. */

ut_fab.fab$l_fna = (char *) path;
ut_fab.fab$b_fns = (unsigned char) strlen( path);

ut_fab.fab$l_dna = "";
ut_fab.fab$b_dns = 0;

/* Point the FAB to the NAMX. */

ut_fab.FAB$L_NAMX = &ut_namx;

/* Install the name buffers into the NAM. */

ut_namx.NAMX$L_ESA = esa_namx;
ut_namx.NAMX$B_ESL = 0;
ut_namx.NAMX$B_ESS = sizeof( esa_namx)- 1;

ut_namx.NAMX$L_RSA = rsa_namx;
ut_namx.NAMX$B_RSL = 0;
ut_namx.NAMX$B_RSS = sizeof( rsa_namx)- 1;

/* Convert the modification time (UTC time_t) to local "tm" time. */

tms = localtime( &(times-> modtime));

/* Move (translate) "tm" structure local time to VMS vector time. */

if (tms != NULL)
   {
   vms_num_vec_time[ 0] = tms-> tm_year+ 1900;
   vms_num_vec_time[ 1] = tms-> tm_mon+ 1;
   vms_num_vec_time[ 2] = tms-> tm_mday;
   vms_num_vec_time[ 3] = tms-> tm_hour;
   vms_num_vec_time[ 4] = tms-> tm_min;
   vms_num_vec_time[ 5] = tms-> tm_sec;
   vms_num_vec_time[ 6] = 0;  /* centiseconds */

/* Convert VMS vector time to VMS absolute time (quadword). */

   sts = lib$cvt_vectim( vms_num_vec_time, vms_abs_time);

   if (sts& STS$M_SUCCESS)
      {
/* Parse the file specification. */

      sts = sys$parse( &ut_fab, 0, 0);

      if (sts& STS$M_SUCCESS)
         {
/* Locate the file. (Gets the FID.) */

         sts = sys$search( &ut_fab, 0, 0);

         if (sts& STS$M_SUCCESS)
            {
/* Form the device name descriptor. */

            dev_dsc.dsc$w_length = ut_namx.NAMX$B_DEV;
            dev_dsc.dsc$a_pointer = (char *) ut_namx.NAMX$L_DEV;

/* Assign a channel to the disk device. */

            sts = sys$assign( &dev_dsc, &chan, 0, 0);

            if (sts& STS$M_SUCCESS)
               {
/* Move the FID (and not the DID) into the FIB. */

               memset( (void *) &ut_fib, 0, sizeof( ut_fib));

               for (i = 0; i < 3; i++)
                  {
                  ut_fib.fib$w_fid[ i] = ut_namx.NAMX$W_FID[ i];
                  ut_fib.fib$w_did[ i] = 0;
                  }

/* Prevent this QIOW from setting the revision time to now. */

               ut_fib.fib$l_acctl = FIB$M_NORECORD;

/* Set the file dates. */

               sts = sys$qiow( 0,
                               chan,
                               IO$_MODIFY,
                               &iosb_q,
                               0,
                               0,
                               &fib_dsc,
                               0,
                               0,
                               0,
                               ut_atr,
                               0);

               if (sts& STS$M_SUCCESS)
                  {
                   sts = iosb_q.iosb$w_status;
                  }
               sts2 = sys$dassgn( chan);

               if (sts& STS$M_SUCCESS)
                  {
                  sts = sts2;
                  }
               }
            }
         }
      }
   }

/* Convert successful VMS status to zero = success status.
   If failure, set errno and vaxc$errno, and return -1 = failure status.
*/

if (sts& STS$M_SUCCESS)
   {
   sts = 0;
   }
else
   {
   errno = EVMSERR;
   vaxc$errno = sts;
   sts = -1;
   }

return sts;
}

/*--------------------------------------------------------------------*/

/* Added J.Lauret 05-Dec-1999 . Copied from Mosaic distribution */

/*
 * Here is a replacement for getpwuid for VMS.  It returns pointers
 * to userid (*pw_name) and owner (*pw_gecos) only.  Other fields
 * may be added later.
 * Note that sys$getuai returns owner as a counted string.
 */

#if __CRTL_VER < 70000000

#include <uaidef.h>

static struct passwd vms_passwd;
static char vms_userid[16];
static char vms_owner[40];

struct passwd *getpwuid()
{
struct  dsc$descriptor_s
{
  unsigned short  dsc$w_length;
  unsigned char   dsc$b_dtype;
  unsigned char   dsc$b_class;
  char            *dsc$a_pointer;
} user_desc = {0, DSC$K_DTYPE_T, DSC$K_CLASS_S, NULL};

  char *t_userid, owner[40];
  int status, length;
  struct {
    short buffer_length;
    short item_code;
    int buffer_address;
    int return_length_address;
    int terminator;
  } itmlst;

#ifdef __GNUC__
  (int)t_userid = cuserid((char *) NULL);
#else
  t_userid = cuserid((char *) NULL);
#endif /* GNU C is strange, GEC */
  user_desc.dsc$w_length       = strlen(t_userid);
  user_desc.dsc$a_pointer      = t_userid;
  itmlst.buffer_length         = sizeof(owner);
  itmlst.item_code             = UAI$_OWNER;
  itmlst.buffer_address        = (int)owner;
  itmlst.return_length_address = (int)&length;
  itmlst.terminator            = 0;
  status = sys$getuai(0, 0, &user_desc, &itmlst, 0, 0, 0);
  if(status & STS$M_SUCCESS) {
    length = (int)owner[0];
    owner[length+1] = '\0';
    strcpy(vms_userid, t_userid);
    strcpy(vms_owner, &owner[1]);
  } else {
    vms_userid[0] = '\0';
    vms_owner[0] = '\0';
  }
  vms_passwd.pw_name = vms_userid;
  vms_passwd.pw_gecos = vms_owner;
  return (&vms_passwd);
}

/* Approximate localtime_r as best we can in its absence.  */
struct tm *
localtime_r (t, tp)
     const time_t *t;
     struct tm *tp;
{
  struct tm *l = localtime (t);
  if (! l)
    return 0;
  *tp = *l;
  return tp;
}

#endif	/* __CRTL_VER < 70000000 */

