#include <bios.h>
#include <stdio.h>
#include <stdlib.h>

struct pt_rec
    {
     unsigned char boot_flag;
     unsigned char start_head;
     unsigned      start_sect:6;
     unsigned      start_cylH:2;
     unsigned char start_cylL;
     unsigned char os_id;
     unsigned char end_head;
     unsigned      end_sect:6;
     unsigned      end_cylH:2;
     unsigned char end_cylL;
     long rel_sect;
     long num_sect;
    };

struct mbr
    {
     unsigned char xcode[0x1BE];
     struct pt_rec part_info[4];
     unsigned char magic_num[2];
    };

int hd=0;
int cmd_line=0;
struct mbr mbr;

extern char README[];
extern unsigned int EP;
extern unsigned int EP_S;
extern unsigned int EP_V;
extern struct mbr PTDOS;
extern struct mbr PTASM;

/*************************************/

int get_num_disks(void);

void read_mbr(int hd,struct mbr *);
void write_mbr(int hd,struct mbr *);


void command(char ch);

void save_mbr(char*);
void load_mbr(char*);
void install_xcode(char*);
void activate(int new_part_num);
void install(struct mbr *new_mbr);

void wait_enter(void);
void print_lines(int,int);
void print_part_info(void);

/*************************************/

void main(int argc, char **argv)
{
 int i;
 int ch, ch2;
 char tmp[30];

 read_mbr(hd,&mbr);

 if( argc>1 )
   {
    cmd_line=1;
    for( i=1 ; i<argc ; i++ )
       {
        ch=argv[i][0];
             if( ch=='K' || ch=='k' ) save_mbr(argv[++i]);
        else if( ch=='L' || ch=='l' ) load_mbr(argv[++i]);
        else if( ch=='X' || ch=='x' ) install_xcode(argv[++i]);
        else command(ch);
       }
    print_part_info();
   }
 else while(1)
    {
     int n=get_num_disks();
     if( n>1 ) sprintf(tmp,"HD %d",(hd+1)%n );
          else sprintf(tmp,"You have only one HD");

     printf("-------------------------------------------------------------------------------\n"
            " Partition Boot Manager.   (C) Mikhail Ranish.   Version 1.03   January 1997.\n"
            "-------------------------------------------------------------------------------\n");

     print_part_info();

     printf("-------------------------------------------------------------------------------\n\n"
            "      A)  What is Partition Table?       K) Save MBR to a file.\n"
            "      B)  Boot Manager description.      L) Load MBR from a file.\n"
            "      C)  Command line usage & notes.    X) Install XCode from file.\n"
            "      D)  Disclimer and author info.\n"
            "      S)  Install PBM option S (Select active boot partition or A:).\n"
            "      V)  Install PBM option V (Select ...  + Boot Virus Detection).\n"
            "      U)  Uninstall by installing standatd MBR, that comes with DOS.\n"
            "      N)  Next Hard Disk (%s)\n"
            "    1-4)  Activate partition.\n\n"
            "      Q)  Quit.\n\n"
            "Selection => ",tmp);
            
     while( (ch=getchar())==' ' || ch=='\n' || ch=='\r' || ch=='\t' );
     while( (ch2=getchar())!=EOF && ch2!='\n');

     if( ch==EOF || ch=='Q' || ch=='q' ) break;
     
     command(ch);
     
     printf("\n\n");
    }
}/* main */

/*****************************************************************************/

void command(char ch)
{
 char name[80];
 if( ch>='a' && ch<='z' ) ch-='a'-'A';

      if( ch=='A' ) { print_lines(16,24); wait_enter(); }
 else if( ch=='B' ) { print_lines(40,19); print_lines(11,5); wait_enter();}
 else if( ch=='C' ) { print_lines(60,24); wait_enter(); }
 else if( ch=='D' ) { print_lines(84,24); wait_enter(); }
 else if( ch=='U' ) { install(&PTDOS);       }
 else if( ch=='S' ) { EP=EP_S; install(&PTASM); }
 else if( ch=='V' ) { EP=EP_V; install(&PTASM); }
 else if( ch=='N' ) { hd=(hd+1)%get_num_disks(); read_mbr(hd,&mbr);  }
 else if( ch>='1' && ch<='4' ) { activate(ch-'0'); write_mbr(hd,&mbr); }
 else if( ch=='K' ) { printf("Enter file name to save MBR to: ");  gets(name); save_mbr(name); }
 else if( ch=='L' ) { printf("Enter file name to load MBR from: ");  gets(name); load_mbr(name); }
 else if( ch=='X' ) { printf("Enter file name to install xcode from: ");  gets(name); install_xcode(name); }
 else printf("\nInvalid command. Run program without arguments to see the menu.\n\n");
}/* choice */

/*****************************************************************************/

void activate(int new_part)
{
 int i;
 for( i=0 ; i<4 ; i++ ) 
   mbr.part_info[i].boot_flag=0;
  
 mbr.part_info[new_part-1].boot_flag=hd+0x80;
}/* activate */

/*****************************************************************************/

void install(struct mbr *new_pt)
{
 int i, ch;
 
 printf("\n\n");

 printf("Reading old partition table ...\n");
 read_mbr(hd,&mbr);

 for( i=0 ; i<0x1BE ; i++ )
   mbr.xcode[i]=new_pt->xcode[i];

 mbr.magic_num[0]=0x55;
 mbr.magic_num[1]=0xAA;

 printf("Writing new partition table ...\n");
 write_mbr(hd,&mbr);

 printf("DONE.\n\n");

 if( !cmd_line ) wait_enter();
}/* install */

/*****************************************************************************/

void save_mbr(char *name)
{
 FILE *f;
 
 printf("Reading MBR from the disk ...\n");
 read_mbr(hd,&mbr);
 printf("Writing MBR into the file ...\n");
 if( (f=fopen(name,"wb"))==0 )
   {
    fprintf(stderr,"\nError opening file \"%s\" for writing.\n",name);
    exit(1);
   }
 if( fwrite(&mbr,512,1,f)!=1 )
   {
    fprintf(stderr,"\nError writing into the file \"%s\".\n",name);
    exit(1);
   }
 fclose(f);
 printf("DONE.\n\n");

 if( !cmd_line ) wait_enter();
}/* save_mbr */

/*****************************************************************************/

void load_mbr(char *name)
{
 FILE *f;
 
 printf("Reading MBR from the file ...\n");
 if( (f=fopen(name,"rb"))==0 )
   {
    fprintf(stderr,"\nError opening file \"%s\" for reading.\n",name);
    exit(1);
   }
 if( fread(&mbr,512,1,f)!=1 )
   {
    fprintf(stderr,"\nError reading from the file \"%s\".\n",name);
    exit(1);
   }
 fclose(f);
 printf("Writing MBR onto the disk ...\n");
 write_mbr(hd,&mbr);
 printf("DONE.\n\n");

 if( !cmd_line ) wait_enter();
}/* load_mbr */

/*****************************************************************************/

void install_xcode(char *name)
{
 FILE *f;
 struct mbr buf;
 
 printf("Reading MBR from the file ...\n");
 if( (f=fopen(name,"rb"))==0 )
   {
    fprintf(stderr,"\nError opening file \"%s\" for reading.\n",name);
    exit(1);
   }
 if( fread(&buf,512,1,f)!=1 )
   {
    fprintf(stderr,"\nError reading from the file \"%s\".\n",name);
    exit(1);
   }
 fclose(f);
 install(&buf);
}/* load_mbr */

/*****************************************************************************/

char *get_os_name(unsigned char);
char *get_pt_size(long num_sect);


void print_part_info(void)
{
 int i;

 printf("Partition table for Hard Disk %d\n", hd );
 printf("------------------------------------------------   Starting   --   Ending   ---\n"
	"#  Active       System Type         Size (KB)   Side  Cyl Sect  Side  Cyl Sect\n");
//	"1   Yes   123456789_123456789_1234  1,111,111   9999 9999 9999  9999 9999 9999"
  for( i=0 ; i<4 ; i++ )
  {
    printf("%d   %-3s   %-24s %10s   %4d %4d %4d  %4d %4d %4d\n", i+1,

    		mbr.part_info[i].boot_flag ? "Yes" : "No",
    		get_os_name(mbr.part_info[i].os_id),
    		get_pt_size(mbr.part_info[i].num_sect/2),
    		mbr.part_info[i].start_head,
    		mbr.part_info[i].start_cylL+(mbr.part_info[i].start_cylH<<8),
    		mbr.part_info[i].start_sect,
    		mbr.part_info[i].end_head,
    		mbr.part_info[i].end_cylL+(mbr.part_info[i].end_cylH<<8),
    		mbr.part_info[i].end_sect );
  }/* for */

}/* print_part_info */



char *get_os_name(unsigned char id)
{
//			"123456789 123456789 1234"
 if( id==0x00 ) return  "Empty";
 if( id==0x01 ) return  "DOS 12-bit FAT";
 if( id==0x02 ) return  "XENIX root";
 if( id==0x03 ) return  "XENIX user";
 if( id==0x04 ) return  "DOS 16-bit FAT  (<32M)";
 if( id==0x05 ) return  "DOS Extended";
 if( id==0x06 ) return  "DOS 16-bit FAT (>=32M)";
 if( id==0x07 ) return  "OS/2 HPFS";
 if( id==0x08 ) return  "AIX";
 if( id==0x09 ) return  "AIX bootable / Coherent";
 if( id==0x0A ) return  "OPUS";
 if( id==0x40 ) return  "Venix";
 if( id==0x51 ) return  "Ontrack Extended";
 if( id==0x52 ) return  "Microport";
 if( id==0x63 ) return  "GNU HURD";
 if( id==0x64 ) return  "Novell";
 if( id==0x75 ) return  "PC/IX";
 if( id==0x80 ) return  "Old MINIX";
 if( id==0x81 ) return  "Linux / MINIX";
 if( id==0x82 ) return  "Linux swap";
 if( id==0x83 ) return  "Linux extended fs";
 if( id==0x93 ) return  "Amiga";
 if( id==0x94 ) return  "Amiga BBT";
 if( id==0xB7 ) return  "BSDI";
 if( id==0xB8 ) return  "Syrinx";
 if( id==0xDB ) return  "CP/M";
 if( id==0xE1 ) return  "DOS access";
 if( id==0xE3 ) return  "DOS R/O";
 if( id==0xF2 ) return  "DOS secondary";
 if( id==0xFF ) return  "BBT";
                return  "???";
}/* get_os_name */


char *get_pt_size(long num_sect)
{
 static char tmp[12];
 int i=10;	// 11,111,111
 tmp[i--]=0;	// 01 345 789

 while( num_sect!=0 )
    {
     if( i==2 || i==6 ) tmp[i--]=',';
     tmp[i--]=num_sect%10+'0';
     num_sect/=10;
    }

 return tmp+i+1;
}/* get_pt_size */


/***  Auxilary I/O functions  ***********************************************/


void wait_enter(void)
{
 int ch;
 printf("Press ENTER to continue...");
 while( (ch=getchar())!=EOF && ch!='\n' );
}/* wait_enter */


void print_lines(int start, int nlines)
{
 int i, l, j;
 char buf[80];

 for( i=0,l=1 ; l<start ; i++ )
  if( README[i]=='\r' && README[i+1]=='\n') { i++; l++; }

 for( l=0 ; l<nlines ; l++ )
    {
     j=0;
     while( !(README[i]=='\r' && README[i+1]=='\n') && j<80 )
       buf[j++]=README[i++];

     if( j<80 ) i+=2;
     buf[j]=0;
     puts(buf);
    }
}/* print_lines */


/***  System specific functions  ********************************************/


int get_num_disks(void)
{
 char far *num_hd;
 *((long*)(&num_hd))=0x00000475L;
 return *num_hd;
}/* get_num_disks */


void read_mbr(int hd, struct mbr *ptr)
{
 if( biosdisk( /*read*/ 2, hd+0x80, 0, 0, 1, 1, ptr ) != 0 )
   {
    fprintf(stderr,"There was an error reading MBR from disk %d.\n",hd);
    exit(1);
   }
}/* read_mbr */


void write_mbr(int hd, struct mbr *ptr)
{
 if( biosdisk( /*write*/ 3, hd+0x80, 0, 0, 1, 1, ptr ) != 0 )
   {
    fprintf(stderr,"There was an error writing MBR to disk %d.\n",hd);
    exit(1);
   }
}/* write_mbr */

