//   Triangulation header file
//   J. David Wendel   April 5, 1997

#ifndef tri2_h
#define tri2_h

#include<math.h>

const double mch_error = 1e-4;
const double PI = 3.14159;
const int max_pts = 60;

//  The absolute value function.
inline double abs(double x) {return (x<0.0) ? -x : x;}

//  The pt is a point that is the three coordinates in the
problem-world.
class pt {
  float x[3];
  public:
    pt(void) {x[0]=0.0; x[1]=0.0; x[2]=0.0;}
    pt(const pt &p) {x[0]=p.x[0]; x[1]=p.x[1]; x[2]=p.x[2];}
    pt(float *p) {x[0]=p[0]; x[1]=p[1]; x[2]=p[2];}
    pt(float x0, float x1, float x2) {x[0]=x0; x[1]=x1; x[2]=x2;}
    //~pt(void){}
    void operator=(const pt &p) {x[0]=p.x[0]; x[1]=p.x[1];
x[2]=p.x[2];}
    float operator[](int i){return x[i];}
    int operator==(const pt &p)
    {return abs(x[0]-p.x[0]) < mch_error &&
            abs(x[1]-p.x[1]) < mch_error &&
            abs(x[2]-p.x[2]) < mch_error;}
    friend int eq_2d(const pt &p, const pt &q)
    {return abs(q.x[0]-p.x[0]) < mch_error &&
            abs(q.x[1]-p.x[1]) < mch_error;}
    pt operator+(const pt &p)
    {return pt(x[0]+p.x[0], x[1]+p.x[1], x[2]+p.x[2]);}
    pt operator-(const pt &p)
    {return pt(x[0]-p.x[0], x[1]-p.x[1], x[2]-p.x[2]);}
    friend pt perp_2d(const pt &p) {return pt(-p.x[1], p.x[0],
p.x[2]);}
    friend double dot_2d(const pt &p, const pt &q)
    {return q.x[0]*p.x[0] + q.x[1]*p.x[1];}
};

//  A vector is made up of its two end points.
class vector {
  protected:
  pt x[2];
  public:

    vector(void){}
    vector(const vector &v) {x[0]=v.x[0]; x[1]=v.x[1];}
    vector(pt &p0, pt &p1) {x[0]=p0; x[1]=p1;}
    //~vector(void){}
    void operator=(const vector &v) {x[0]=v.x[0]; x[1]=v.x[1];}
    vector operator+(const vector &v) {return
vector(x[0]+v.x[0], x[1]+v.x[1]);}
    vector operator-(const vector &v) {return vector(x[0]-v.x[0],
x[1]-v.x[1]);}
    pt operator[](int i) {return x[i];}
    int operator<(vector &v);
};

//  A squaring function.
inline double sq(double x) {return x*x;}

//  The segment is a vector with its length.
class segment:public vector {
   double length;
   public:
      segment(void):length(0.0) {}
      segment(const segment &s) {x[0]=s.x[0]; x[1]=s.x[1];
length=s.length;}
      segment(pt &p0, pt &p1) {x[0]=p0; x[1]=p1; pt
diff=x[0]-x[1];
        length=sqrt(sq(diff[0])+sq(diff[1]));}
      segment(vector &v) {x[0]=v[0]; x[1]=v[1]; pt diff=x[0]-x[1];
        length=sqrt(sq(diff[0])+sq(diff[1]));}
      //~vector(void){}
      void operator=(const segment &s)
      {x[0]=s.x[0]; x[1]=s.x[1]; length=s.length;}
      int operator<(const segment &s){return length <
s.length;}
      friend double angle(vector &a, vector &b);
};

//  Everything needed to create a triangulation and save it to
disk.
class triangulation {
  pt *a_array;
  int a_array_size;
  segment *b_array;
  int b_array_size;
  vector *c_array;
  int c_array_size;
  vector *d_array;
  int d_array_size;
  int *conjugate_idx;
  int between(pt point, pt low, pt high);
  int intersect(vector a, segment b);
  void shell_sort_b_array(void);
  void shell_sort_d_array(void);
  public:
    void give_vertex(pt &p);
    void create_triangulation(void);
    void create_nlogn_triangulation(void);
    void save_triangulation(char *filename);
    triangulation(void);
    ~triangulation(void);
};

//  A triangle is three vertices.
struct triangle {
  pt x[3];
};

//  A class used at run time to load and search the
triangulation.
class run_time {
  vector *d_array;
  int d_array_size;
  int *conjugate_idx;
  int idx1, idx2, idx3;
  pt helo;
  double height;
  int postfix(int d_idx);
  triangle find_triangle(void);
  public:
    void load_triangulation(char *filename);
    void give_position(pt &p);
    double do_it(void);
    run_time(void);
    ~run_time(void);
};


#endif

