/* ------------------------------------ ComputeBezierPoints ------*/

   /*    Compute "numPoints" points along the cubic Bezier curve
    *    described by the four control points ("b0x", "b0y"),
    *    ("b1x", "b1y"), ("b2x", "b2y"), and ("b3x", "b3y").  Store
    *    these points into the array "vertices", which *must* be
    *    large enough to hold all of them.  "numPoints" must be at
    *    least 2.  The first and last points stored in "vertices"
    *    will be the endpoints of the curve, which, of course, are
    *    the same as the first and last control points. (cab)
    */ 

void ComputeBezierPoints(
            VertexRec vertices[], int numPoints,
            double b0x, double b0y, 
            double b1x, double b1y, 
            double b2x, double b2y, 
            double b3x, double b3y
) 

{
    double ax, ay, bx, by, cx, cy, dx, dy;
    int numSteps, i;
    double h;
    double pointX, pointY;
    double firstFDX, firstFDY;
    double secondFDX, secondFDY;
    double thirdFDX, thirdFDY;

    assert(vertices != NULL);
    assert(numPoints >= 2);

        /* Compute polynomial coefficients from Bezier points */

    ax = -b0x + 3 * b1x + -3 * b2x + b3x;
    ay = -b0y + 3 * b1y + -3 * b2y + b3y;

    bx = 3 * b0x + -6 * b1x + 3 * b2x;
    by = 3 * b0y + -6 * b1y + 3 * b2y;

    cx = -3 * b0x + 3 * b1x;
    cy = -3 * b0y + 3 * b1y;

    dx = b0x;
    dy = b0y;

        /* Set up the number of steps and step size */

    numSteps = numPoints - 1;        //    arbitrary choice
    h = 1.0 / (double) numSteps;    //    compute our step size

        /* Compute forward differences from Bezier points and "h" */

    pointX = dx;
    pointY = dy;

    firstFDX = ax * (h * h * h) + bx * (h * h) + cx * h;
    firstFDY = ay * (h * h * h) + by * (h * h) + cy * h;

    secondFDX = 6 * ax * (h * h * h) + 2 * bx * (h * h);
    secondFDY = 6 * ay * (h * h * h) + 2 * by * (h * h);

    thirdFDX = 6 * ax * (h * h * h);
    thirdFDY = 6 * ay * (h * h * h);    

        /* Compute points at each step */

    vertices[0].x = (int)pointX;
    vertices[0].y = (int)pointY;

    for (i = 0; i < numSteps; i++) {

        pointX += firstFDX;
        pointY += firstFDY;

        firstFDX += secondFDX;
        firstFDY += secondFDY;

        secondFDX += thirdFDX;
        secondFDY += thirdFDY;

        vertices[i + 1].x = (int)pointX;
        vertices[i + 1].y = (int)pointY;

    }
}
