#ifndef _TLS_INC
#define _TLS_INC

#include <windows.h>
#include <vector>

using namespace std;

class TlsBase {
	typedef vector<TlsBase*> FreeListVector;
	static DWORD flist_slot;		// TLS slot for FreeListVector pointer
	static void SetFreeList(FreeListVector *flist);
	static FreeListVector *GetFreeList();
protected:
	TlsBase();
	virtual ~TlsBase() { DeleteFromFreeList(); };
	virtual void Free()	 = 0;		// ThreadLocalStoragePtr<T> deletes T*
	void AddToFreeList();
	bool DeleteFromFreeList();
public:
	static void Cleanup();			// calls Free() for all allocated TLS vars
};

inline void TlsCleanup()	{ TlsBase::Cleanup(); }

template <class T>
class ThreadLocalStoragePtr : public TlsBase {
	DWORD tls_slot;					// TLS slot assigned to this variable
protected:
	T *Alloc();
 	void Free();
	void SetPtr(T *ptr);
	void SetPtr(const T *ptr);
	T *GetPtr();
public:
	ThreadLocalStoragePtr(const T *value = NULL);
 	virtual ~ThreadLocalStoragePtr();
 	operator T*();
	const T *operator=(const T *value);
};

template <class T> inline
ThreadLocalStoragePtr<T>::ThreadLocalStoragePtr(const T *value) {
	tls_slot = TlsAlloc();
	if ( tls_slot  == MAXDWORD ) throw GetLastError();
	if ( value ) operator=(value);
}

template <class T> inline
ThreadLocalStoragePtr<T>::~ThreadLocalStoragePtr() {
	if ( DeleteFromFreeList() ) Free();	// auto-cleanup for main thread
	if ( !TlsFree(tls_slot) ) throw GetLastError();
}

template <class T> inline
void ThreadLocalStoragePtr<T>::SetPtr(T *ptr) {
	if ( !TlsSetValue(tls_slot,ptr) ) {
		DWORD ecode = GetLastError();
		if ( ptr ) delete ptr;
		throw ecode;
	}
}

template <class T> inline
void ThreadLocalStoragePtr<T>::SetPtr(const T *ptr) {
	if ( !TlsSetValue(tls_slot,const_cast<T*>(ptr)) ) throw GetLastError();
}

template <class T> inline
T *ThreadLocalStoragePtr<T>::GetPtr()	{
	DWORD orig = GetLastError();	// TlsGetValue zeros LastError on success
	T *p = reinterpret_cast<T*>(TlsGetValue(tls_slot));
	DWORD ecode = GetLastError();
	if ( ecode ) throw ecode;		// throw exception on failure
	SetLastError(orig);				// restore the LastError code
	return p;
}

template <class T> inline
ThreadLocalStoragePtr<T>::operator T*() {
	T *p = GetPtr();				// returns NULL on first reference
	if ( p == NULL ) p = Alloc();	// allocate T if this is first reference
	return p;
}

template <class T> inline
void ThreadLocalStoragePtr<T>::Free() {
	T *p = GetPtr();
	if ( p ) delete p;
}

template <class T> inline
const T *ThreadLocalStoragePtr<T>::operator=(const T *value) {
	if ( DeleteFromFreeList() ) Free();	// deallocate old stuff
	SetPtr(value);					// store ptr value in TLS slot
	return value;
}

template <class T> inline
T *ThreadLocalStoragePtr<T>::Alloc() {
	T *newT = new T;				// allocate TLS storage
	SetPtr(newT);					// store allocated ptr in TLS slot
	AddToFreeList();				// keep track of allocated TLS storage
	return newT;
}

#endif	// _TLS_INC
