//=====================================================================
//
// Diskkey.c
//
// Copyright (C) 1997 Mark Russinovich
//
// This program dumps information from the 
// HKLM\SYSTEM\DISK\Information value.
//
//====================================================================
#include <windows.h>
#include <stdio.h>
#include <winioctl.h>
#include <conio.h>
#include "diskkey.h"

//
// Name of the DISK key and information value
//
char	DiskKeyName[] = "SYSTEM\\Disk";
char	DiskValueName[] = "Information";		 

//
// The disk information
//
PDISK_CONFIG_HEADER		DiskInformation;
char					*DiskInfoBuffer;

//
// FT-type strings
//
char PartTypes[][32] = {
	"Mirror",
	"Stripe",
	"ParityStripe",
	"VolumeSet",
	"NonFt",
	"WholeDisk"
};

//
// FT-volume states
//
char FtStates[][32] = {
	"Healthy",
	"Orphaned",
	"Regenerating",
	"Initializing",
	"SyncWithCopy"
};

//
// FT-partition states
//
char FtPartStates[][32] = {
    "FtStateOk",      
    "FtHasOrphan",  
    "FtDisabled",    
    "FtRegenerating", 
    "FtInitializing",
    "FtCheckParity",  
    "FtNoCheckData"  
};


//----------------------------------------------------------------------
//
// PrintError
//
// Formats an error message for the last error
//
//----------------------------------------------------------------------
void PrintError()
{
	char *errMsg;

	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, GetLastError(), 
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
			(LPTSTR) &errMsg, 0, NULL );
	printf("%s\n", errMsg );
	LocalFree( errMsg );
}

//----------------------------------------------------------------------
//
// DumpMember
//
// Dumps information on an FT Member
//
//----------------------------------------------------------------------
void DumpMember( PFT_MEMBER_DESCRIPTION MemberInfo )
{
	printf("      Disk [%d] Parition [%d]\n", 
			MemberInfo->LogicalNumber >> 16, 
			MemberInfo->LogicalNumber & 0xFFFF );
	printf("         State          : %s\n", FtPartStates[ MemberInfo->State ] );
	printf("\n");
}

//----------------------------------------------------------------------
//
// DumpFtInformation
//
// Dumps information on FT groups.
//
//----------------------------------------------------------------------
int DumpFtInformation( int GroupNum, PFT_DESCRIPTION FtDescription )
{
	int					i;

	printf("  [%d] FT Group of type %s\n", GroupNum, 
			PartTypes[ FtDescription->Type ] );
	printf("      State             : %s\n\n",
			FtStates[ FtDescription->FtVolumeState ] );
	for( i = 0; i < FtDescription->NumberOfMembers; i++ ) {

		DumpMember( &FtDescription->FtMemberDescription[i] );
	}
	return (sizeof( FT_DESCRIPTION ) +
		    ((FtDescription->NumberOfMembers - 1) * sizeof( FT_MEMBER_DESCRIPTION )));
}


//----------------------------------------------------------------------
//
// DumpPartition
// 
// Dumps the partition buffer.
//
//----------------------------------------------------------------------
void DumpPartition( PDISK_PARTITION PartInfo )
{
	printf("     [%d] Partition type : %s\n", PartInfo->PartitionNumber,
		PartTypes[ PartInfo->Type ]);
	if( !PartInfo->DriveLetter )
		printf("         Drive letter   : system-assigned\n" );
	else
		printf("         Drive letter   : %c\n", PartInfo->DriveLetter );
	printf("         Start offset   : %I64d MB\n", PartInfo->StartingOffset.QuadPart / (1024*1024) );
	printf("         Length         : %I64d MB\n", PartInfo->Length.QuadPart / (1024*1024) );
	if( PartInfo->Type != NonFT ) {
		printf("         FT-State       : %s\n", FtStates[ PartInfo->FtState ] );
		printf("         FT-Group       : %d\n", PartInfo->FtGroup );
		printf("         FT-Member      : %d\n", PartInfo->FtMember );
	}
	printf("\n");
}


//----------------------------------------------------------------------
//
// DumpDiskInformation
//
// Dump information about one disk.
//
//---------------------------------------------------------------------- 
int DumpDiskInformation( int DiskNum, PDISK_DESCRIPTION DiskDescription )
{
	int					i;
	PDISK_PARTITION		partInfo;
	PPARTITION_HEADER	partHeader;

	if( DiskDescription->NumberOfPartitions ) {

		partHeader = (PPARTITION_HEADER) (char *) &DiskDescription->PartitionHeader;

		
		printf("  [%d] Disk Signature    : %08x\n\n", DiskNum,
				partHeader->Signature );

		for( i = 0; i < DiskDescription->NumberOfPartitions; i++ ) {

			partInfo = (PDISK_PARTITION) ( (char *) &partHeader->Partitions +
								i * sizeof( DISK_PARTITION ));
			DumpPartition( partInfo );
		}		  
	}
	return  ( (DiskDescription->NumberOfPartitions ? sizeof( PARTITION_HEADER ): 0) + 
			( DiskDescription->NumberOfPartitions * sizeof( DISK_PARTITION )));
}


//----------------------------------------------------------------------
//
// DumpAllInformation
//
// Starts with the header, and calls sub-dumping functions.
//
//----------------------------------------------------------------------
void DumpAllInformation()
{
	PDISK_REGISTRY		diskInfo;
	char				*diskDescription;
	PFT_REGISTRY		ftInfo;
	char				*ftDescription;
	int					i;

	printf("Disk key version: %d\n", DiskInformation->Version );

	//
	// Dump the individual disk information
	//
	printf("\nDisk Information\n"
		     "----------------\n");
	diskInfo = (PDISK_REGISTRY) &DiskInfoBuffer[ DiskInformation->DiskInformationOffset ];
	diskDescription = (char *) &diskInfo->Disks[0];
	for( i = 0; i < diskInfo->NumberOfDisks; i++ ) {

		diskDescription += DumpDiskInformation( i, (PDISK_DESCRIPTION) diskDescription );
	}

	//
	// Dump Fault-tolerant group information
	//
	if( DiskInformation->FtInformationSize ) {
		printf("\nFT Information\n"
			     "--------------\n");
		ftInfo = (PFT_REGISTRY) &DiskInfoBuffer[ DiskInformation->FtInformationOffset ];
		ftDescription = (char *) &ftInfo->FtDescription[0];
		for( i = 0; i < ftInfo->NumberOfComponents; i++ ) {
		
			ftDescription += DumpFtInformation( i, (PFT_DESCRIPTION) ftDescription );
		}
	}
}



//----------------------------------------------------------------------
//
// ReadDiskInformation
//
// Just reads the disk information value.
//
//----------------------------------------------------------------------
BOOLEAN ReadDiskInformation()
{
	int		informationSize;
	DWORD	status, disktype, len;
	HANDLE	hDiskKey;

	//
	// Open the disk key
	//
	if( RegOpenKey( HKEY_LOCAL_MACHINE, DiskKeyName, &hDiskKey ) != ERROR_SUCCESS) {
		printf("Error opening DISK key: ");
		PrintError();
		return FALSE;
	}

	//
	// Read the value, allocating larger buffers if necessary
	//
	informationSize = 5000;
	len = informationSize;
	DiskInformation = (PDISK_CONFIG_HEADER) malloc( informationSize );
	while( (status = RegQueryValueEx( hDiskKey, DiskValueName, 0, &disktype, 
		(PBYTE) DiskInformation, &len )) == ERROR_MORE_DATA ) {

		free( DiskInformation );
		informationSize += 5000;
		len = informationSize;
		DiskInformation = (PDISK_CONFIG_HEADER) malloc( informationSize );
	}
	
	//
	// Did we read the value successfully?
	//
	if( status != ERROR_SUCCESS ) {

		printf("Error reading DISK\\Information value: " );
		PrintError();
		RegCloseKey( hDiskKey );
		return FALSE;
	}

	DiskInfoBuffer = (char *) DiskInformation;
	RegCloseKey( hDiskKey );
	return TRUE;
}


//----------------------------------------------------------------------
//
// main
//
// Drive the show. 
//
//----------------------------------------------------------------------
int main( int argc, char *argv[])
{
	printf("\nDiskkey V1.0 - HKLM\\SYSTEM\\DISK dump program\n");
	printf("Copyright (C) 1997 Mark Russinovich\n");
	printf("http://www.ntinternals.com\n\n");

	//
	// Read the disk key
	//
	if( !ReadDiskInformation() ) return -1;

	//
	// Dump it
	//
	DumpAllInformation();
	return 0;
}
