#include "std.h"
#pragma hdrstop
#include "registry.h"



/*******************************************************************
 *                                                                 *
 * The inscancel code, source and everything, is hereby placed in  *
 * the Public Domain. That means you may do with it whatever you   *
 * wish, including any alterations, selling, or leasing it. The    *
 * only action you may not take is claim copyright on property     *
 * placed in the Public Domain.                                    *
 *                                                                 *
 *                                 - Felix Kasza (felixk@mvps.org) *
 *                                                                 *
 *******************************************************************/



/*
inscancel has two principal modes of operation: the config mode to
ask you how to access your news server (the info is saved for later),
and cancel mode to actually issue cancel messages. In cancel mode,
the program reads the command line (and stdin, if so desired) and
parses the input into message-IDs. It contacts the configured NNTP
server, goes into reader mode, authenticates itself, and issues
ARTICLE commands to retrieve those articles. inscancel proceeds to
analyze the header, creates and formats an appropriate forged-cancel,
and attempts to post it to the server (unless directed otherwise).

inscancel is _dangerous_: to newsgroups first, because inscancel
makes it easy for complete dweebs to forge cancels and thus to
censor others; and to the cancelling dweeb second, because the Wrath
Of The Net will come down like a ton of bricks on the twit.

To give the Net at least a slight chance to track down abusers, the
program will add the X-Inscancel-Host: header to every cancel it
creates. PGP-signed cancels are planned for a later release.
*/


// constants
#define	MAXL	2048 // twice the NNTP legal limit



// typedefs
typedef const char *ccp;



// config info
class Ccfg
{
protected:
	string m_hostname;
	string m_proto;
	string m_user;
	string m_pass;
	string m_email;
	string m_path;
	int m_useauthn;
	int m_usehead;
	int m_debugnntp;
	int m_useihave;

	static char m_keyname[];

public:
	Ccfg(): m_debugnntp( 0 ), m_useauthn( 0 ), m_usehead( 0 ), m_useihave( 0 ) { }
	ccp hostname() { return m_hostname.c_str(); }
	ccp proto() { return m_proto.c_str(); }
	ccp user() { return m_user.c_str(); }
	ccp pass() { return m_pass.c_str(); }
	ccp email() { return m_email.c_str(); }
	ccp path() { return m_path.c_str(); }
	int useauthn() { return m_useauthn; }
	int usehead() { return m_usehead; }
	int useihave() { return m_useihave; }
	int debugnntp() { return m_debugnntp; }
	void setdebugnntp( int flag = 0 ) { m_debugnntp = flag; }

	void read(); // read config
	void ask(); // interactively set config
	void save(); // save config
	void dump();
};

char Ccfg::m_keyname[] = "software\\felixk\\inscancel";


void Ccfg::dump()
{
	cout << "News server name:   " << hostname() << endl;
	cout << "Port number to use: " << proto() << endl;
	cout << "Your email address: " << email() << endl;
	cout << "Authenticate:       " << ( useauthn()? "yes": "no" ) << endl;
	if ( useauthn() )
	{
		cout << "Your username:      " << user() << endl;
		cout << "Your password:      " << pass() << endl;
	}
	cout << "ARTICLE or HEAD:    " << ( usehead()? "HEAD": "ARTICLE" ) << endl;
	cout << "POST or IHAVE:      " << ( useihave()? "IHAVE": "POST" ) << endl;
	if ( useihave() )
		cout << "Your Path: entry:   " << path() << endl;
	cout << "NNTP debug flag:    " << ( m_debugnntp? "on": "off" ) << " (obviously)" << endl;
}


void Ccfg::read()
{
	DWORD havecfg;

	CRegistry r;

	if ( !r.Connect() )
		throw string( "Ccfg::read(): registry connect failed" );

	if ( !r.Create( m_keyname ) )
		throw string( "Ccfg::read(): key open failed" );

	if ( !r.GetValue( "HaveConfig", havecfg ) || !havecfg )
		throw string( "Ccfg::read(): no configuration found" );

	if ( !r.GetValue( "NntpServer", m_hostname ) )
		m_hostname = "localhost";

	if ( !r.GetValue( "Port", m_proto ) )
		m_proto = "119";

	if ( !r.GetValue( "Email", m_email ) )
		throw string( "Ccfg::read(): no email address found" );

	if ( !r.GetValue( "UseAuthentication", (DWORD&) m_useauthn ) )
		m_useauthn = 0;

	if ( m_useauthn )
	{
		if ( !r.GetValue( "Username", m_user ) )
			throw string( "Ccfg::read(): authn enabled, but no username found" );
		if ( !r.GetValue( "Password", m_pass ) )
			throw string( "Ccfg::read(): authn enabled, but no password found" );
	}

	if ( !r.GetValue( "UseHead", (DWORD&) m_usehead ) )
		m_usehead = 0;

	if ( !r.GetValue( "UseIhave", (DWORD&) m_useihave ) )
		m_useihave = 0;

	if ( !r.GetValue( "Path", m_path ) )
		m_path = "not-for-mail";

	r.Close();
}


void Ccfg::save()
{
	CRegistry r;

	if ( !r.Connect() )
		throw string( "Ccfg::save(): registry connect failed" );

	if ( !r.Create( m_keyname ) )
		throw string( "Ccfg::save(): key open failed" );

	if ( !r.SetValue( "HaveConfig", 1 ) )
		throw string( "Ccfg::save(): HaveConfig" );

	if ( !r.SetValue( "NntpServer", hostname() ) )
		throw string( "Ccfg::save(): NntpServer" );

	if ( !r.SetValue( "Port", proto() ) )
		throw string( "Ccfg::save(): Port" );

	if ( !r.SetValue( "Email", email() ) )
		throw string( "Ccfg::save(): Email" );

	if ( !r.SetValue( "UseAuthentication", useauthn() ) )
		throw string( "Ccfg::save(): UseAuthentication" );

	if ( !r.SetValue( "Username", user() ) )
		throw string( "Ccfg::save(): Username" );

	if ( !r.SetValue( "Password", pass() ) )
		throw string( "Ccfg::save(): Password" );

	if ( !r.SetValue( "UseHead", usehead() ) )
		throw string( "Ccfg::save(): UseHead" );

	if ( !r.SetValue( "UseIhave", useihave() ) )
		throw string( "Ccfg::save(): UseIhave" );

	if ( !r.SetValue( "Path", path() ) )
		throw string( "Ccfg::save(): Path" );

	r.Close();
}


void Ccfg::ask()
{
	string auth, cmd;

	cout << endl << "*** inscancel configuration ***" << endl << endl;

	cout << "News server name:       ";
	cin >> m_hostname;
	cout << "Port number to use:     ";
	cin >> m_proto;
	cout << "Your email address:     ";
	cin >> m_email;

	auth = 'x';
	while ( auth[0] != 'y' && auth[0] != 'Y' && auth[0] != 'n' && auth[0] != 'N' )
	{
		cout << "Authenticate? [Y/N]:    ";
		cin >> auth;
	}

	if ( auth[0] == 'n' || auth[0] == 'N' )
		m_useauthn = 0;
	else
	{
		m_useauthn = 1;
		cout << "Your username:          ";
		cin >> m_user;
		cout << "Your password:          ";
		cin >> m_pass;
	}

	auth = 'x';
	while ( auth[0] != 'a' && auth[0] != 'A' && auth[0] != 'h' && auth[0] != 'H' )
	{
		cout << "ARTICLE or HEAD? [A/H]: ";
		cin >> auth;
	}

	if ( auth[0] == 'h' || auth[0] == 'H' )
		m_usehead = 1;
	else
		m_usehead = 0;

	cmd = 'x';
	while ( cmd[0] != 'p' && cmd[0] != 'P' && cmd[0] != 'i' && cmd[0] != 'I' )
	{
		cout << "POST or IHAVE? [P/I]:   ";
		cin >> cmd;
	}

	if ( cmd[0] == 'i' || cmd[0] == 'I' )
	{
		m_useihave = 1;
		cout << "Your Path: entry:       ";
		cin >> m_path;
	}
	else
		m_useihave = 0;

	save();

	cout << endl << "Your settings have been saved." << endl;
}



Ccfg cfg;
int notext = 1;



// global vars? Who, _me_?
SOCKET sock; // the socket over which we talk
string inschdr; // our X-Inscancel-Host: header


// function prototypes
int main( int argc, char *argv[] );
void wserr( int rc, const char * const funcname );
int r( char *p, int ignoredbg = 0 ); // read a line from the other guy
void w( const char *const p, int ignoredbg = 0 ); // write a line to the other guy
void readfile( FILE *fp ); // read an article list
int getl( FILE *fp, char *p ); // read a line from a file
void cancelone( ccp killid ); // cancel one article



// getl() reads a line and returns -1 if end of file,
// 0 otherwise
int getl( FILE *fp, char *p )
{
	int c, cnt = 0;

	while ( ( c = getc( fp ) ) != EOF && c != '\n' && c != '\r' )
	{
		if ( c == '\r' )
			continue; // gobble up CRs
		*( p ++ ) = (char) c;
		cnt ++;
		// check line length
		if ( cnt >= MAXL - 2 ) // reserve 1 for '\0', one for doubled dot
			break;
	}
	*p = '\0';

	return c == EOF? EOF: 0;
}



// rdc() returns the next character from the socket.
char rdc()
{
	static char buf[MAXL];
	static int nbuf = 0; // chars in buffer
	static char *bufp; // pointer to next byte to return

	while ( nbuf == 0 ) // must fill buffer?
	{
		bufp = buf;
		nbuf = recv( sock, buf, sizeof buf, 0 );
		if ( nbuf == SOCKET_ERROR )
			wserr( nbuf, "recv" );
	}

	nbuf --;
	return *( bufp ++ );
}



// r() reads a line from the socket and returns the value of the
// integer at its start
int r( char *p, int ignoredbg /* = 0 */ )
{
	char c;
	char *orig_p = p;

	// gobble up \r, \n
	while ( ( c = rdc() ) == '\r' || c == '\n' )
		;
	// the first byte
	*( p ++ ) = c;
	while ( ( c = rdc() ) != '\r' && c != '\n' )
		*( p ++ ) = c;

	*p = '\0'; // terminate received string

	if ( cfg.debugnntp() )
	{
		if ( !ignoredbg )
		{
			fputs( " -> ", stderr );
			fputs( orig_p, stderr );
			fputs( "\n", stderr );
		}
		else
		{
			if ( *orig_p == '.' && orig_p[1] == '\0' ) // article end?
				fputs( " -> [article]\n", stderr );
		}
	}

	return atoi( orig_p );
}



// w() writes a string to the other guy and terminates it with
// carriage-return./linefeed. ignoredbg, if true, suppresses debug output for article lines.
void w( const char *const p, int ignoredbg /* = 0 */ )
{
	static char crlf[] = "\r\n";

	if ( strlen( p ) != (size_t) send( sock, p, strlen( p ), 0 ) )
		wserr( 3, "send" );
	if ( 2 != send( sock, crlf, 2, 0 ) )
		wserr( 4, "send" );

	if ( cfg.debugnntp() )
	{
		if ( !ignoredbg )
		{
			fputs( "<-  ", stderr );
			fputs( p, stderr );
			fputs( "\n", stderr );
		}
		else
		{
			if ( *p == '.' && p[1] == '\0' ) // article end?
				fputs( "<-  [article]\n", stderr );
		}
	}
}



// wserr() displays winsock errors and aborts. No grace there.
void wserr( int rc, const char * const funcname )
{
	if ( rc == 0 )
		return;

	fprintf( stderr, "\nWinsock error %d [%d] returned by %s().\n"
		"Sorry, no bonus!\n", rc, WSAGetLastError(), funcname );
	WSACleanup();
	exit( rc );
}



// initialize() handles banners, "mode reader", and authentication
void initialize()
{
	char line[MAXL];
	int rc;

	// read the banner line
	rc = r( line );
	if ( rc != 200 )
	{
		fprintf( stderr, "Bad banner: '%s'\n", line );
		exit( 2 );
	}

	if ( cfg.useauthn() )
	{
		char cmd[MAXL];

		// send authinfo user
		sprintf( cmd, "authinfo user %s", cfg.user() );
		w( cmd );
		// wait for 381
		rc = r( line );
		if ( rc != 381 )
		{
			fprintf( stderr, "Authentication (user) failed: '%s'\n", line );
			exit( 2 );
		}

		// send authinfo pass
		sprintf( cmd, "authinfo pass %s", cfg.pass() );
		w( cmd );
		// wait for 2xx
		rc = r( line );
		if ( rc < 200 || rc > 299 )
		{
			fprintf( stderr, "Authentication (password) failed: '%s'\n", line );
			exit( 2 );
		}
	}
}



// readfile() read a list article-IDs and issues cancels for each one
void readfile( FILE *fp )
{
	char id[MAXL];

	while ( 1 == fscanf( fp, " %s", id ) )
	{
		cancelone( id );
	}
}



int main( int argc, char *argv[] )
{
	int rc, i, usage = 0, port, errors = 0;
	char l[MAXL];
	unsigned long naddr;
	WSADATA wsadata;
	PHOSTENT phe;
	PSERVENT pse;
	SOCKADDR_IN addr;

	// parse switches
	if ( argc == 1 )
	{
show_usage:
		fputs( "usage: inscancel -config\n", stderr );
		fputs( "       inscancel [-d] [id ...] [-]\n", stderr );
		fputs( "\n", stderr );
		fputs( "The first version makes inscancel ask for necessary configuration\n", stderr );
		fputs( "data, such as the news server name, the canceller's email address, and\n", stderr );
		fputs( "so on. The second variant reads article-IDs from the command line, if\n", stderr );
		fputs( "any are present, and posts forged cancels for them. Angle brackets <>\n", stderr );
		fputs( "around article-IDs may optionally be supplied, but should be escaped\n", stderr );
		fputs( "to avoid their interpretation as input redirection characters. If the\n", stderr );
		fputs( "special argument \"-\" is encountered, reading continues from stdin\n", stderr );
		fputs( "until the end-of-file is encountered. Processing resumes with the next\n", stderr );
		fputs( "argument when stdin is exhausted.\n", stderr );
		fputs( "\n", stderr );
		fputs( "inscancel running in cancel mode understands one additional switch, -d.\n", stderr );
		fputs( "It causes the NNTP session to be logged to stderr.\n", stderr );
		fputs( "\n", stderr );
		fputs( "inscancel has two principal modes of operation: the config mode to\n", stderr );
		fputs( "ask you how to access your news server (the info is saved for later),\n", stderr );
		fputs( "and cancel mode to actually issue cancel messages. In cancel mode,\n", stderr );
		fputs( "the program reads the command line (and stdin, if so desired) and\n", stderr );
		fputs( "parses the input into message-IDs. It contacts the configured NNTP\n", stderr );
		fputs( "server, goes into reader mode, authenticates itself, and issues\n", stderr );
		fputs( "ARTICLE commands to retrieve those articles. inscancel proceeds to\n", stderr );
		fputs( "analyze the header, creates and formats an appropriate forged-cancel,\n", stderr );
		fputs( "and attempts to post it to the server (unless directed otherwise).\n", stderr );
		fputs( "\n", stderr );
		fputs( "inscancel is _dangerous_: to newsgroups first, because inscancel\n", stderr );
		fputs( "makes it easy for complete dweebs to forge cancels and thus to\n", stderr );
		fputs( "censor others; and to the cancelling dweeb second, because the Wrath\n", stderr );
		fputs( "Of The Net will come down like a ton of bricks on the twit.\n", stderr );
		fputs( "\n", stderr );
		fputs( "To give the Net at least a slight chance to track down abusers, the\n", stderr );
		fputs( "program will add the X-Inscancel-Host: header to every cancel it\n", stderr );
		fputs( "creates. PGP-signed cancels are planned for a later release.\n", stderr );
		fputs( "\n", stderr );
		fputs( "inscancel is\n", stderr );
		fputs( "                     MILITANTLY PUBLIC DOMAIN\n", stderr );
		return 1;
	}

	if ( argc == 2 && stricmp( argv[1], "-config" ) == 0 )
	{
		cfg.ask();
		cfg.save();
		return 0;
	}

	try
	{
		cfg.read(); // read registry
	}
	catch ( string& s )
	{
		fprintf( stderr, "Reading config: %s\n", s.c_str() );
		return 1;
	}

	// look for flags here
	for ( i = 1; i < argc; i ++ )
	{
		if ( argv[i] != NULL && argv[i][0] == '-' )
		{
			switch ( argv[i][1] )
			{
			case '\0':
				break;
			case 'd':
			case 'D':
				cfg.setdebugnntp( 1 );
				cfg.dump();
				if ( argv[i][2] == 'd' || argv[i][2] == 'D' )
					notext = 0;
				break;
			default:
				fprintf( stderr, "\"%s\" is not a valid option.\n", argv[i] );
				errors ++;
				break;
			}
		}
	}
	if ( errors )
	{
		putc( '\n', stderr );
		goto show_usage;
	}

	rc = WSAStartup( 2, &wsadata );
	wserr( rc, "WSAStartup" );

	sock = socket( AF_INET, SOCK_STREAM, 0 );
	if ( sock == INVALID_SOCKET )
		wserr( 999, "socket" );

	addr.sin_family = AF_INET;
	// try numeric IP address first (inet_addr)
	naddr = inet_addr( cfg.hostname() );
	if ( naddr != INADDR_NONE )
	{
		addr.sin_addr.s_addr = naddr;
	}
	else
	{
		phe = gethostbyname( cfg.hostname() );
		if ( phe == NULL )
			wserr( 1, "gethostbyname" );
		addr.sin_addr.s_addr = *( (unsigned long *) (phe->h_addr) );
		memcpy( (char *) &addr.sin_addr, phe->h_addr, phe->h_length );
	}

	// try numeric protocol first
	port = atoi( cfg.proto() );
	if ( port > 0 && port < 23768 )
		addr.sin_port = htons( (short) port );
	else
	{
		pse = getservbyname( cfg.proto(), "tcp" );
		if ( pse == NULL )
			wserr( 1, "getservbyname" );
		addr.sin_port = pse->s_port;
	}

	rc = connect( sock, (SOCKADDR *) &addr, sizeof addr );
	wserr( rc, "connect" );

	struct sockaddr name;
	int namelen = sizeof name;;
	rc = getsockname( sock, &name, &namelen );
	wserr( rc, "getsockname()" );
	sprintf( l, "X-Inscancel-Host: %u.%u.%u.%u", (unsigned int) (unsigned char) name.sa_data[2],
		(unsigned int) (unsigned char) name.sa_data[3], (unsigned int) (unsigned char) name.sa_data[4],
		(unsigned int) (unsigned char) name.sa_data[5] );
	inschdr = l;

	// initialize, authenticate, and so on
	initialize();

	// process input
	for ( i = 1; i < argc; i ++ )
	{
		if ( argv[i] != NULL )
		{
			if ( *argv[i] == '-' && argv[i][1] == '\0' ) // stdin?
				readfile( stdin );
			else if ( argv[i][0] == '-' ) // flag?
				;
			else
				cancelone( argv[i] );
		}
	}

	// close connection with QUIT
	w( "quit" );
	rc = r( l );
	if ( rc < 200 || rc > 299 )
		fprintf( stderr, "Bad response to QUIT: '%s'\n", l );

	// close socket
	rc = closesocket( sock );
	wserr( rc, "closesocket" );

	rc = WSACleanup();
	wserr( rc, "WSACleanup" );

	return 0;
}



struct hdr_t {
	string from, ng, date, subject, id, control, approved, path;
};



void cancelone( ccp killid ) // cancel one article
{
	ccp p;
	string id, canid, s, *lasthdr, cmd;
	hdr_t h;
	char in[MAXL];
	int rc; // NNTP result code
	int inhdr = 0; // state flag for hdr/body parsing
	SYSTEMTIME st;

	static ccp wday[] = { "Sun", "Mon", "tue", "Wed", "Thu", "Fri", "Sat" };
	static ccp mon[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
		"Aug", "Sep", "Oct", "Nov", "Dec" };

	cout << endl;
	
	// prep article ID for sending
	if ( killid[0] == '<' )
		killid ++;
	p = strchr( killid, '>' );
	if ( p == NULL )
		p = killid + strlen( killid );

	id = string( killid, p - killid );
	canid = "<cancel." + id + '>';
	id = '<' + id + '>';

	// issue article command
	s = ( cfg.usehead()? "head ": "article " ) + id;
	w( s.c_str() );
	rc = r( in );
	if ( rc < 200 || rc > 299 )
	{
		fprintf( stderr, "%s: %s\n", id.c_str(), in );
		return;
	}

	// read article, save needed headers
	inhdr = 1;
	lasthdr = NULL;
	while ( 1 )
	{
		r( in, 1 ); // get next line
		if ( in[0] == '.' && in[1] == '\0' ) // last line?
			break;

		for ( p = in; *p == ' ' || *p == '\t'; ++ p )
			;

		if ( *p == '\0' ) // an empty line?
			inhdr = 0; // now, go and skip remaining lines

		if ( inhdr ) // still in header?
		{ // if so, try to parse this line
			// first, check for continuation lines
			if ( in[0] == ' ' || in[0] == '\t' )
			{
				if ( lasthdr != NULL )
				{
					for ( p = in; *p == ' ' || *p == '\t'; ++ p )
						;
					*lasthdr += ' ' + string( p );
				}
			}
			else // not a continuation? check for From: and Newsgroups:
			{
				if ( strnicmp( in, "from:", 5 ) == 0 )
					lasthdr = &h.from;
				else if ( strnicmp( in, "newsgroups:", 11 ) == 0 )
					lasthdr = &h.ng;
				else // uninteresting header
					lasthdr = NULL; // make sure we stop picking up continuation lines

				if ( lasthdr != NULL )
					*lasthdr = in; // save this header
			}
		}
	}

	GetSystemTime( &st );
	sprintf( in, "Date: %s, %02hu %s %04hu %02hu:%02hu:%02hu GMT", wday[st.wDayOfWeek],
		st.wDay, mon[st.wMonth - 1], st.wYear, st.wHour, st.wMinute, st.wSecond );
	h.date = in;
	h.subject = "Subject: cmsg cancel " + id;
	h.id = "Message-ID: " + canid;
	h.control = "Control: cancel " + id;
	h.approved = "Approved: " + string( cfg.email() );
	h.path = (string) "Path: " + cfg.path();

	// post cancel
	if ( cfg.useihave() ) // use simulated feed?
		cmd = "ihave " + canid;
	else
		cmd = "post";

	w( cmd.c_str() );
	rc = r( in );
	if ( rc < 300 || rc > 399 )
	{
		fprintf( stderr, "%s: %s\n", id.c_str(), in );
		return;
	}

	w( h.from.c_str(), notext );
	w( h.ng.c_str(), notext );
	w( h.date.c_str(), notext );
	w( h.subject.c_str(), notext );
	w( h.id.c_str(), notext );
	w( h.control.c_str(), notext );
	w( h.approved.c_str(), notext );
	w( h.path.c_str(), notext );
	w( inschdr.c_str(), notext );
	w( "", 1 );
	w( "Cancelled by inscancel.", 1 );
	w( ".", 1 );

	rc = r( in );
	if ( rc < 200 || rc > 299 )
		fprintf( stderr, "%s: %s\n", id.c_str(), in );

	return;
}
