#include "std.h"
#pragma hdrstop
#include "registry.h"


// --- modified by felixk@mvps.org --- thanks, Sam

/*
** Author: Samuel R. Blackburn
** CI$: 76300,326
** Internet: sblackbu@erols.com
**
** You can use it any way you like as long as you don't try to sell it.
**
** Any attempt to sell WFC in source code form must have the permission
** of the original author. You can produce commercial executables with
** WFC but you can't sell WFC.
**
** Copyright, 1997, Samuel R. Blackburn
**
** $Workfile: registry.cpp $
** $Revision: 20 $
** $Modtime: 4/06/97 6:28a $
*/

static LONG _recursively_delete_all_sub_keys( HKEY key_handle, LPCTSTR key_name )
{
   HKEY child_key_handle = NULL;

   LONG return_value = 0;

   LPTSTR temporary_key_name = NULL;

   return_value = RegOpenKeyEx( key_handle, key_name, NULL, KEY_ALL_ACCESS, &child_key_handle );

   if ( return_value != ERROR_SUCCESS )
   {
      return( return_value );
   }

   temporary_key_name = new TCHAR[ MAX_PATH ];

   if ( temporary_key_name == NULL )
   {
      return( ERROR_NOT_ENOUGH_MEMORY );
   }

   return_value = RegEnumKey( child_key_handle, 0, temporary_key_name, MAX_PATH );

   while( return_value == ERROR_SUCCESS )
   {
      _recursively_delete_all_sub_keys( child_key_handle, temporary_key_name );

      return_value = RegEnumKey( child_key_handle, 0, temporary_key_name, MAX_PATH );
   }

   delete [] temporary_key_name;

   temporary_key_name = NULL;

   RegCloseKey( child_key_handle );

   return_value = RegDeleteKey( key_handle, key_name );

   return( return_value );
}

CRegistry::CRegistry()
{
   m_Initialize();
}

CRegistry::~CRegistry()
{
   if ( m_RegistryHandle != (HKEY) NULL )
   {
      Close();
   }

   m_Initialize();
}

void CRegistry::m_Initialize( void )
{
   /*
   ** Make sure everything is zeroed out
   */

   m_ClassName.remove();
   m_ComputerName.remove();
   m_KeyName.remove();
   m_RegistryName.remove();

   m_KeyHandle                    = (HKEY) NULL;
   m_ErrorCode                    = 0L;
   m_NumberOfSubkeys              = 0;
   m_LongestSubkeyNameLength      = 0;
   m_LongestClassNameLength       = 0;
   m_NumberOfValues               = 0;
   m_LongestValueNameLength       = 0;
   m_LongestValueDataLength       = 0;
   m_SecurityDescriptorLength     = 0;
   m_LastWriteTime.dwLowDateTime  = 0;
   m_LastWriteTime.dwHighDateTime = 0;

   // Thanks go to Chad Christenson (chadc@cwcinc.com) for finding
   // the bug where m_RegistryHandle was never initialized

   m_RegistryHandle               = (HKEY) NULL;
}

BOOL CRegistry::Close( void )
{
   // Thanks go to Gareth Isaac (humbert@tcp.co.uk) for finding a resource leak here

   if ( m_KeyHandle != (HKEY) NULL )
   {
      ::RegCloseKey( m_KeyHandle );
      m_KeyHandle = (HKEY) NULL;
   }

   if ( m_RegistryHandle == (HKEY) NULL )
   {
      return( TRUE );
   }

   m_ErrorCode = ::RegCloseKey( m_RegistryHandle );

   if ( m_ErrorCode == ERROR_SUCCESS )
   {
      m_RegistryHandle = (HKEY) NULL;
      m_Initialize();

      return( TRUE );
   }
   else
   {
      return( FALSE );
   }
}

BOOL CRegistry::Connect( HKEY key_to_open, LPCTSTR name_of_computer )
{
   // We were passed a pointer, do not trust it

   try
   {
      /*
      ** name_of_computer can be NULL
      */

      if ( key_to_open == (HKEY) keyClassesRoot || key_to_open == (HKEY) keyCurrentUser )
      {
         if ( name_of_computer == NULL )
         {
            m_RegistryHandle = key_to_open;
            m_ErrorCode      = ERROR_SUCCESS;
         }
         else
         {
            /*
            ** NT won't allow you to connect to these hives via RegConnectRegistry so we'll just skip that step
            */

            m_ErrorCode = ERROR_INVALID_HANDLE;
         }
      }
      else
      {
         m_ErrorCode = ::RegConnectRegistry( (char *) name_of_computer, key_to_open, &m_RegistryHandle );
      }

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         if ( name_of_computer == NULL )
         {
            TCHAR computer_name[ MAX_PATH ];
            DWORD size = MAX_PATH;

            if ( ::GetComputerName( computer_name, &size ) == TRUE )
            {
               m_ComputerName = computer_name;
            }
            else
            {
               m_ComputerName.remove();
            }
         }
         else
         {
            m_ComputerName = name_of_computer;
         }

         /*
         ** It would be nice to use a switch statement here but I get a "not integral" error!
         */

         if ( key_to_open == HKEY_LOCAL_MACHINE )
         {
            m_RegistryName = "HKEY_LOCAL_MACHINE";
         }
         else if ( key_to_open == HKEY_CLASSES_ROOT )
         {
            m_RegistryName = "HKEY_CLASSES_ROOT";
         }
         else if ( key_to_open == HKEY_USERS )
         {
            m_RegistryName = "HKEY_USERS";
         }
         else if ( key_to_open == HKEY_CURRENT_USER )
         {
            m_RegistryName = "HKEY_CURRENT_USER";
         }
         else if ( key_to_open == HKEY_PERFORMANCE_DATA )
         {
            m_RegistryName = "HKEY_PERFORMANCE_DATA";
         }
#if ( WINVER >= 0x400 )
         else if ( key_to_open == HKEY_CURRENT_CONFIG )
         {
            m_RegistryName = "HKEY_CURRENT_CONFIG";
         }
         else if ( key_to_open == HKEY_DYN_DATA )
         {
            m_RegistryName = "HKEY_DYN_DATA";
         }
#endif
         else
         {
            m_RegistryName = "Unknown";
         }

         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::Create( LPCTSTR               name_of_subkey, 
                        LPCTSTR               name_of_class,
                        CreateOptions         options, 
                        CreatePermissions     permissions, 
                        LPSECURITY_ATTRIBUTES security_attributes_p,
                        CreationDisposition * disposition_p )
{
   if ( name_of_subkey == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      DWORD disposition = 0;

      if ( name_of_class == NULL )
      {
         name_of_class = "";
      }

      if ( m_KeyHandle != (HKEY) NULL )
      {
         ::RegCloseKey( m_KeyHandle );
         m_KeyHandle = (HKEY) NULL;
      }

      m_ErrorCode = ::RegCreateKeyEx( m_RegistryHandle,
                                      name_of_subkey,
                              (DWORD) 0,
                             (char *) name_of_class,
                                      options,
                                      permissions,
                                      security_attributes_p,
                                     &m_KeyHandle,
                                     &disposition );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         if ( disposition_p != NULL )
         {
            *disposition_p = (CreationDisposition) disposition;
         }

         m_KeyName = name_of_subkey;

         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::DeleteKey( LPCTSTR name_of_key_to_delete )
{
   if ( name_of_key_to_delete == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      /*
      ** You can't delete a key given a full path. What you have to do is back up one level and then do a delete
      */

      string full_key_name = name_of_key_to_delete;

      if ( full_key_name.find( '\\' ) == os_npos )
      {
         /*
         ** User had not given us a full path so assume the name of the key he passed us
         ** is a key off of the current key
         */

         m_ErrorCode = ::_recursively_delete_all_sub_keys( m_KeyHandle, name_of_key_to_delete );
      }
      else
      {
         int last_back_slash_location = full_key_name.size() - 1;

         /*
         ** We know this loop will succeed because a back slash was found in the above if statement
         */

         while( full_key_name[ last_back_slash_location ] != '\\' )
         {
            last_back_slash_location--;
         }

         string currently_opened_key_name = m_KeyName;

         string parent_key_name = full_key_name.substr( 0, last_back_slash_location );
         string child_key_name  = full_key_name.substr( last_back_slash_location + 1 );

         /*
         ** Now we open the parent key and delete the child
         */

         if ( Open( parent_key_name.c_str() ) == TRUE )
         {
            m_ErrorCode = ::_recursively_delete_all_sub_keys( m_KeyHandle, child_key_name.c_str() );
         }
         else
         {
            m_KeyName = currently_opened_key_name;
            return( FALSE );
         }
      }

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}                  

BOOL CRegistry::DeleteValue( LPCTSTR name_of_value_to_delete )
{
   /*
   ** name_of_value_to_delete can be NULL
   */

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegDeleteValue( m_KeyHandle, (LPTSTR) name_of_value_to_delete );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::EnumerateKeys( const DWORD subkey_index, string& subkey_name, string& class_name )
{
   TCHAR subkey_name_string[ 2048 ];
   TCHAR class_name_string[ 2048 ];

   DWORD size_of_subkey_name_string = sizeof( subkey_name_string ) - 1;
   DWORD size_of_class_name_string  = sizeof( class_name_string  ) - 1;

   ::ZeroMemory( subkey_name_string, sizeof( subkey_name_string ) );
   ::ZeroMemory( class_name_string,  sizeof( class_name_string  ) );

   m_ErrorCode = ::RegEnumKeyEx( m_KeyHandle, 
                                 subkey_index, 
                                 subkey_name_string, 
                                &size_of_subkey_name_string,
                                 NULL,
                                 class_name_string,
                                &size_of_class_name_string,
                                &m_LastWriteTime );

   if ( m_ErrorCode == ERROR_SUCCESS )
   {
      subkey_name = subkey_name_string;
      class_name  = class_name_string;

      return( TRUE );
   }
   else
   {
      return( FALSE );
   }
}

BOOL CRegistry::EnumerateValues( const DWORD    value_index, 
                                 string&       name_of_value, 
                                 KeyValueTypes& type_code,
                                 LPBYTE         data_buffer,
                                 DWORD&         size_of_data_buffer )
{
   /*
   ** data_buffer and size_of_data_buffer can be NULL
   */

   DWORD temp_type_code = type_code;

   TCHAR temp_name[ 2048 ];

   ::ZeroMemory( temp_name, sizeof( temp_name ) );

   DWORD temp_name_size = sizeof( temp_name );

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegEnumValue( m_KeyHandle,
                                    value_index,
                                    temp_name,
                                   &temp_name_size,
                                    NULL,
                                   &temp_type_code,
                                    data_buffer,
                                   &size_of_data_buffer );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         type_code     = (KeyValueTypes) temp_type_code;
         name_of_value = temp_name;

         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::Flush( void )
{

   m_ErrorCode = ::RegFlushKey( m_KeyHandle );

   if ( m_ErrorCode == ERROR_SUCCESS )
   {
      return( TRUE );
   }
   else
   {
      return( FALSE );
   }
}

void CRegistry::GetClassName( string& class_name ) const
{
   class_name = m_ClassName;
}

void CRegistry::GetComputerName( string& computer_name ) const
{
   computer_name = m_ComputerName;
}

BOOL CRegistry::GetDoubleWordValue( LPCTSTR name_of_value, DWORD& return_value )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   DWORD size_of_buffer = sizeof( DWORD );

   KeyValueTypes type = typeDoubleWord;

   return( QueryValue( name_of_value, type, (LPBYTE) &return_value, size_of_buffer ) );
}

BOOL CRegistry::GetErrorCode( void ) const
{
   return( m_ErrorCode );
}

void CRegistry::GetKeyName( string& key_name ) const
{
   key_name = m_KeyName;
}

DWORD CRegistry::GetNumberOfSubkeys( void ) const
{
   return( m_NumberOfSubkeys );
}

DWORD CRegistry::GetNumberOfValues( void ) const
{
   return( m_NumberOfValues );
}

void CRegistry::GetRegistryName( string& registry_name ) const
{
   registry_name = m_RegistryName;
}

BOOL CRegistry::GetSecurity( const SECURITY_INFORMATION what_you_want_to_know,
                             PSECURITY_DESCRIPTOR       data_buffer,
                             DWORD&                     size_of_data_buffer )
{
   if ( data_buffer == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegGetKeySecurity( m_KeyHandle,
                                         what_you_want_to_know,
                                         data_buffer,
                                        &size_of_data_buffer );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::GetStringValue( LPCTSTR name_of_value, string& return_string )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   TCHAR temp_string[ 2048 ];
   DWORD size_of_buffer = 2048;

   ::ZeroMemory( temp_string, sizeof( temp_string ) );

   KeyValueTypes type = typeString;

   if ( QueryValue( name_of_value, type, (LPBYTE) temp_string, size_of_buffer ) == TRUE )
   {
      return_string = temp_string;
      return( TRUE );
   }
   else
   {
      return_string.remove();
      return( FALSE );
   }
}

BOOL CRegistry::GetValue( LPCTSTR name_of_value, DWORD& return_value )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( GetDoubleWordValue( name_of_value, return_value ) );
}

BOOL CRegistry::GetValue( LPCTSTR name_of_value, string& return_string )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( GetStringValue( name_of_value, return_string ) );
}

BOOL CRegistry::Load( LPCTSTR name_of_subkey, LPCTSTR name_of_file_containing_information )
{
   if ( name_of_subkey == NULL || name_of_file_containing_information == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passeda pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegLoadKey( m_RegistryHandle, name_of_subkey, name_of_file_containing_information );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::NotifyChange( const HANDLE             event_handle, 
                              const NotifyChangeFilter changes_to_be_reported,
                              const BOOL               this_subkey_or_all_subkeys,
                              const BOOL               wait_for_change_or_signal_event )
{
   m_ErrorCode = ::RegNotifyChangeKeyValue( m_KeyHandle,
                                            this_subkey_or_all_subkeys,
                                            changes_to_be_reported,
                                            event_handle,
                                            wait_for_change_or_signal_event );

   if ( m_ErrorCode == ERROR_SUCCESS )
   {
      return( TRUE );
   }
   else
   {
      return( FALSE );
   }
}

BOOL CRegistry::Open( LPCTSTR name_of_subkey_to_open, const CreatePermissions security_access_mask )
{
   /*
   ** name_of_subkey_to_open can be NULL
   */

   // We were passed a pointer, do not trust it

   try
   {
      if ( m_KeyHandle != (HKEY) NULL )
      {
         ::RegCloseKey( m_KeyHandle );
         m_KeyHandle = (HKEY) NULL;
      }

      m_ErrorCode = ::RegOpenKeyEx( m_RegistryHandle, name_of_subkey_to_open, NULL, security_access_mask, &m_KeyHandle );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         QueryInfo();
         m_KeyName = name_of_subkey_to_open;

         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::QueryInfo( void )
{
   TCHAR class_name[ 2048 ];

   ::ZeroMemory( class_name, sizeof( class_name ) );

   DWORD size_of_class_name = sizeof( class_name ) - 1;
   
   m_ErrorCode = ::RegQueryInfoKey( m_KeyHandle,
                                    class_name,
                                   &size_of_class_name,
                          (LPDWORD) NULL,
                                   &m_NumberOfSubkeys,
                                   &m_LongestSubkeyNameLength,
                                   &m_LongestClassNameLength,
                                   &m_NumberOfValues,
                                   &m_LongestValueNameLength,
                                   &m_LongestValueDataLength,
                                   &m_SecurityDescriptorLength,
                                   &m_LastWriteTime );

   if ( m_ErrorCode == ERROR_SUCCESS )
   {
      m_ClassName = class_name;

      return( TRUE );
   }
   else
   {
      return( FALSE );
   }
}

BOOL CRegistry::QueryValue( LPCTSTR        name_of_value, 
                            KeyValueTypes& value_type, 
                            LPBYTE         address_of_buffer, 
                            DWORD&         size_of_buffer )
{
   /*
   ** address_of_buffer and size_of_buffer can be NULL
   */

   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      DWORD temp_data_type = (DWORD) value_type;

      m_ErrorCode = ::RegQueryValueEx( m_KeyHandle,
                              (char *) name_of_value,
                                       NULL,
                                      &temp_data_type,
                                       address_of_buffer,
                                      &size_of_buffer );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         value_type = (KeyValueTypes) temp_data_type;
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::Replace( LPCTSTR name_of_subkey,
                         LPCTSTR name_of_file_with_new_data,
                         LPCTSTR name_of_backup_file )
{
   if ( name_of_subkey             == NULL ||
        name_of_file_with_new_data == NULL ||
        name_of_backup_file        == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegReplaceKey( m_KeyHandle, 
                                     name_of_subkey,
                                     name_of_file_with_new_data,
                                     name_of_backup_file );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::Restore( LPCTSTR name_of_file_holding_saved_tree, const DWORD volatility_flags )
{
   if ( name_of_file_holding_saved_tree == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegRestoreKey( m_KeyHandle,
                                     name_of_file_holding_saved_tree,
                                     volatility_flags );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::Save( LPCTSTR name_of_file_to_hold_tree, LPSECURITY_ATTRIBUTES security_attributes_p )
{
   if ( name_of_file_to_hold_tree == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegSaveKey( m_KeyHandle, name_of_file_to_hold_tree, security_attributes_p );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::SetDoubleWordValue( LPCTSTR name_of_value, DWORD value_to_write )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( SetValue( name_of_value, typeDoubleWord, (const PBYTE) &value_to_write, sizeof( DWORD ) ) );
}

BOOL CRegistry::SetSecurity( const SECURITY_INFORMATION& security_information,
                             const PSECURITY_DESCRIPTOR  security_descriptor_p )
{
   if ( security_descriptor_p == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegSetKeySecurity( m_KeyHandle, security_information, security_descriptor_p );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::SetStringValue( LPCTSTR name_of_value, const string& string_value )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( SetValue( name_of_value, typeString, (const PBYTE) string_value.c_str(), string_value.size() + 1 ) );
}

BOOL CRegistry::SetValue( LPCTSTR name_of_value, DWORD value )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( SetDoubleWordValue( name_of_value, value ) );
}

BOOL CRegistry::SetValue( LPCTSTR name_of_value, const string& string_to_write )
{
   if ( name_of_value == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   return( SetStringValue( name_of_value, string_to_write ) );
}

BOOL CRegistry::SetValue( LPCTSTR             name_of_value, 
                          const KeyValueTypes type_of_value_to_set, 
                          const PBYTE         address_of_value_data, 
                          const DWORD         size_of_data )
{
   if ( name_of_value == NULL || address_of_value_data == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegSetValueEx( m_KeyHandle,
                                     name_of_value,
                                     0,
                                     type_of_value_to_set,
                                     address_of_value_data,
                                     size_of_data );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}

BOOL CRegistry::UnLoad( LPCTSTR name_of_subkey_to_unload )
{
   if ( name_of_subkey_to_unload == NULL )
   {
      m_ErrorCode = ERROR_INVALID_PARAMETER;
      return( FALSE );
   }

   // We were passed a pointer, do not trust it

   try
   {
      m_ErrorCode = ::RegUnLoadKey( m_KeyHandle, name_of_subkey_to_unload );

      if ( m_ErrorCode == ERROR_SUCCESS )
      {
         return( TRUE );
      }
      else
      {
         return( FALSE );
      }
   }
   catch( ... )
   {
      m_ErrorCode = ERROR_EXCEPTION_IN_SERVICE;
      return( FALSE );
   }
}
