/*
 *----------------------------------------------------------------------
 *
 * Module Name:
 *	skeleton.c
 *
 * Abstract:
 *	This module contains the code for a skeleton driver that
 *	can just map a board into user space memory.
 *
 * Environment:
 *	Kernel mode
 *
 * Author:
 *	Gordon Chaffee
 *
 *----------------------------------------------------------------------
 */


#include <ntddk.h>
#include <stdarg.h>

/*
 * Skeleton includes
 */
#include "skeleton_nt.h"
#include "skeleton_dev.h"

/*
 *----------------------------------------------------------------------
 * Static Variables
 *----------------------------------------------------------------------
 */
static int nopens;

static int timeout_interval = 100;

/*
 * Internally defined routines
 */
static NTSTATUS	CloseDevice(IN PDEVICE_OBJECT devObj, IN PFILE_OBJECT fileObj);
static NTSTATUS	Dispatch(IN PDEVICE_OBJECT devObj, IN PIRP Irp);
static NTSTATUS	MapMemory(IN PDEVICE_OBJECT, IN OUT PVOID, IN ULONG, IN ULONG);
static NTSTATUS	OpenDevice(IN PDEVICE_OBJECT devObj, IN PFILE_OBJECT fileObj);
static NTSTATUS	ProbePCI(IN PDRIVER_OBJECT drvObj, IN PUNICODE_STRING regPath);
static BOOLEAN  ServiceInterrupt(IN PKINTERRUPT Interrupt,
		    IN PVOID ServiceContext);
static VOID     StartIo(IN PDEVICE_OBJECT devObj, IN PIRP Irp);
static VOID	Unload(IN PDRIVER_OBJECT);
static NTSTATUS	UnmapMemory(IN PDEVICE_OBJECT, IN OUT PVOID,
		    IN ULONG, IN ULONG);

/*
 *----------------------------------------------------------------------
 * Under NT, we can specify the parts of the code that are no longer
 * needed after initialization with the pragma alloc_text.
 *---------------------------------------------------------------------- 
 */
#if 0
#ifdef ALLOC_PRAGMA
#    pragma alloc_text(INIT,DriverEntry)
#endif
#endif

/*
 *----------------------------------------------------------------------
 * DriverEntry --
 *
 *	This routine is called at system initialization time to initialize
 *	this driver.
 *
 * Arguments:
 *	DriverObject    - Supplies the driver object.
 *	RegistryPath    - Supplies the registry path for this driver.
 *
 * Return Value:
 *	STATUS_SUCCESS          - We could initialize at least one device.
 *	STATUS_NO_SUCH_DEVICE   - We could not initialize even one device.
 *      STATUS_UNSUCCESSFUL     - For other errors?
 *----------------------------------------------------------------------
 */
NTSTATUS
DriverEntry(IN PDRIVER_OBJECT drvObj, IN PUNICODE_STRING regPath)
{
    NTSTATUS status;
    KdPrint(("SKELETON.SYS(DriverEntry): Entering\n"));

    status = ProbePCI(drvObj, regPath);

    if (NT_SUCCESS(status)) {
	/*
	 * Create dispatch points for device control, create, close.
	 */
	drvObj->MajorFunction[IRP_MJ_CREATE]		 = Dispatch;
	drvObj->MajorFunction[IRP_MJ_CLOSE]		 = Dispatch;
	drvObj->MajorFunction[IRP_MJ_DEVICE_CONTROL]	 = Dispatch;
	drvObj->MajorFunction[IRP_MJ_READ]		 = Dispatch;
	drvObj->MajorFunction[IRP_MJ_WRITE]		 = Dispatch;
	drvObj->DriverUnload				 = Unload;
	drvObj->DriverStartIo				 = StartIo;
    }

    KdPrint(("SKELETON.SYS(DriverEntry): Exiting\n"));
    return status;
}

/*
 *----------------------------------------------------------------------
 * Dispatch --
 *
 *	This routine handles all IRPs sent to this device.
 *
 * Arguments:
 *	devObj:		Pointer to the device object
 *	irp:		Pointer to an I/O request packet
 *
 * Results:
 *	Standard NT result
 *
 * Notes:
 *	We only handle ioctls to map and unmap the skeleton board.
 *----------------------------------------------------------------------
 */
static NTSTATUS
Dispatch(IN PDEVICE_OBJECT devObj, IN PIRP irp)
{
    PIO_STACK_LOCATION	irpStack;
    PVOID		ioBuf;
    ULONG		inBufLen;
    ULONG		outBufLen;
    ULONG		ioctlCode;
    NTSTATUS		status;
    PSKELETON_DEVICE	skelDev;
    ULONG		key;

    skelDev = devObj->DeviceExtension;

    /*
     * enter device mutex  to ensure one request at a time
     */
    ExAcquireFastMutex(&skelDev->IrpMutex);

    irp->IoStatus.Status      = STATUS_SUCCESS;
    irp->IoStatus.Information = 0;

    irpStack = IoGetCurrentIrpStackLocation(irp);
    ioBuf = irp->AssociatedIrp.SystemBuffer;

    switch (irpStack->MajorFunction) {
      case IRP_MJ_CREATE:
        KdPrint(("SKELETON.SYS: IRP_MJ_CREATE\n"));
	status = OpenDevice(devObj, irpStack->FileObject);
        break;

      case IRP_MJ_CLOSE:
        KdPrint(("SKELETON.SYS: IRP_MJ_CLOSE\n"));
	status = CloseDevice(devObj, irpStack->FileObject);
        break;

      case IRP_MJ_DEVICE_CONTROL:
        ioctlCode = irpStack->Parameters.DeviceIoControl.IoControlCode;
	inBufLen  = irpStack->Parameters.DeviceIoControl.InputBufferLength;
	outBufLen = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
        switch (ioctlCode) {
	  case IOCTL_SKELETON_MAP_USER_PHYSICAL_MEMORY: /* PV_MMAP */

            status = MapMemory(devObj, ioBuf, inBufLen, outBufLen);
            if (NT_SUCCESS(status)) {
                /*
                 * Success! Set the following to sizeof(PVOID) to
                 *     indicate we're passing valid data back.
                 */

                irp->IoStatus.Information = sizeof(PVOID);
            } else {
                status = STATUS_INVALID_PARAMETER;
                KdPrint(("SKELETON.SYS: memory map failed :(\n"));
            }
            break;

	  case IOCTL_SKELETON_UNMAP_USER_PHYSICAL_MEMORY:
	    status = UnmapMemory(devObj, ioBuf, inBufLen, outBufLen);
            break;

	  default:
            KdPrint(("SKELETON.SYS: unknown IRP_MJ_DEVICE_CONTROL\n"));
            status = STATUS_INVALID_PARAMETER;
            break;
        }
        break;

      default:
	KdPrint(("SKELETON.SYS: unknown Major Function\n"));
	status = STATUS_INVALID_PARAMETER;
    }

    /*
     * Don't get cute and try to use the status field of
     * the irp in the return status.  That IRP IS GONE as
     * soon as you call IoCompleteRequest.
     */
    if (status != STATUS_PENDING) {
	irp->IoStatus.Status = status;
	IoCompleteRequest(irp, IO_VIDEO_INCREMENT);
    } else {
	IoMarkIrpPending(irp);
	IoStartPacket(devObj, irp, &key, NULL);
    }

    ExReleaseFastMutex(&skelDev->IrpMutex);
    return status;
}

/*
 *----------------------------------------------------------------------
 * Unload --
 *
 *	Just delete the associated device and return
 *
 * Arguments:
 *	drvObj:		Pointer to the driver object
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
static VOID
Unload(IN PDRIVER_OBJECT drvObj)
{
    WCHAR		devLinkBuf[]  = L"\\DosDevices\\SKELETON0";
    UNICODE_STRING	devLinkUniStr;
    WCHAR		devNum;
    PDEVICE_OBJECT	devObj, nextDev;
    PSKELETON_DEVICE	skelDev;
    int			tmp;
    CM_RESOURCE_LIST	EmptyList;
    BOOLEAN		ResourceConflict;

    /*
     * For each device that is on the machine:
     *
     * 1. Delete the symbolic links
     * 2. Turn off the board interrupts and disconnect the interrupt.
     * 3. Unmap the board memory from system space.
     * 4. Unreport the resources that were assigned by HalAssignSlotResources
     * 5. Delete the device object
     */

    for (devNum = 0, devObj = drvObj->DeviceObject; devObj != NULL;
	 devObj = nextDev, devNum++) {
	devLinkBuf[sizeof(devLinkBuf) - 1] = L'0' + devNum;

	RtlInitUnicodeString(&devLinkUniStr, devLinkBuf);
	IoDeleteSymbolicLink(&devLinkUniStr);

	skelDev = devObj->DeviceExtension;
	IoDisconnectInterrupt(skelDev->KIntrObj);
	MmUnmapIoSpace(skelDev->FrameBase, skelDev->MemLength);

	/* un-report any resources used by the driver and the device */
	EmptyList.Count = 0;
	IoReportResourceUsage(NULL, drvObj, &EmptyList, sizeof(ULONG),
			      drvObj->DeviceObject, &EmptyList, sizeof(ULONG),
			      FALSE, &ResourceConflict);

	nextDev = devObj->NextDevice;
	IoDeleteDevice(devObj);
    }
    KdPrint(("SKELETON.SYS: unloading\n"));
}

/*
 *----------------------------------------------------------------------
 * StartTransferTimeout --
 *
 *	Starts a timer that can check on the DMA operation.  Hopefully,
 *	it never goes off
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
static void
StartTransferTimeout(IN PDEVICE_OBJECT devObj, int msTimeout, PVOID Ignore)
{
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;

    skelDev->TransferDone = FALSE;
    /*
     * Timer is in 100 ns units.
     */
    KeSetTimer(&skelDev->DeviceCheckTimer,
	       RtlConvertLongToLargeInteger(-msTimeout * 10000), &skelDev->TimerDpc);
    skelDev->TimerStarted = TRUE;
}

/*
 *----------------------------------------------------------------------
 * CancelTransferTimeout --
 *
 *	Remove a previously set DMA timeout timer.
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
static void
CancelTransferTimeout(PVOID Context)
{
    PDEVICE_OBJECT devObj = Context;
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;

    skelDev->TransferDone = TRUE;
}

/*
 *----------------------------------------------------------------------
 * ProgramDMAUtil
 *
 *	Utility routine that starts the DMA transfer
 *
 * Results:
 *	TRUE
 *----------------------------------------------------------------------
 */
static BOOLEAN
ProgramDMAUtil(IN PVOID Context)
{
    PDEVICE_OBJECT devObj = Context;
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;
    ULONG bufLen;
    BOOLEAN writeOp;
    ULONG toMapLen, mappedLen;
    PHYSICAL_ADDRESS physAddr;
    PVOID virtualAddr;
    PIRP irp;

    irp = (PIRP) skelDev->syncParam1;
    bufLen = skelDev->IrpBufLen;
    virtualAddr = skelDev->VirtualAddress;

    writeOp = (skelDev->OperationType == IRP_MJ_READ) ? FALSE : TRUE;

    toMapLen = bufLen;
    while (toMapLen > 0) {
	mappedLen = (toMapLen >= 4096) ? 4096 : toMapLen;
	physAddr = IoMapTransfer(NULL, irp->MdlAddress,
				 skelDev->MapRegisterBase,
				 virtualAddr, &mappedLen, writeOp);

	/*
	 * XXX: set addr on the board.  This will be different per board.
	 * Maybe you don't even have to do anything.
	 */
#if 0
	board_write(start_address) = LITTLE_ENDIAN_32(physAddr.u.LowPart);
#endif
	toMapLen -= mappedLen;
	virtualAddr = (((char *) virtualAddr) + mappedLen);
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 * ProgramDMA
 *
 *	This routine gets called back by NT when an adapter channel
 *	is free to use.  It then uses ProgramDMAUtil to start the
 *	actual transfer
 *
 * Results:
 *	
 *----------------------------------------------------------------------
 */
static IO_ALLOCATION_ACTION
ProgramDMA(IN PDEVICE_OBJECT devObj, IN PIRP irp, 
	   IN PVOID MapRegisterBase, IN PVOID Context)
{
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;

    skelDev->MapRegisterBase = MapRegisterBase;
    skelDev->syncParam1 = (ULONG) irp;
    KeSynchronizeExecution(skelDev->KIntrObj, ProgramDMAUtil, devObj);
    StartTransferTimeout(devObj, timeout_interval, NULL);

    /*
     * return a value that says we want to keep the map registers.
     */
    return DeallocateObjectKeepRegisters;
}


/*
 *----------------------------------------------------------------------
 * StartIo --
 *
 *	This gets called when we are about to start a DMA operation.
 *	This can occur because another DMA operation just completed,
 *	or it can occur because this is the first DMA operation.  Either
 *	way, we don't expect anything to interfere with its operation.
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
static VOID
StartIo(IN PDEVICE_OBJECT devObj, IN PIRP irp)
{
    PIO_STACK_LOCATION  irpStack;
    PVOID		ioBuf;
    ULONG		inBufLen;
    ULONG		outBufLen;
    ULONG		ioctlCode;
    NTSTATUS		status;
    BOOLEAN		writeOp;
    PSKELETON_DEVICE	skelDev;

    skelDev = devObj->DeviceExtension;
    irpStack = IoGetCurrentIrpStackLocation(irp);

    KdPrint(("SKELETON.SYS(StartIo): Beginning irp %p\n", irp));

    switch (irpStack->MajorFunction) {
      case IRP_MJ_READ:
      case IRP_MJ_WRITE:
	break;

      case IRP_MJ_DEVICE_CONTROL:
	ioBuf     = irp->AssociatedIrp.SystemBuffer;
	inBufLen  = irpStack->Parameters.DeviceIoControl.InputBufferLength;
	outBufLen = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
	ioctlCode = irpStack->Parameters.DeviceIoControl.IoControlCode;
        switch (ioctlCode) {
	  default:
            KdPrint(("SKELETON.SYS(StartIo): unexpected IRP_MJ_DEVICE_CONTROL\n"));
            status = STATUS_INVALID_PARAMETER;
            break;
        }

	if (status != STATUS_PENDING) {
	    irp->IoStatus.Status = status;
	    IoCompleteRequest(irp, IO_VIDEO_INCREMENT);
	    IoStartNextPacket(devObj, TRUE);
	}
	return;

      default:
	KdPrint(("SKELETON.SYS(StartIo): unexpected major function\n"));
	irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
	IoCompleteRequest(irp, IO_NO_INCREMENT);
	IoStartNextPacket(devObj, TRUE);
	return;
    }

    skelDev->OperationType = irpStack->MajorFunction;
    skelDev->IrpSystemBuffer = irp->AssociatedIrp.SystemBuffer;
    if (skelDev->OperationType == IRP_MJ_READ) {
	skelDev->IrpBufLen = irpStack->Parameters.Read.Length;
    } else {
	skelDev->IrpBufLen  = irpStack->Parameters.Write.Length;
    }
    if (skelDev->IrpBufLen == 0 || irp->MdlAddress == NULL) {
	irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
	IoCompleteRequest(irp, IO_NO_INCREMENT);
	IoStartNextPacket(devObj, TRUE);
    }
	
    skelDev->VirtualAddress = MmGetMdlVirtualAddress(irp->MdlAddress);

    if (skelDev->TimerStarted) {
	KeCancelTimer(&skelDev->DeviceCheckTimer);
	skelDev->TimerStarted = FALSE;
    }

    writeOp = (skelDev->OperationType == IRP_MJ_READ) ? FALSE : TRUE;

    KeFlushIoBuffers(irp->mdlAddress, !writeOp, TRUE);
    status = IoAllocateAdapterChannel(skelDev->AdaptorObj, devObj,
				      skelDev->DmaMapRegisters, ProgramDMA, devObj);

    KdPrint(("SKELETON.SYS(StartIo): Exiting irp %p\n", irp));
    if (!NT_SUCCESS(status)) {
	KdPrint(("SKELETON.SYS: Unable to allocate adaptor channel for DMA\n"));
	irp->IoStatus.Status = status;
	IoCompleteRequest(irp, IO_NO_INCREMENT);
	return;
    }
}

/*
 *----------------------------------------------------------------------
 * TransferDPC --
 *
 *	This routine is called at DISPATCH_LEVEL by the system at the
 *	ServiceInterrupt().
 *
 *	This routine is protected against interrupts since it was queued
 *	by an interrupt, and the next DMA related interrupt won't occur
 *	until something else happens.
 *
 *	This routine is called when a DMA transfer has not been completed.
 *	It sets everything up to continue the tranfer.
 *----------------------------------------------------------------------
 */
static VOID
TransferDPC(IN PKDPC Dpc, IN PVOID Context, IN PVOID Arg1, IN PVOID Arg2)
{
    PDEVICE_OBJECT devObj = Context;
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;
    PIRP irp;
    BOOLEAN writeOp;

    KdPrint(("SKELETON.SYS(TransferDPC): Finished irp %p\n",
	     devObj->CurrentIrp));
    CancelTransferTimeout(devObj);

    irp = devObj->CurrentIrp;
    writeOp = (skelDev->OperationType == IRP_MJ_WRITE) ? TRUE : FALSE;

    IoFlushAdapterBuffers(NULL, irp->MdlAddress,
			  skelDev->MapRegisterBase,
			  skelDev->VirtualAddress, skelDev->IrpBufLen, writeOp);

    IoFreeMapRegisters(skelDev->AdaptorObj, skelDev->MapRegisterBase, 
		       skelDev->DmaMapRegisters);

    if (skelDev->OperationType == IRP_MJ_READ) {
	KeFlushIoBuffers(irp->MdlAddress, TRUE, TRUE);
    }
    irp->IoStatus.Status = skelDev->IrpStatus;
    if (skelDev->IrpStatus == STATUS_SUCCESS) {
	irp->IoStatus.Information = skelDev->IrpBytesTransferred;
    }
    IoCompleteRequest(irp, IO_VIDEO_INCREMENT);
    IoStartNextPacket(devObj, TRUE);
    skelDev->DpcRequested = FALSE;

    return;
}

/*
 *----------------------------------------------------------------------
 * ServiceTimeoutUtil --
 *
 *	Utility routine for ServiceTimeout.  Runs code that is 
 *	sensitive to interrupts.
 *----------------------------------------------------------------------
 */
static BOOLEAN
ServiceTimeoutUtil(IN PCONTEXT Context)
{
    PSKELETON_DEVICE skelDev = (PSKELETON_DEVICE) Context;

    return TRUE;
}

/*
 *----------------------------------------------------------------------
 * ServiceTimeout --
 *
 *	Service a timeout.  Is this a routine to check on the board
 *	if nothing happens after a little while?  If so,
 *	ddk/src/multimedia/soundlib/wave.c does something similar.
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
void
ServiceTimeout(PDEVICE_OBJECT devObj)
{
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;

    KeSynchronizeExecution(skelDev->KIntrObj, ServiceTimeoutUtil, skelDev);

    skelDev->IrpStatus = STATUS_UNSUCCESSFUL;
    skelDev->IrpBytesTransferred = 0L;
    skelDev->RequestDpc = TRUE;

    KdPrint(("SKELETON.SYS: ServiceTimeout calling TransferDPC\n"));
    TransferDPC(NULL, devObj, NULL, NULL);
}

/*
 *----------------------------------------------------------------------
 * TimeoutDPC --
 *
 *	This routine gets called when a timeout occurs.  We then
 *	need to check plxDev->TransferDone to see if the transfer
 *	finished before this timer went off.  If is did, then we
 *	can just ignore this DPC call.  If not, we need to clear
 *	everything up, fail the request, and move on.
 *
 * Results:
 *	None
 *----------------------------------------------------------------------
 */
static VOID
TimeoutDPC(IN PKDPC Dpc, IN PVOID Context, IN PVOID Param1, IN PVOID Param2)
{
    PDEVICE_OBJECT devObj = Context;
    PSKELETON_DEVICE skelDev = devObj->DeviceExtension;

    skelDev->TimerStarted = FALSE;
    if (! skelDev->TransferDone) {
	/*
	 * XXX: Clean up the hardware here if necessary.
	 */
	ServiceTimeout(devObj);
    }
}

/*
 *----------------------------------------------------------------------
 * OpenDevice --
 *
 *	Open the device.  We will allow multiple opens to the device.
 *
 * Results:
 *	A standard NT result
 *----------------------------------------------------------------------
 */
static NTSTATUS
OpenDevice(IN PDEVICE_OBJECT devObj, IN PFILE_OBJECT fileObj)
{
    PSKELETON_DEVICE skelDev;

    KdPrint(("SKELETON.SYS: OpenDevice called\n"));

    skelDev = devObj->DeviceExtension;
    ++nopens;			/* inc global open */
    return STATUS_SUCCESS;
}

/*
 *----------------------------------------------------------------------
 * CloseDevice --
 *
 *	Close up device and free resources allocated by OpenDevice
 *
 * Results:
 *	A standard NT result
 *----------------------------------------------------------------------
 */
static NTSTATUS
CloseDevice(IN PDEVICE_OBJECT devObj, IN PFILE_OBJECT fileObj)
{
    PSKELETON_DEVICE skelDev;

    skelDev = devObj->DeviceExtension;
    nopens--;			/* decrement global open */
    return STATUS_SUCCESS;
}
/*
 *----------------------------------------------------------------------
 * MapMemory --
 *
 *	Given a physical address, maps this address into a user mode
 *	process's address space
 *
 * Arguments:
 *	devObj		pointer to a device object
 *	ioBuf		pointer to the I/O buffer
 *	inBufLen	input buffer length
 *	outBufLen	output buffer length
 *
 * Results:
 *	STATUS_SUCCESS
 *	STATUS_UNSUCCESSFUL
 *	STATUS_BUFFER_TOO_SMALL,
 *	(other STATUS_* as returned by kernel APIs)
 *----------------------------------------------------------------------
 */
static NTSTATUS
MapMemory(IN PDEVICE_OBJECT devObj, IN OUT PVOID ioBuf,
	  IN ULONG inBufLen, IN ULONG outBufLen)
{
    INTERFACE_TYPE     interfaceType;
    ULONG              busNumber;
    PHYSICAL_ADDRESS   physicalAddr;
    ULONG              len;
    UNICODE_STRING     physicalMemUniStr;
    OBJECT_ATTRIBUTES  objAttrs;
    HANDLE             physicalMemHandle  = NULL;
    PVOID              physicalMemSection = NULL;
    ULONG              inIoSpace, inIoSpace2;
    NTSTATUS           status;
    PHYSICAL_ADDRESS   physicalAddrBase;
    PHYSICAL_ADDRESS   physicalAddrEnd;
    PHYSICAL_ADDRESS   viewBase;
    PHYSICAL_ADDRESS   mappedLen;
    BOOLEAN            xlateBaseAddr;
    BOOLEAN            xlateEndAddr;
    PVOID              virtualAddr;
    PSKELETON_DEVICE   skelDev;

    skelDev = devObj->DeviceExtension;

    if (outBufLen < sizeof(PVOID)) {
	KdPrint(("SKELETON.SYS(MapMemory): Insufficient output buffer\n"));
	status = STATUS_BUFFER_TOO_SMALL;

	goto done;
    }

    /*
     * Get a pointer to physical memory...
     *
     * - Create the name
     * - Initialize the data to find the object
     * - Open a handle to the oject and check the status
     * - Get a pointer to the object
     * - Free the handle
     */

    RtlInitUnicodeString(&physicalMemUniStr, L"\\Device\\PhysicalMemory");

    InitializeObjectAttributes(&objAttrs, &physicalMemUniStr,
			       OBJ_CASE_INSENSITIVE, (HANDLE) NULL,
			       (PSECURITY_DESCRIPTOR) NULL);

    status = ZwOpenSection(&physicalMemHandle, SECTION_ALL_ACCESS, &objAttrs);
    if (!NT_SUCCESS(status)) {
        KdPrint(("SKELETON.SYS: ZwOpenSection failed\n"));
        goto done;
    }

    status =
	ObReferenceObjectByHandle(physicalMemHandle, SECTION_ALL_ACCESS,
				       (POBJECT_TYPE) NULL, KernelMode,
				       &physicalMemSection,
				       (POBJECT_HANDLE_INFORMATION) NULL);

    if (!NT_SUCCESS(status)) {
        KdPrint(("SKELETON.SYS: ObReferenceObjectByHandle failed\n"));
        goto close_handle;
    }

    interfaceType = PCIBus;
    busNumber     = skelDev->BusId;
    physicalAddr  = skelDev->MemStart;
    len           = skelDev->MemLength;
    inIoSpace     = skelDev->FrameMemType;
    inIoSpace2    = skelDev->FrameMemType;

    /*
     * Initialize the physical addresses that will be translated
     */

    physicalAddrEnd = RtlLargeIntegerAdd(physicalAddr,
					 RtlConvertUlongToLargeInteger(len));

    /*
     * Translate the physical addresses.
     */

    xlateBaseAddr = HalTranslateBusAddress(interfaceType, busNumber,
					   physicalAddr, &inIoSpace,
					   &physicalAddrBase);

    xlateEndAddr = HalTranslateBusAddress(interfaceType, busNumber,
					  physicalAddrEnd, &inIoSpace2,
    					  &physicalAddrEnd);

    if (!(xlateBaseAddr && xlateEndAddr)) {
        KdPrint(("SKELETON.SYS: HalTranslatephysicalAddress failed\n"));
        status = STATUS_UNSUCCESSFUL;
        goto close_handle;
    }

    /*
     * Calculate the length of the memory to be mapped
     */

    mappedLen = RtlLargeIntegerSubtract(physicalAddrEnd, physicalAddrBase);

    /*
     * If the mappedlen is zero, somthing very weird happened in the HAL
     * since the Length was checked against zero.
     */

    if (mappedLen.LowPart == 0) {
        KdPrint(("SKELETON.SYS: mappedLen.LowPart == 0\n"));
        status = STATUS_UNSUCCESSFUL;
        goto close_handle;
    }

    len = mappedLen.LowPart;

    /*
     * If the address is in io space, just return the address, otherwise
     * go through the mapping mechanism
     */

    if (inIoSpace) {
        *((PVOID *) ioBuf) = (PVOID) physicalAddrBase.LowPart;
    } else {
        /*
         * initialize view base that will receive the physical mapped
         * address after the MapViewOfSection call.
         */

        viewBase = physicalAddrBase;

        /*
         * Let ZwMapViewOfSection pick an address
         */

        virtualAddr = NULL;

        /*
         * Map the section
         */

        status =
	    ZwMapViewOfSection(physicalMemHandle, (HANDLE) -1, &virtualAddr,
			       0L, len, &viewBase, &len, ViewShare, 0,
			       PAGE_READWRITE | PAGE_NOCACHE);

        if (!NT_SUCCESS(status)) {
            KdPrint(("SKELETON.SYS: ZwMapViewOfSection failed\n"));
            goto close_handle;
        }

        /*
         * Mapping the section above rounded the physical address down to the
         * nearest 64 K boundary. Now return a virtual address that sits where
         * we want by adding in the offset from the beginning of the section.
         */

        (ULONG) virtualAddr += (ULONG)physicalAddrBase.LowPart -
                                  (ULONG)viewBase.LowPart;

        *((PVOID *) ioBuf) = virtualAddr;

    }

    status = STATUS_SUCCESS;

close_handle:
    ZwClose(physicalMemHandle);

done:
    return status;
}

/*
 *----------------------------------------------------------------------
 * UnmapMemory --
 *
 *	Unmaps board's memory from the process's address space.
 *
 * Results:
 *	STATUS_SUCCESS
 *	STATUS_BUFFER_TOO_SMALL,
 *----------------------------------------------------------------------
 */
static NTSTATUS
UnmapMemory(IN PDEVICE_OBJECT devObj, IN OUT PVOID ioBuf,
	  IN ULONG inBufLen, IN ULONG outBufLen)
{
    if (inBufLen < sizeof(PVOID)) {
	KdPrint(("SKELETON.SYS(UnmapMemory): Insufficient input buffer\n"));
	return STATUS_BUFFER_TOO_SMALL;
    }
    KdPrint(("SKELETON.SYS: memory unmapped\n"));
    return ZwUnmapViewOfSection((HANDLE) -1, *((PVOID *) ioBuf));
}

/*
 *----------------------------------------------------------------------
 * ResetBoard --
 *
 *	Does a hard reset of the board
 *
 *----------------------------------------------------------------------
 */
static VOID
ResetBoard(PSKELETON_DEVICE skelDev)
{
    PUCHAR	 		    base;

    /*
     * Reset the board
     */

    base = skelDev->FrameBase;
    *((unsigned long *)(base + 0x7f0000)) = 0;

    KeStallExecutionProcessor(500);

    /*
     * Enable the board
     */
    *((unsigned long *)(base + 0x7f0000)) = LITTLE_ENDIAN_32(1);
}

/*
 *----------------------------------------------------------------------
 * CreateDevice --
 *
 *	Create a Skeleton device
 *
 *----------------------------------------------------------------------
 */
static NTSTATUS
CreateDevice(IN PDRIVER_OBJECT drvObj, IN PUNICODE_STRING regPath,
	     ULONG busId, ULONG slotId, IN PPCI_COMMON_CONFIG pciData)
{
    PDEVICE_OBJECT		    devObj = NULL;
    WCHAR			    devNameBuf[] = L"\\Device\\Skeleton0";
    UNICODE_STRING		    devNameUniStr;
    WCHAR			    devLinkBuf[] = L"\\DosDevices\\SKELETON0";
    UNICODE_STRING		    devLinkUniStr;
    NTSTATUS			    status;
    INTERFACE_TYPE		    interfaceType;
    ULONG			    busNumber;
    PCM_RESOURCE_LIST		    resourceList;
    PCM_PARTIAL_RESOURCE_LIST	    partialResourceList;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR partialDescriptor;
    PHYSICAL_ADDRESS		    xlatedAddr;
    PHYSICAL_ADDRESS		    start;
    ULONG			    length;
    ULONG			    addressSpace;
    PSKELETON_DEVICE		    skelDev;
    ULONG			    i, j;
    BOOLEAN			    b;
    int				    found;
    DEVICE_DESCRIPTION		    devDesc;
    static WCHAR		    devNum = 0;
    

    devNameBuf[sizeof(devNameBuf) - 2] = L'0' + devNum;
    devLinkBuf[sizeof(devLinkBuf) - 2] = L'0' + devNum;
    devNum++;

    RtlInitUnicodeString(&devNameUniStr, devNameBuf);

    status = IoCreateDevice(drvObj, sizeof(SKELETON_DEVICE), &devNameUniStr,
			    FILE_DEVICE_SKELETON, 0, FALSE, &devObj);

    devObj->Flags |= DO_DIRECT_IO;

    /*
     * 1. Create dispatch points for device control, create, close.
     * 2. Create a symbolic link, i.e. a nmae that a Win32 app can
     *    specify to open the device.  If this fails, delete the
     *    device object.
     */
    if (! NT_SUCCESS(status)) {
        KdPrint(("SKELETON.SYS: IoCreateDevice failed\n"));
	return status;
    }

    RtlInitUnicodeString(&devLinkUniStr, devLinkBuf);
    status = IoCreateSymbolicLink(&devLinkUniStr, &devNameUniStr);

    if (!NT_SUCCESS(status)) {
	KdPrint(("SKELETON.SYS: IoCreateSymbolicLink failed\n"));
	goto error;
    }

    resourceList = NULL;
    status = HalAssignSlotResources(regPath, NULL, drvObj, devObj,
				    PCIBus, busId, slotId,
				    &resourceList);
    if (!NT_SUCCESS(status)) {
	KdPrint(("SKELETON.SYS: HalAssignSlotResources failed\n"));
	goto error;
    }

    /*
     * Now, we hopefully have an address for the card on the bus,
     * but who knows for sure.  We need to translate the returned
     * address and map the address space into kernel space.
     */

    skelDev = devObj->DeviceExtension;
    skelDev->BusId = busId;
    skelDev->SlotId = slotId;

    found = 0;
    for (i = 0; i < resourceList->Count; i++) {
	interfaceType = resourceList->List[i].InterfaceType;
	busNumber = resourceList->List[i].BusNumber;
	partialResourceList = &resourceList->List[i].PartialResourceList;
	for (j = 0; j < partialResourceList->Count; j++) {
	    partialDescriptor = &partialResourceList->PartialDescriptors[j];
	    if (partialDescriptor->Type == CmResourceTypeMemory) {
		addressSpace = 0; /* Memory */
		start = partialDescriptor->u.Memory.Start;
		length = partialDescriptor->u.Memory.Length;
		b = HalTranslateBusAddress(interfaceType, busNumber,
					   start,
					   &skelDev->FrameMemType,
					   &xlatedAddr);
		if (!b) {
		    KdPrint(("SKELETON.SYS: HalTranslateBusAddress failed\n"));
		    status = STATUS_UNSUCCESSFUL;
		    goto error;
		}
		skelDev->MemStart = xlatedAddr;
		skelDev->MemLength = length;

		if (skelDev->FrameMemType == 0) {
		    skelDev->FrameBase = MmMapIoSpace(xlatedAddr, length, FALSE);
		} else {
		    skelDev->FrameBase = (PUCHAR) xlatedAddr.LowPart;
		}

		found++;

	    } else if (partialDescriptor->Type == CmResourceTypeInterrupt) {
		/*
		 * Get the system interrupt vector for IoConnectInterrupt
		 */
		ULONG level    = partialDescriptor->u.Interrupt.Level;
		ULONG vector   = partialDescriptor->u.Interrupt.Vector;
		ULONG affinity = partialDescriptor->u.Interrupt.Affinity;

		skelDev->KIntrVector =
		    HalGetInterruptVector(PCIBus, busId, level, vector,
					  &skelDev->KIrql,
					  &skelDev->KIntrAffinity);
		devDesc.Version = DEVICE_DESCRIPTION_VERSION;
		devDesc.Master = TRUE;
		devDesc.ScatterGather = TRUE;
		devDesc.DemandMode = FALSE;
		devDesc.AutoInitialize = FALSE;
		devDesc.Dma32BitAddresses = TRUE;
		devDesc.IgnoreCount = FALSE;
		devDesc.Reserved1 = FALSE;
		devDesc.Reserved2 = FALSE;
		devDesc.BusNumber = busId;
		devDesc.DmaChannel = 0; /* ? */
		devDesc.InterfaceType = PCIBus;
		devDesc.DmaWidth = Width32Bits;
		devDesc.DmaSpeed = Compatible;
		devDesc.MaximumLength = 255 * 1024;
		devDesc.DmaPort = 0;
		skelDev->AdaptorObj = 
		    HalGetAdapter(&devDesc, &skelDev->DmaMapRegisters);
		found++;
	    }
	}
    }
    if (found != 2) {
	KdPrint(("SKELETON.SYS: Failed to find frame buffer address or interrupt\n"));
	status = STATUS_UNSUCCESSFUL;
	goto error;
    }

    /*
     * Enable Memory Space and Bus Master control bits
     */
    pciData->Command = 6;
    HalSetBusDataByOffset(PCIConfiguration, busId, slotId,
			  &pciData->Command,
			  offsetof(PCI_COMMON_CONFIG, Command),
			  sizeof(pciData->Command));


    ResetBoard(skelDev);

    /*
     * 1. Initialize the device mutex
     * 2. Initialize the device spin lock to protect the DPC routine
     *    for callers to SynchronizeDPC.
     * 3. Initialize the DPC data and register with Io system
     * 4. Connect the interrupt
     */
    ExInitializeFastMutex(&skelDev->IrpMutex);
    KeInitializeSpinLock(&skelDev->DeviceSpinLock);
    KeInitializeTimer(&skelDev->DeviceCheckTimer);
    KeInitializeTimer(&skelDev->StartIoTimer);
    skelDev->TimerStarted = FALSE;
    KeInitializeDpc(&skelDev->TimerDpc, TimeoutDPC, devObj);

    skelDev->DpcRequested = FALSE;
    IoInitializeDpcRequest(devObj, TransferDPC);

    status = IoConnectInterrupt(&skelDev->KIntrObj, ServiceInterrupt,
				devObj, NULL,
				skelDev->KIntrVector,
				skelDev->KIrql, skelDev->KIrql,
				LevelSensitive,
				TRUE, /* ShareVector */
				skelDev->KIntrAffinity, FALSE);

    if (!NT_SUCCESS(status)) {
	KdPrint(("SKELETON.SYS: Unable to connect interrupt\n"));
	status = STATUS_UNSUCCESSFUL;
	goto error;
    }

    if (0) {
  error:
	IoDeleteDevice(devObj);
    } else {
	ExFreePool(resourceList);
    }
    return status;
}

/*
 *----------------------------------------------------------------------
 *
 * ProbePCI
 *
 *	Attempt to find all Skeleton adapters in PCI address space
 *
 * Return Value:
 *	STATUS_SUCCESSFUL if everything went OK, STATUS_UNSUCCESSFUL
 *	if not.
 *
 *----------------------------------------------------------------------
 */
static NTSTATUS
ProbePCI(IN PDRIVER_OBJECT drvObj, IN PUNICODE_STRING regPath)
{
    PCI_SLOT_NUMBER     slotNumber;
    PPCI_COMMON_CONFIG  pciData;
    UCHAR               buf[PCI_COMMON_HDR_LENGTH];
    ULONG               i, f, j, bus;
    BOOLEAN             flag;
    UCHAR               vendorString[5] = {0};
    UCHAR               deviceString[5] = {0};
    NTSTATUS		status;
    ULONG		total = 0;

    pciData = (PPCI_COMMON_CONFIG) buf;
    slotNumber.u.bits.Reserved = 0;

    flag = TRUE;
    for (bus = 0; flag; bus++) {

        for (i = 0; i < PCI_MAX_DEVICES  &&  flag; i++) {
            slotNumber.u.bits.DeviceNumber = i;

            for (f = 0; f < PCI_MAX_FUNCTION; f++) {
                slotNumber.u.bits.FunctionNumber = f;

                j = HalGetBusData(PCIConfiguration, bus, slotNumber.u.AsULONG,
                    pciData, PCI_COMMON_HDR_LENGTH);

                if (j == 0) {
                    /* out of buses */
                    flag = FALSE;
                    break;
                }

                if (pciData->VendorID == PCI_INVALID_VENDORID) {
                    /* skip to next slot */
                    break;
                }

#if 0
                KdPrint(("PciData: ------------------------\n"
			 "  Bus: %d\n"
			 "  Device: %d\n"
			 "  Function: %d\n"
			 "  Vendor Id: %x\n"
			 "  Device Id: %x\n"
			 "  Command: %x\n"
			 "  Status: %x\n"
			 "  Rev Id: %x\n"
			 "  Pro`gIf: %x\n"
			 "  SubClass: %x\n"
			 "  BaseClass: %x\n"
			 "  CacheLine: %x\n"
			 "  Latency: %x\n"
			 "  Header Type: %x\n"
			 "  BIST: %x\n"
			 "  Base Reg[0]: %x\n"
			 "  Base Reg[1]: %x\n"
			 "  Base Reg[2]: %x\n"
			 "  Base Reg[3]: %x\n"
			 "  Base Reg[4]: %x\n"
			 "  Base Reg[5]: %x\n"
			 "  Rom Base: %x\n"
			 "  Interrupt Line: %x\n"
			 "  Interrupt Pin: %x\n"
			 "  Min Grant: %x\n"
			 "  Max Latency: %x\n",
			 bus,
			 i,
			 f,
			 pciData->VendorID,
			 pciData->DeviceID,
			 pciData->Command,
			 pciData->Status,
			 pciData->RevisionID,
			 pciData->ProgIf,
			 pciData->SubClass,
			 pciData->BaseClass,
			 pciData->CacheLineSize,
			 pciData->LatencyTimer,
			 pciData->HeaderType,
			 pciData->BIST,
			 pciData->u.type0.BaseAddresses[0],
			 pciData->u.type0.BaseAddresses[1],
			 pciData->u.type0.BaseAddresses[2],
			 pciData->u.type0.BaseAddresses[3],
			 pciData->u.type0.BaseAddresses[4],
			 pciData->u.type0.BaseAddresses[5],
			 pciData->u.type0.ROMBaseAddress,
			 pciData->u.type0.InterruptLine,
			 pciData->u.type0.MinimumGrant,
			 pciData->u.type0.MaximumLatency));
#endif

		/*
		 * If we find the Skeleton id, create a device
		 */
		if (pciData->VendorID == 0x115b && pciData->DeviceID == 0) {
		    status = CreateDevice(drvObj, regPath, bus,
					  slotNumber.u.AsULONG, pciData);
		    if (NT_SUCCESS(status)) {
			total++;
		    }
                }
            }
        }
    }
    if (total > 0) {
	return STATUS_SUCCESS;
    } else {
	return STATUS_NO_SUCH_DEVICE;
    }
}

/*
 *----------------------------------------------------------------------
 * ServiceInterrupt --
 *
 *	Service an interrupt from the Skeleton board
 *
 * Results:
 *	TRUE if the interrupt was handled, FALSE otherwise.
 *----------------------------------------------------------------------
 */
static BOOLEAN
ServiceInterrupt(IN PKINTERRUPT Interrupt, IN PVOID ServiceContext)
{
    PDEVICE_OBJECT devObj = (PDEVICE_OBJECT) ServiceContext;
    PSKELETON_DEVICE skelDev;

    skelDev = devObj->DeviceExtension;

    /*
     * XXX: Check if this interrupt was really intended for your board.
     * If not, return FALSE;
     */
    if (skelDev->RequestDpc) {
	if (!skelDev->DpcRequested) {
	    skelDev->DpcRequested = TRUE;
	    IoRequestDpc(devObj, NULL, devObj);
	} else {
	    KdPrint(("SKELETON.SYS: dpc overrun\n"));
	}
    }

    /*
     * Change this to TRUE when this routine does something
     */
    return TRUE;
}

/*
 * Overrides for Emacs to get consistency.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * tab-width: 8
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -4
 * c-argdecl-indent: 4
 * c-label-offset: -2
 * c-continued-statement-offset: 4
 * c-continued-brace-offset: 0
 * c-indent-level: 4
 * End:
 */
