// netnode.cpp

#include "windows.h"
#include "netnode.h"

// Initialize static data members
int CNetNode::m_platformID = -1;

// The default constructor is to create the root node
CNetNode::CNetNode() :
  m_haveEnumerated(FALSE),
  m_isRoot(TRUE),
  m_pNetResource(NULL),
  m_nodeArray(NULL),
  m_nodeCount(0)
{
  m_text = "Network"; // top-level network node

  m_pNetResource = new NETRESOURCE;
  m_pNetResource->dwScope = RESOURCE_GLOBALNET;
  m_pNetResource->dwType = RESOURCETYPE_ANY;
  m_pNetResource->dwUsage = RESOURCEUSAGE_CONTAINER;
  m_pNetResource->dwDisplayType = RESOURCEDISPLAYTYPE_ROOT;
  m_pNetResource->lpLocalName = NULL;
  m_pNetResource->lpRemoteName = NULL;
  m_pNetResource->lpComment = NULL;
  m_pNetResource->lpProvider = NULL;
}

// This constructor constructs a non-root node, such as
// a network provider, workgroup/domain, computer, or share
CNetNode::CNetNode(LPNETRESOURCE pNetResource) :
  m_haveEnumerated(FALSE),
  m_isRoot(FALSE),
  m_nodeArray(NULL),
  m_nodeCount(0)
{
  // Calculate the size of the netresource structure
  // plus the strings that it points to
  DWORD size = sizeof(NETRESOURCE);

  if (pNetResource->lpLocalName)
    size += (strlen(pNetResource->lpLocalName) + 1);
  if (pNetResource->lpComment)
    size += (strlen(pNetResource->lpComment) + 1);
  if (pNetResource->lpProvider)
    size += (strlen(pNetResource->lpProvider) + 1);
  if (pNetResource->lpRemoteName)
    size += (strlen(pNetResource->lpRemoteName) + 1);

  // Allocate a contiguous block of memory for the
  // netresource and its strings
  m_pNetResource = (LPNETRESOURCE) new char[size];

  // Copy the base netresource structure
  memcpy(m_pNetResource, pNetResource,
   sizeof(NETRESOURCE));

  // Move the strings into the new netresource area
  char * pString = ((char *) m_pNetResource) +
    sizeof(NETRESOURCE);

  if (pNetResource->lpLocalName)
  {
    strcpy(pString, pNetResource->lpLocalName);
    m_pNetResource->lpLocalName = pString;
    pString += (strlen(pNetResource->lpLocalName) + 1);
  }

  if (pNetResource->lpComment)
  {
    strcpy(pString, pNetResource->lpComment);
    m_pNetResource->lpComment = pString;
    pString += (strlen(pNetResource->lpComment) + 1);
  }

  if (pNetResource->lpProvider)
  {
    strcpy(pString, pNetResource->lpProvider);
    m_pNetResource->lpProvider = pString;
    pString += (strlen(pNetResource->lpProvider) + 1);
  }

  if (pNetResource->lpRemoteName)
  {
    strcpy(pString, pNetResource->lpRemoteName);
    m_pNetResource->lpRemoteName = pString;
    m_text = pString;
    pString += (strlen(pNetResource->lpRemoteName) + 1);
  }
  else
  {
    m_text = "Unknown";
  }
}


CNetNode::~CNetNode()
{
  if (m_pNetResource)
    delete m_pNetResource;

  // Delete all child nodes
  for (DWORD i = 0; i < m_nodeCount; i++)
  {
    if (m_nodeArray[i])
     delete m_nodeArray[i];
  }
  delete [] m_nodeArray;
  m_nodeArray = NULL;
  m_nodeCount = 0;
}


DWORD CNetNode::EnumerateNetwork()
{
  static const DWORD  ENUM_BUF_SIZE = 16000;

  if (m_haveEnumerated)
    return(0);

  // Special case for Win95 root node
  if (m_isRoot && PlatformIsWindows95())
    return(EnumerateNetworkWin95Root());

  DWORD         enumBufSize = ENUM_BUF_SIZE;
  DWORD         enumNumEntries = 0xffffffff;
  HANDLE        hEnum = NULL;
  LPNETRESOURCE pNetResBuf = NULL;
  DWORD         ret = 0;

  ret = WNetOpenEnum(RESOURCE_GLOBALNET,
    RESOURCETYPE_ANY,
     0,
     m_pNetResource,
     &hEnum);

  if (ret == NO_ERROR)
  {
    pNetResBuf = (LPNETRESOURCE) new char[enumBufSize];

    ret = WNetEnumResource(hEnum,
       &enumNumEntries,
      pNetResBuf,
      &enumBufSize);
  }

  if (ret == NO_ERROR)
  {
    m_haveEnumerated = TRUE;
    m_nodeCount = enumNumEntries;
    m_nodeArray = new CNetNode * [m_nodeCount];
    memset(m_nodeArray, 0, m_nodeCount * sizeof(CNetNode *));

    CNetNode * pNetNode;
    for (DWORD i = 0; i < enumNumEntries; i++)
    {
      pNetNode = new CNetNode(&pNetResBuf[i]);
      m_nodeArray[i] = pNetNode;
    }
  }

  if (pNetResBuf)
    delete [] pNetResBuf;

  if (hEnum)
    ret = WNetCloseEnum(hEnum);

  return(ret);
}

int CNetNode::InitPlatformID()
{
  if (m_platformID < 0)
  {
    OSVERSIONINFO   osvi;
    memset(&osvi, 0, sizeof(OSVERSIONINFO));
    osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    DWORD ret = ::GetVersionEx(&osvi);
    m_platformID = osvi.dwPlatformId;
  }

  return(m_platformID);
}


DWORD CNetNode::EnumerateNetworkWin95Root()
{
  // Win95 does not return the top-level network providers
  // in the same manner as NT, so we get the provider names
  // from the registry.
  char netProviders[4096];
  DWORD numProviders;
  EnumerateWin95NetProviders(netProviders, &numProviders);

  if (numProviders == 0)
    return(ERROR_NO_NETWORK);

  CNetNode * pNetNode;
  m_haveEnumerated = TRUE;
  m_nodeCount = numProviders;
  m_nodeArray = new CNetNode * [m_nodeCount];
  memset(m_nodeArray, 0, m_nodeCount * sizeof(CNetNode *));

  NETRESOURCE netResource;
  netResource.dwType = RESOURCETYPE_ANY;
#if(WINVER >= 0x0400)
  netResource.dwDisplayType = RESOURCEDISPLAYTYPE_NETWORK;
#else
  netResource.dwDisplayType = RESOURCEDISPLAYTYPE_GENERIC;
#endif
  netResource.lpLocalName = NULL;
  netResource.lpRemoteName = NULL;
  netResource.lpComment = NULL;
  netResource.dwUsage = RESOURCEUSAGE_CONTAINER;
  netResource.dwScope = RESOURCE_GLOBALNET;

  char * pNetProvider = netProviders;
  for (DWORD i = 0; i < numProviders; i++)
  {
    netResource.lpProvider = pNetProvider;
    pNetNode = new CNetNode(&netResource);
    m_nodeArray[i] = pNetNode;
    pNetNode->m_text =
      pNetNode->m_pNetResource->lpProvider;
    pNetProvider = pNetProvider + strlen(pNetProvider) + 1;
  }
  return(0);
}

void CNetNode::EnumerateWin95NetProviders(char *
  pNetProviders, DWORD * pNumNetProviders)
{
  // This routine is specific to Win95, because the registry
  // structure on NT is slightly different
  DWORD   ret;
  HKEY    hKeyProviderOrder = NULL;
  HKEY    hKeyProviderName = NULL;

  ret = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
    "System\\CurrentControlSet\\"  // continued...
    "control\\NetworkProvider\\Order",
    0,
    KEY_QUERY_VALUE,
    &hKeyProviderOrder);

  if (ret)
    return;

  char        providerServiceName[MAX_PATH];
  char        providerData[MAX_PATH];
  char        providerNameKey[MAX_PATH];
  char        providerName[MAX_PATH];
  DWORD       buflenProviderServiceName;
  DWORD       buflenProviderName;
  DWORD       buflenData;
  DWORD       regType;

  DWORD index = 0;

  while (1)
  {
    buflenProviderServiceName = MAX_PATH;
    buflenProviderName = MAX_PATH;
    buflenData = MAX_PATH;

    // Get a network provider "service name"
    ret = RegEnumValue(hKeyProviderOrder,
      index,
      providerServiceName,
      &buflenProviderServiceName,
      NULL,
      &regType,
      (LPBYTE) providerData,
      &buflenData);

    if (ret)
      break;

    strcpy(providerNameKey,
      "System\\CurrentControlSet\\Services\\");
    strcat(providerNameKey, providerServiceName);
    strcat(providerNameKey, "\\NetworkProvider");

    ret = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
      providerNameKey,
      0,
      KEY_QUERY_VALUE,
      &hKeyProviderName);

    if (ret)
      break;

    // Get the provider name
    ret = RegQueryValueEx(hKeyProviderName,
      "Name",
      NULL,
      &regType,
      (LPBYTE) providerName,
      &buflenProviderName);

    if (hKeyProviderName)
      RegCloseKey(hKeyProviderName);

    if (ret)
      break;

    strcat(pNetProviders, providerName);
    pNetProviders = pNetProviders +
      strlen(providerName) + 1;
    index++;
  }

  if (hKeyProviderOrder)
    RegCloseKey(hKeyProviderOrder);

  *pNumNetProviders = index;
}
