#ifndef ubi_BinTree_H
#define ubi_BinTree_H
/* ========================================================================== **
 *                              ubi_BinTree.h
 *
 *  Copyright (C) 1991-1997 by Christopher R. Hertel
 *
 *  Email:  crh@ubiqx.mn.org
 * -------------------------------------------------------------------------- **
 *
 *  This module implements simple binary trees.
 *
 * -------------------------------------------------------------------------- **
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * -------------------------------------------------------------------------- **
 *
 * Log: ubi_BinTree.h,v
 * Revision 3.0  1997/12/08 06:49:15  crh
 * This is a new major revision level for all ubiqx binary tree modules.
 * In previous releases, the ubi_trNode structure looked like this:
 *
 *   typedef struct ubi_btNodeStruct
 *     {
 *     struct ubi_btNodeStruct *Link[3];
 *     signed char              gender;
 *     } ubi_btNode;
 *
 * As a result, the pointers were indexed as
 *
 *   Link[0] == Left Child
 *   Link[1] == Parent
 *   Link[2] == Right Child
 *
 * With this release, the node structure changes to:
 *
 *   typedef struct ubi_btNodeStruct
 *     {
 *     struct ubi_btNodeStruct *leftlink
 *     struct ubi_btNodeStruct *Link[2];
 *     signed char              gender;
 *     } ubi_btNode;
 *
 * The leftlink field is used as a place holder, and the pointers are now
 * index as
 *
 *   Link[-1] == Left Child  (aka. leftlink)
 *   Link[ 0] == Parent
 *   Link[ 1] == Right Child
 *
 * which is much nicer.  Doing things this way removes the need to shift
 * values between the two numbering schemes, thus removing one macro,
 * simplifying another, and getting rid of a whole bunch of increment &
 * decrement operations.
 *
 * Revision 2; 1995/02/27 - 1997/12/07 included:
 *  - The addition of the ModuleID static string and ubi_ModuleID() function.
 *  - The addition of the public functions FirstOf() and LastOf(). These
 *    functions are used with trees that allow duplicate keys.
 *  - The addition of the ubi_btLeafNode() function.
 *  - A rewrite of the Locate() function.
 *  - A change to the parameter list in function ubi_btInitTree().
 *  - Bugfixes.
 *
 * Revision 1; 93/10/15 - 95/02/27:
 * Revision 1 introduced a set of #define's that provide a single API to all
 * of the existing tree modules.  Each of these modules has a different name
 * prefix, as follows:
 *
 *       Module        Prefix
 *     ubi_BinTree     ubi_bt
 *     ubi_AVLtree     ubi_avl
 *     ubi_SplayTree   ubi_spt
 *
 * Only those portions of the base module (ubi_BinTree) that are superceeded
 * in the descendant module have new names.  For example, the AVL node
 * structure in ubi_AVLtree.h is named "ubi_avlNode", but the root structure
 * is still "ubi_btRoot".  Using SplayTree, the locate function is called
 * "ubi_sptLocate", but the next and previous functions remained "ubi_btNext"
 * and "ubi_btPrev".
 *
 * This is confusing.
 *
 * So, I added a set of defined names that get redefined in any of the
 * descendant modules.  To use this standardized interface in your code,
 * simply replace all occurances of "ubi_bt", "ubi_avl", and "ubi_spt" with
 * "ubi_tr".  The "ubi_tr" names will resolve to the correct function or
 * datatype names for the module that you are using.  Just remember to
 * include the header for that module in your program file.  Because these
 * names are handled by the preprocessor, there is no added run-time
 * overhead.
 *
 * Note that the original names do still exist, and can be used if you wish
 * to write code directly to a specific module.  This should probably only be
 * done if you are planning to implement a new descendant type, such as
 * red/black trees, or if you plan to use two or more specific tree types
 * in the same piece of code.  CRH
 *
 *  V0.0 - June, 1991   -  Written by Christopher R. Hertel (CRH).
 *
 * ========================================================================== **
 */

/* -------------------------------------------------------------------------- **
 * Constants...
 *
 *  General purpose:
 *    ubi_trTRUE  - Boolean TRUE.
 *    ubi_trFALSE - Boolean FALSE.
 *
 *  Flags used in the tree header:
 *    ubi_trOVERWRITE   - This flag indicates that an existing node may be
 *                        overwritten by a new node with a matching key.
 *    ubi_trDUPKEY      - This flag indicates that the tree allows duplicate
 *                        keys.  If the tree allows duplicates, the overwrite
 *                        flag is ignored.
 *
 *  Node link array index constants:
 *    (Each node has an array of three pointers; pointer to the left child,
 *    pointer to the right child, and a pointer back to the parent node.)
 *      ubi_trLEFT    - Left child pointer.
 *      ubi_trPARENT  - Parent pointer.
 *      ubi_trRIGHT   - Right child pointer.
 *      ubi_trEQUAL   - Synonym for PARENT.
 *
 *  ubi_trCompOps:  These values are used in the ubi_trLocate() function.
 *    ubi_trLT  - request the first instance of the greatest key less than
 *                the search key.
 *    ubi_trLE  - request the first instance of the greatest key that is less
 *                than or equal to the search key.
 *    ubi_trEQ  - request the first instance of key that is equal to the
 *                search key.
 *    ubi_trGE  - request the first instance of a key that is greater than
 *                or equal to the search key.
 *    ubi_trGT  - request the first instance of the first key that is greater
 *                than the search key.
 */

#define ubi_trTRUE  0xFF
#define ubi_trFALSE 0x00

#define ubi_trOVERWRITE 0x01        /* Turn on allow overwrite      */
#define ubi_trDUPKEY    0x02        /* Turn on allow duplicate keys */

/* Pointer array index constants... */
#define ubi_trLEFT   -1
#define ubi_trPARENT  0
#define ubi_trRIGHT   1
#define ubi_trEQUAL   ubi_trPARENT

typedef enum
  {
  ubi_trLT = 1,
  ubi_trLE,
  ubi_trEQ,
  ubi_trGE,
  ubi_trGT
  } ubi_trCompOps;

/* -------------------------------------------------------------------------- **
 * Macros...
 *  ubi_trNormalize() - "Normalize" a value with respect to ubi_trLEFT,
 *                      ubi_trRIGHT, and ubi_trPARENT.  This macro calls
 *                      ubi_btSgn() to convert the input to -1, 0, or 1.
 *                      The resultant value is returned as a signed char.
 *
 *  ubi_trRevWay()    - converts ubi_trLEFT to ubi_trRIGHT and vice versa.
 *                      ubi_trPARENT (ubi_trEQUAL) is left as is.
 *
 *  ubi_trDups_OK()   - returns TRUE if the tree allows duplicates.
 *
 *  ubi_trOvwt_OK()   - returns TRUE if the overwrite flag is on.  Note
 *                      that overwrites will not occur in a tree that
 *                      allows duplicates.
 */

#define ubi_trNormalize(W) ((signed char)ubi_btSgn(W))
#define ubi_trRevWay(W) (-(W))

#define ubi_trDups_OK(A) \
        ((ubi_trDUPKEY & ((A)->flags))   ? (ubi_trTRUE) : (ubi_trFALSE))
#define ubi_trOvwt_OK(A) \
        ((ubi_trOVERWRITE & ((A)->flags))? (ubi_trTRUE) : (ubi_trFALSE))


/* -------------------------------------------------------------------------- **
 * Typedefs...
 * 
 * ubi_trBool     - Your typcial true or false...
 *
 * ubi_btItemPtr  - The Item Pointer is a generic pointer.  It is used to
 *                  indicate a key for which to search within the tree.
 *                  The ubi_trFind(), ubi_trLocate(), and  ubi_trInsert()
 *                  functions all perform searches.
 */

typedef unsigned char ubi_trBool;   /* Our own name for "boolean".            */

typedef void *ubi_btItemPtr;        /* A pointer to (key) data within a node. */

/* -------------------------------------------------------------------------- **
 * Typedefs continued...
 *
 *  Binary Tree Node Structure:  This structure defines the basic elements of
 *       the tree nodes.  In general you *SHOULD NOT PLAY WITH THESE FIELDS*!
 *       But, of course, I have to put the structure into this header so that
 *       you can use it as a building block.
 *
 *  The fields are as follows:
 *    leftlink -  pointer to the left child of the node.  This field will
 *                be accessed as Link[-1].
 *    Link     -  an array of pointers.  These pointers are manipulated by
 *                the BT routines.  The pointers indicate the left and right
 *                child nodes and the parent node.  By keeping track of the
 *                parent pointer, we avoid the need for recursive routines or
 *                hand-tooled stacks to keep track of our path back to the
 *                root.  The use of these pointers is subject to change without
 *                notice.
 *    gender   -  a one-byte field indicating whether the node is the RIGHT or
 *                LEFT child of its parent.  If the node is the root of the
 *                tree, gender will be PARENT.
 */

typedef struct ubi_btNodeStruct
  {
  struct ubi_btNodeStruct *leftlink;    /* Will be accessed as Link[-1].      */
  struct ubi_btNodeStruct *Link[2];     /* Parent & Right links.              */
  signed char              gender;      /* Indicates Left/Right of parent.    */
  } ubi_btNode;

typedef ubi_btNode *ubi_btNodePtr;     /* Pointer to an ubi_btNode structure. */

/* -------------------------------------------------------------------------- **
 * Typedefs continued...
 *
 *  The next three typedefs define standard function types used by the binary
 *  tree management routines.  In particular:
 *
 *    ubi_btCompFunc    is a pointer to a comparison function.  Comparison
 *                      functions are passed an ubi_btItemPtr and an
 *                      ubi_btNodePtr.  They return a value that is (<0), 0,
 *                      or (>0) to indicate that the Item is (respectively)
 *                      "less than", "equal to", or "greater than" the Item
 *                      contained within the node.  (See ubi_btInitTree()).
 *    ubi_btActionRtn   is a pointer to a function that may be called for each
 *                      node visited when performing a tree traversal (see
 *                      ubi_btTraverse()).  The function will be passed two
 *                      parameters: the first is a pointer to a node in the
 *                      tree, the second is a generic pointer that may point to
 *                      anything that you like.
 *    ubi_btKillNodeRtn is a pointer to a function that will deallocate the
 *                      memory used by a node (see ubi_btKillTree()).  Since
 *                      memory management is left up to you, deallocation may
 *                      mean anything that you want it to mean.  Just remember
 *                      that the tree *will* be destroyed and that none of the
 *                      node pointers will be valid any more.
 */

typedef  int (*ubi_btCompFunc)( ubi_btItemPtr, ubi_btNodePtr );

typedef void (*ubi_btActionRtn)( ubi_btNodePtr, void * );

typedef void (*ubi_btKillNodeRtn)( ubi_btNodePtr );

/* -------------------------------------------------------------------------- **
 * Typedefs continued...
 *
 *  Tree Root Structure: This structure gives us a convenient handle for
 *                       accessing whole AVL trees.  The fields are:
 *    root  -  A pointer to the root node of the AVL tree.
 *    count -  A count of the number of nodes stored in the tree.
 *    cmp   -  A pointer to the comparison routine to be used when building or
 *             searching the tree.
 *    flags -  A set of bit flags.  Two flags are currently defined:
 *
 *       ubi_trOVERWRITE -  If set, this flag indicates that a new node should
 *         (bit 0x01)       overwrite an old node if the two have identical
 *                          keys (ie., the keys are equal).
 *       ubi_trDUPKEY    -  If set, this flag indicates that the tree is
 *         (bit 0x02)       allowed to contain nodes with duplicate keys.
 *
 *       NOTE: ubi_trInsert() tests ubi_trDUPKEY before ubi_trOVERWRITE.
 *             If duplicate keys are allowed, then no entry will be
 *             overwritten.
 *
 *  All of these values are set when you initialize the root structure by
 *  calling ubi_trInitTree().
 */

typedef struct
  {
  ubi_btNodePtr  root;     /* A pointer to the root node of the tree       */
  unsigned long  count;    /* A count of the number of nodes in the tree   */
  ubi_btCompFunc cmp;      /* A pointer to the tree's comparison function  */
  unsigned char  flags;    /* Overwrite Y|N, Duplicate keys Y|N...         */
  } ubi_btRoot;

typedef ubi_btRoot *ubi_btRootPtr;  /* Pointer to an ubi_btRoot structure. */


/* -------------------------------------------------------------------------- **
 * Function Prototypes.
 */

long ubi_btSgn( long x );
  /* ------------------------------------------------------------------------ **
   * Return the sign of x; {negative,zero,positive} ==> {-1, 0, 1}.
   *
   *  Input:  x - a signed long integer value.
   *
   *  Output: -1, 0, or 1 representing the "sign" of x as follows:
   *            -1 == negative
   *             0 == zero (no sign)
   *             1 == positive
   *
   *  Notes:  This utility is provided in order to facilitate the conversion
   *          of C comparison function return values into BinTree direction
   *          values: {ubi_trLEFT, ubi_trPARENT, ubi_trEQUAL}.  It is
   *          incorporated into the Normalize() conversion macro.
   *
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btInitNode( ubi_btNodePtr NodePtr );
  /* ------------------------------------------------------------------------ **
   * Initialize a tree node.
   *
   *  Input:   a pointer to a ubi_btNode structure to be initialized.
   *  Output:  a pointer to the initialized ubi_btNode structure (ie. the
   *           same as the input pointer).
   * ------------------------------------------------------------------------ **
   */

ubi_btRootPtr ubi_btInitTree( ubi_btRootPtr   RootPtr,
                              ubi_btCompFunc  CompFunc,
                              unsigned char   Flags );
  /* ------------------------------------------------------------------------ **
   * Initialize the fields of a Tree Root header structure.
   *  
   *  Input:   RootPtr   - a pointer to an ubi_btRoot structure to be
   *                       initialized.   
   *           CompFunc  - a pointer to a comparison function that will be used
   *                       whenever nodes in the tree must be compared against
   *                       outside values.
   *           Flags     - One bytes worth of flags.  Flags include
   *                       ubi_trOVERWRITE and ubi_trDUPKEY.  See the header
   *                       file for more info.
   *
   *  Output:  a pointer to the initialized ubi_btRoot structure (ie. the
   *           same value as RootPtr).
   * 
   *  Note:    The interface to this function has changed from that of 
   *           previous versions.  The <Flags> parameter replaces two      
   *           boolean parameters that had the same basic effect.
   * ------------------------------------------------------------------------ **
   */

ubi_trBool ubi_btInsert( ubi_btRootPtr  RootPtr,
                         ubi_btNodePtr  NewNode,
                         ubi_btItemPtr  ItemPtr,
                         ubi_btNodePtr *OldNode );
  /* ------------------------------------------------------------------------ **
   * This function uses a non-recursive algorithm to add a new element to the
   * tree.
   *
   *  Input:   RootPtr  -  a pointer to the ubi_btRoot structure that indicates
   *                       the root of the tree to which NewNode is to be added.
   *           NewNode  -  a pointer to an ubi_btNode structure that is NOT
   *                       part of any tree.
   *           ItemPtr  -  A pointer to the sort key that is stored within
   *                       *NewNode.  ItemPtr MUST point to information stored
   *                       in *NewNode or an EXACT DUPLICATE.  The key data
   *                       indicated by ItemPtr is used to place the new node
   *                       into the tree.
   *           OldNode  -  a pointer to an ubi_btNodePtr.  When searching
   *                       the tree, a duplicate node may be found.  If
   *                       duplicates are allowed, then the new node will
   *                       be simply placed into the tree.  If duplicates
   *                       are not allowed, however, then one of two things
   *                       may happen.
   *                       1) if overwritting *is not* allowed, this
   *                          function will return FALSE (indicating that
   *                          the new node could not be inserted), and
   *                          *OldNode will point to the duplicate that is
   *                          still in the tree.
   *                       2) if overwritting *is* allowed, then this
   *                          function will swap **OldNode for *NewNode.
   *                          In this case, *OldNode will point to the node
   *                          that was removed (thus allowing you to free
   *                          the node).
   *                          **  If you are using overwrite mode, ALWAYS  **
   *                          ** check the return value of this parameter! **
   *                 Note: You may pass NULL in this parameter, the
   *                       function knows how to cope.  If you do this,
   *                       however, there will be no way to return a
   *                       pointer to an old (ie. replaced) node (which is
   *                       a problem if you are using overwrite mode).
   *
   *  Output:  a boolean value indicating success or failure.  The function
   *           will return FALSE if the node could not be added to the tree.
   *           Such failure will only occur if duplicates are not allowed,
   *           nodes cannot be overwritten, AND a duplicate key was found
   *           within the tree.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btRemove( ubi_btRootPtr RootPtr,
                            ubi_btNodePtr DeadNode );
  /* ------------------------------------------------------------------------ **
   * This function removes the indicated node from the tree.
   *
   *  Input:   RootPtr  -  A pointer to the header of the tree that contains
   *                       the node to be removed.
   *           DeadNode -  A pointer to the node that will be removed.
   *
   *  Output:  This function returns a pointer to the node that was removed
   *           from the tree (ie. the same as DeadNode).
   *
   *  Note:    The node MUST be in the tree indicated by RootPtr.  If not,
   *           strange and evil things will happen to your trees.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btLocate( ubi_btRootPtr RootPtr,
                            ubi_btItemPtr FindMe,
                            ubi_trCompOps CompOp );
  /* ------------------------------------------------------------------------ **
   * The purpose of ubi_btLocate() is to find a node or set of nodes given
   * a target value and a "comparison operator".  The Locate() function is
   * more flexible and (in the case of trees that may contain dupicate keys)
   * more precise than the ubi_btFind() function.  The latter is faster,
   * but it only searches for exact matches and, if the tree contains
   * duplicates, Find() may return a pointer to any one of the duplicate-
   * keyed records.
   *
   *  Input:
   *     RootPtr  -  A pointer to the header of the tree to be searched.
   *     FindMe   -  An ubi_btItemPtr that indicates the key for which to
   *                 search.
   *     CompOp   -  One of the following:
   *                    CompOp     Return a pointer to the node with
   *                    ------     ---------------------------------
   *                   ubi_trLT - the last key value that is less
   *                              than FindMe.
   *                   ubi_trLE - the first key matching FindMe, or
   *                              the last key that is less than
   *                              FindMe.
   *                   ubi_trEQ - the first key matching FindMe.
   *                   ubi_trGE - the first key matching FindMe, or the
   *                              first key greater than FindMe.
   *                   ubi_trGT - the first key greater than FindMe.
   *  Output:
   *     A pointer to the node matching the criteria listed above under
   *     CompOp, or NULL if no node matched the criteria.
   *
   *  Notes:
   *     In the case of trees with duplicate keys, Locate() will behave as
   *     follows:
   *
   *     Find:  3                       Find: 3
   *     Keys:  1 2 2 2 3 3 3 3 3 4 4   Keys: 1 1 2 2 2 4 4 5 5 5 6
   *                  ^ ^         ^                   ^ ^
   *                 LT EQ        GT                 LE GE
   *
   *     That is, when returning a pointer to a node with a key that is LESS
   *     THAN the target key (FindMe), Locate() will return a pointer to the
   *     LAST matching node.
   *     When returning a pointer to a node with a key that is GREATER
   *     THAN the target key (FindMe), Locate() will return a pointer to the
   *     FIRST matching node.
   *
   *  See Also: ubi_btFind(), ubi_btFirstOf(), ubi_btLastOf().
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btFind( ubi_btRootPtr RootPtr,
                          ubi_btItemPtr FindMe );
  /* ------------------------------------------------------------------------ **
   * This function performs a non-recursive search of a tree for any node
   * matching a specific key.
   *
   *  Input:
   *     RootPtr  -  a pointer to the header of the tree to be searched.
   *     FindMe   -  a pointer to the key value for which to search.
   *
   *  Output:
   *     A pointer to a node with a key that matches the key indicated by
   *     FindMe, or NULL if no such node was found.
   *
   *  Note:   In a tree that allows duplicates, the pointer returned *might
   *          not* point to the (sequentially) first occurance of the
   *          desired key.  In such a tree, it may be more useful to use
   *          ubi_btLocate().
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btNext( ubi_btNodePtr P );
  /* ------------------------------------------------------------------------ **
   * Given the node indicated by P, find the (sorted order) Next node in the
   * tree.
   *  Input:   P  -  a pointer to a node that exists in a binary tree.
   *  Output:  A pointer to the "next" node in the tree, or NULL if P pointed
   *           to the "last" node in the tree or was NULL.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btPrev( ubi_btNodePtr P );
  /* ------------------------------------------------------------------------ **
   * Given the node indicated by P, find the (sorted order) Previous node in
   * the tree.
   *  Input:   P  -  a pointer to a node that exists in a binary tree.
   *  Output:  A pointer to the "previous" node in the tree, or NULL if P
   *           pointed to the "first" node in the tree or was NULL.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btFirst( ubi_btNodePtr P );
  /* ------------------------------------------------------------------------ **
   * Given the node indicated by P, find the (sorted order) First node in the
   * subtree of which *P is the root.
   *  Input:   P  -  a pointer to a node that exists in a binary tree.
   *  Output:  A pointer to the "first" node in a subtree that has *P as its
   *           root.  This function will return NULL only if P is NULL.
   *  Note:    In general, you will be passing in the value of the root field
   *           of an ubi_btRoot structure.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btLast( ubi_btNodePtr P );
  /* ------------------------------------------------------------------------ **
   * Given the node indicated by P, find the (sorted order) Last node in the
   * subtree of which *P is the root.
   *  Input:   P  -  a pointer to a node that exists in a binary tree.
   *  Output:  A pointer to the "last" node in a subtree that has *P as its
   *           root.  This function will return NULL only if P is NULL.
   *  Note:    In general, you will be passing in the value of the root field
   *           of an ubi_btRoot structure.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btFirstOf( ubi_btRootPtr RootPtr,
                             ubi_btItemPtr MatchMe,
                             ubi_btNodePtr p );
  /* ------------------------------------------------------------------------ **
   * Given a tree that a allows duplicate keys, and a pointer to a node in
   * the tree, this function will return a pointer to the first (traversal
   * order) node with the same key value.
   *
   *  Input:  RootPtr - A pointer to the root of the tree.
   *          MatchMe - A pointer to the key value.  This should probably
   *                    point to the key within node *p.
   *          p       - A pointer to a node in the tree.
   *  Output: A pointer to the first node in the set of nodes with keys
   *          matching <FindMe>.
   *  Notes:  Node *p MUST be in the set of nodes with keys matching
   *          <FindMe>.  If not, this function will return NULL.
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btLastOf( ubi_btRootPtr RootPtr,
                            ubi_btItemPtr MatchMe,
                            ubi_btNodePtr p );
  /* ------------------------------------------------------------------------ **
   * Given a tree that a allows duplicate keys, and a pointer to a node in
   * the tree, this function will return a pointer to the last (traversal
   * order) node with the same key value.
   *
   *  Input:  RootPtr - A pointer to the root of the tree.
   *          MatchMe - A pointer to the key value.  This should probably
   *                    point to the key within node *p.
   *          p       - A pointer to a node in the tree.
   *  Output: A pointer to the last node in the set of nodes with keys
   *          matching <FindMe>.
   *  Notes:  Node *p MUST be in the set of nodes with keys matching
   *          <FindMe>.  If not, this function will return NULL.
   * ------------------------------------------------------------------------ **
   */

ubi_trBool ubi_btTraverse( ubi_btRootPtr   RootPtr,
                           ubi_btActionRtn EachNode,
                           void           *UserData );
  /* ------------------------------------------------------------------------ **
   * Traverse a tree in sorted order (non-recursively).  At each node, call
   * (*EachNode)(), passing a pointer to the current node, and UserData as the
   * second parameter.
   *  Input:   RootPtr  -  a pointer to an ubi_btRoot structure that indicates
   *                       the tree to be traversed.
   *           EachNode -  a pointer to a function to be called at each node
   *                       as the node is visited.
   *           UserData -  a generic pointer that may point to anything that
   *                       you choose.
   *  Output:  A boolean value.  FALSE if the tree is empty, otherwise TRUE.
   * ------------------------------------------------------------------------ **
   */

ubi_trBool ubi_btKillTree( ubi_btRootPtr     RootPtr,
                           ubi_btKillNodeRtn FreeNode );
  /* ------------------------------------------------------------------------ **
   * Delete an entire tree (non-recursively) and reinitialize the ubi_btRoot
   * structure.  Note that this function will return FALSE if either parameter
   * is NULL.
   *
   *  Input:   RootPtr  -  a pointer to an ubi_btRoot structure that indicates
   *                       the root of the tree to delete.
   *           FreeNode -  a function that will be called for each node in the
   *                       tree to deallocate the memory used by the node.
   *
   *  Output:  A boolean value.  FALSE if either input parameter was NULL, else
   *           TRUE.
   *
   * ------------------------------------------------------------------------ **
   */

ubi_btNodePtr ubi_btLeafNode( ubi_btNodePtr leader );
  /* ------------------------------------------------------------------------ **
   * Returns a pointer to a leaf node.
   *
   *  Input:  leader  - Pointer to a node at which to start the descent.
   *
   *  Output: A pointer to a leaf node selected in a somewhat arbitrary
   *          manner.
   *
   *  Notes:  I wrote this function because I was using splay trees as a
   *          database cache.  The cache had a maximum size on it, and I
   *          needed a way of choosing a node to sacrifice if the cache
   *          became full.  In a splay tree, less recently accessed nodes
   *          tend toward the bottom of the tree, meaning that leaf nodes
   *          are good candidates for removal.  (I really can't think of
   *          any other reason to use this function.)
   *        + In a simple binary tree or an AVL tree, the most recently
   *          added nodes tend to be nearer the bottom, making this a *bad*
   *          way to choose which node to remove from the cache.
   *        + Randomizing the traversal order is probably a good idea.  You
   *          can improve the randomization of leaf node selection by passing
   *          in pointers to nodes other than the root node each time.  A
   *          pointer to any node in the tree will do.  Of course, if you
   *          pass a pointer to a leaf node you'll get the same thing back.
   *        + If using a splay tree, splaying the tree will tend to randomize
   *          things a bit too.  See ubi_SplayTree for more info.
   *
   * ------------------------------------------------------------------------ **
   */


int ubi_btModuleID( int size, char *list[] );
  /* ------------------------------------------------------------------------ **
   * Returns a set of strings that identify the module.
   *
   *  Input:  size  - The number of elements in the array <list>.
   *          list  - An array of pointers of type (char *).  This array
   *                  should, initially, be empty.  This function will fill
   *                  in the array with pointers to strings.
   *  Output: The number of elements of <list> that were used.  If this value
   *          is less than <size>, the values of the remaining elements are
   *          not guaranteed.
   *
   *  Notes:  Please keep in mind that the pointers returned indicate strings
   *          stored in static memory.  Don't free() them, don't write over
   *          them, etc.  Just read them.
   * ------------------------------------------------------------------------ **
   */

/* -------------------------------------------------------------------------- **
 * Masquarade...
 *
 * This set of defines allows you to write programs that will use any of the
 * implemented binary tree modules (currently BinTree, AVLtree, and SplayTree).
 * Instead of using ubi_bt..., use ubi_tr..., and select the tree type by
 * including the appropriate module header.
 */

#define ubi_trItemPtr ubi_btItemPtr

#define ubi_trNode    ubi_btNode
#define ubi_trNodePtr ubi_btNodePtr

#define ubi_trRoot    ubi_btRoot
#define ubi_trRootPtr ubi_btRootPtr

#define ubi_trCompFunc    ubi_btCompFunc
#define ubi_trActionRtn   ubi_btActionRtn
#define ubi_trKillNodeRtn ubi_btKillNodeRtn

#define ubi_trSgn( x ) ubi_btSgn( x )

#define ubi_trInitNode( Np ) ubi_btInitNode( (ubi_btNodePtr)(Np) )

#define ubi_trInitTree( Rp, Cf, Fl ) \
        ubi_btInitTree( (ubi_btRootPtr)(Rp), (ubi_btCompFunc)(Cf), (Fl) )

#define ubi_trInsert( Rp, Nn, Ip, On ) \
        ubi_btInsert( (ubi_btRootPtr)(Rp), (ubi_btNodePtr)(Nn), \
                      (ubi_btItemPtr)(Ip), (ubi_btNodePtr *)(On) )

#define ubi_trRemove( Rp, Dn ) \
        ubi_btRemove( (ubi_btRootPtr)(Rp), (ubi_btNodePtr)(Dn) )

#define ubi_trLocate( Rp, Ip, Op ) \
        ubi_btLocate( (ubi_btRootPtr)(Rp), \
                      (ubi_btItemPtr)(Ip), \
                      (ubi_trCompOps)(Op) )

#define ubi_trFind( Rp, Ip ) \
        ubi_btFind( (ubi_btRootPtr)(Rp), (ubi_btItemPtr)(Ip) )

#define ubi_trNext( P ) ubi_btNext( (ubi_btNodePtr)(P) )

#define ubi_trPrev( P ) ubi_btPrev( (ubi_btNodePtr)(P) )

#define ubi_trFirst( P ) ubi_btFirst( (ubi_btNodePtr)(P) )

#define ubi_trLast( P ) ubi_btLast( (ubi_btNodePtr)(P) )

#define ubi_trFirstOf( Rp, Ip, P ) \
        ubi_btFirstOf( (ubi_btRootPtr)(Rp), \
                       (ubi_btItemPtr)(Ip), \
                       (ubi_btNodePtr)(P) )

#define ubi_trLastOf( Rp, Ip, P ) \
        ubi_btLastOf( (ubi_btRootPtr)(Rp), \
                      (ubi_btItemPtr)(Ip), \
                      (ubi_btNodePtr)(P) )

#define ubi_trTraverse( Rp, En, Ud ) \
        ubi_btTraverse((ubi_btRootPtr)(Rp), (ubi_btActionRtn)(En), (void *)(Ud))

#define ubi_trKillTree( Rp, Fn ) \
        ubi_btKillTree( (ubi_btRootPtr)(Rp), (ubi_btKillNodeRtn)(Fn) )

#define ubi_trLeafNode( Nd ) \
        ubi_btLeafNode( (ubi_btNodePtr)(Nd) )

#define ubi_trModuleID( s, l ) ubi_btModuleID( s, l )

/* ========================================================================== */
#endif /* ubi_BinTree_H */
